<?php
/**
 * Public my account class
 *
 * @package    WordPress
 * @author     David Perez <david@close.technology>
 * @copyright  2024 CLOSE
 * @version    1.0.0
 */

namespace CLOSE\ConnectEcommerce\Frontend;

defined( 'ABSPATH' ) || exit;

use CLOSE\ConnectEcommerce\Base;
/**
 * My Account.
 *
 * @since 1.6.0
 */
class MyAccount {
	/**
	 * Options of plugin.
	 *
	 * @var array
	 */
	private $options;

	/**
	 * Settings all
	 *
	 * @var array
	 */
	private $settings_all;

	/**
	 * Settings
	 *
	 * @var array
	 */
	private $settings;

	/**
	 * All options
	 *
	 * @var array
	 */
	private $all_options;

	/**
	 * API Object
	 *
	 * @var object
	 */
	private $connapi_erp;

	/**
	 * Connector
	 *
	 * @var object
	 */
	private $connector;

	/**
	 * Construct of Class
	 *
	 * @param array $connector Connector.
	 */
	public function __construct( $connector ) {
		$this->settings_all = $connector['settings_all'] ?? get_option( 'connect_ecommerce' );
		$this->connector    = $connector['connector'] ?? '';
		$this->settings     = $connector['settings'] ?? array();
		$this->all_options  = $connector['all_options'];
		$this->options      = $connector['options'] ?? array();
		$this->connapi_erp  = $connector['connapi_erp'] ?? null;

		if ( ! empty( $this->connector ) && empty( $this->options ) ) {
			return;
		}

		add_filter( 'woocommerce_account_orders_columns', array( $this, 'add_account_orders_column' ), 10, 1 );
		add_action( 'woocommerce_my_account_my_orders_column_custom-column', array( $this, 'add_account_orders_column_rows' ) );
		add_action( 'wp_ajax_cwc_document_download', array( $this, 'cwc_document_download' ) );
	}

	/**
	 * Add custom column to my account orders.
	 *
	 * @param array $columns Columns of orders.
	 * @return array
	 */
	public function add_account_orders_column( $columns ) {
		$order_actions  = $columns['order-actions']; // Save Order actions.
		unset( $columns['order-actions'] ); // Remove Order actions.

		// Add your custom column key / label.
		$columns['custom-column'] = __( 'Invoice', 'woocommerce-es' );

		// Add back previously saved "Order actions".
		$columns['order-actions'] = $order_actions;

		return $columns;
	}

	/**
	 * Add custom column rows to my account orders.
	 *
	 * @param WC_Order $order Order object.
	 */
	public function add_account_orders_column_rows( $order ) {
		$api_doc_id = $order->get_meta( '_' . $this->options['slug'] . '_doc_id' );

		if ( ! empty( $api_doc_id ) ) {
			$api_doc_type  = $order->get_meta( '_' . $this->options['slug'] . '_doc_type' );
			if ( empty( $this->connapi_erp ) || ! method_exists( $this->connapi_erp, 'get_order_pdf' ) ) {
				return;
			}
			$document_file = $this->connapi_erp->get_order_pdf( $this->settings, $api_doc_type, $api_doc_id );
			if ( $document_file ) {
				$nonce        = wp_create_nonce( 'cwc-document-nonce' );
				echo '<a href=' . esc_url( admin_url( 'admin-ajax.php?action=cwc_document_download&doc_id=' . esc_attr( $api_doc_id ) . '&doc_type=' . esc_attr( $api_doc_type ) . '&nonce=' . $nonce ) ) . ' class="button button-primary" target="_blank">';
				echo esc_html__( 'Download', 'woocommerce-es' ) . '</a>';
			}
		}
	}

	/**
	 * Ajax to download the file
	 *
	 * @return void
	 */
	public function cwc_document_download() {
		check_ajax_referer( 'cwc-document-nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( "Hmmm, you're not supposed to be here." );
		}
		$api_doc_id   = isset( $_GET['doc_id'] ) ? sanitize_text_field( wp_unslash( $_GET['doc_id'] ) ) : '';
		$api_doc_type = isset( $_GET['doc_type'] ) ? sanitize_text_field( wp_unslash( $_GET['doc_type'] ) ) : '';

		$file_document_path = false;
		if ( $api_doc_id && $this->connapi_erp ) {
			$settings           = get_option( $this->options['slug'] );
			$file_document_path = $this->connapi_erp->get_order_pdf( $settings, $api_doc_type, $api_doc_id );
		}

		if ( ! file_exists( $file_document_path ) ) {
			wp_die();
		}

		$basename = basename( $file_document_path );
		$filesize = filesize( $file_document_path );

		header( 'Content-Description: File Transfer' );
		header( 'Content-Type: text/plain' );
		header( 'Cache-Control: no-cache, must-revalidate' );
		header( 'Expires: 0' );
		header( 'Content-Disposition: attachment; filename=' . $basename );
		header( 'Content-Length: ' . $filesize );
		header( 'Pragma: public' );

		flush();

		readfile( $file_document_path );

		wp_die();
	}
}
