<?php

namespace Mint\MRM\Internal\Admin;

use MRM\Common\MrmCommon;

/**
 * SpecialOccasionBanner Class
 *
 * This class is responsible for displaying a special occasion banner in the WordPress admin.
 *
 * @package YourVendor\SpecialOccasionPlugin
 */
class SpecialOccasionBanner
{

    /**
     * The occasion identifier.
     *
     * @var string
     */
    private $occasion;

    /**
     * The start date and time for displaying the banner.
     *
     * @var int
     */
    private $start_date;

    /**
     * The end date and time for displaying the banner.
     *
     * @var int
     */
    private $end_date;

    /**
     * Constructor method for SpecialOccasionBanner class.
     *
     * @param string $occasion   The occasion identifier.
     * @param string $start_date The start date and time for displaying the banner.
     * @param string $end_date   The end date and time for displaying the banner.
     */
    public function __construct($occasion, $start_date, $end_date)
    {
        $this->occasion = $occasion;
        $this->start_date = strtotime($start_date);
        $this->end_date = strtotime($end_date);

        // Hook into the admin_notices action to display the banner
        add_action('admin_notices', array($this, 'display_banner'));

        // Add styles
        add_action('admin_head', array($this, 'add_styles'));
    }

    /**
     * Calculate time remaining until blackfriday
     *
     * @return array Time remaining in days, hours, and minutes
     */
    function mint_get_promotional_countdown()
    {
        $end_date  = strtotime('2025-10-20 23:59:59');
        $now       = current_time('timestamp');
        $diff      = $end_date - $now;

        return array(
            'days' => floor($diff / (60 * 60 * 24)),
            'hours' => floor(($diff % (60 * 60 * 24)) / (60 * 60)),
            'mins' => floor(($diff % (60 * 60)) / 60),
            'secs' => $diff % 60,
        );
    }


    /**
     * Displays the special occasion banner if the current date and time are within the specified range.
     */
    public function display_banner()
    {
        $screen = get_current_screen();
        $promotional_notice_pages = ['dashboard', 'plugins', 'toplevel_page_mrm-admin'];
        $current_date_time = current_time('timestamp');
        $btn_link = 'https://getwpfunnels.com/pricing/?utm_source=plugin&utm_medium=dashboard-banner-mm&utm_campaign=bfcm2025';

        if (!in_array($screen->id, $promotional_notice_pages)) {
            return;
        }

        if (defined('MAIL_MINT_PRO_VERSION') || ($current_date_time < $this->start_date || $current_date_time > $this->end_date) || 'no' === get_option('_is_show_blackfriday_banner') || MrmCommon::is_wpfnl_active()) {
            return;
        }

        // Calculate the time remaining in seconds
        $time_remaining = $this->end_date - $current_date_time;

        // Calculate the countdown
        $countdown = $this->mint_get_promotional_countdown();
        ?>

        <div class="gwpf-promotional-notice mailmint-promotional-notice <?php echo esc_attr($this->occasion); ?>-banner notice">
            <div class="gwpf-tb__notification">
                <div class="banner-overflow">
                    <section class="gwpf-promotional-banner" aria-labelledby="wpf-blackfriday-offer">
                        <div class="gwpf-container">
                            <div class="promotional-banner">
                                <div class="banner-content">
                                    <span class="blackfriday-logo">
                                        <svg width="125" height="36" fill="none" viewBox="0 0 125 36" xmlns="http://www.w3.org/2000/svg"><path fill="#02C4FB" d="M12.994.556c6.706 0 9.833 2.675 9.833 6.358 0 .11-.006.22-.012.33-.202-3.494-3.34-5.993-9.82-5.993H0V.556h12.994zm21.384.695h-8.2V.556h8.2v.695zm73.067 0h-8.06V.556h8.06v.695zm17.408 23.626h-.522l-9.693-12.925.285-.314 9.93 13.239zm-50.903 0h-.303L63.318 1.251h-8.06l-9.9 22.647v-.696l9.9-22.646h8.06L73.95 24.877zm7.577-11.816c.138 3.518 2.393 5.736 5.692 5.736 2.189 0 3.892-1.043 5.316-2.85l5.177 4.621c-.106.138-.215.272-.326.405l-4.85-4.33c-1.425 1.806-3.128 2.85-5.317 2.85-3.405-.001-5.697-2.364-5.697-6.082 0-.118 0-.235.005-.35zm-65.851 4.52c.017.11.03.226.03.348-.001 1.25-.905 1.91-2.712 1.91H8.061v-.694h4.933c1.632 0 2.524-.54 2.682-1.565zm18.702.938h10.978v.695H34.378v-.695zM20.16 12.45c2.418 1.115 3.813 3.084 3.813 5.72 0 .117-.005.231-.01.345-.13-2.703-1.775-4.66-4.507-5.66a7.43 7.43 0 00.704-.405zm41.151 2.28l.27.696h-4.727l.269-.696h4.188zM86.802 0c4.76 0 8.512 1.737 10.91 4.864l-.326.29C94.98 2.284 91.353.695 86.803.695c-7.695 0-13.37 5.083-13.547 12.366a14.004 14.004 0 01-.004-.344C73.252 5.247 78.984 0 86.802 0zM14.528 8.442c.018.11.03.226.03.348 0 1.181-.834 1.807-2.675 1.807H8.06v-.695h3.822c1.651 0 2.49-.503 2.645-1.46zm109.206-7.191h-8.297l-7.992 8.825v-.695l7.992-8.825h8.929l-.632.695z"/><path fill="url(#paint0_linear_4105_1718)" d="M115.472 25.572l-6.081-8.061-1.841 2.05v6.01h-8.061V1.252h8.061v8.825l7.991-8.825h8.929l-9.728 10.7 10.215 13.62h-9.485z"/><path fill="url(#paint1_linear_4105_1718)" d="M86.906 26.128c-7.817 0-13.55-5.247-13.55-12.717 0-7.47 5.733-12.716 13.55-12.716 4.76 0 8.513 1.737 10.91 4.864l-5.177 4.62c-1.424-1.806-3.127-2.848-5.316-2.848-3.404 0-5.698 2.362-5.698 6.08s2.293 6.08 5.698 6.08c2.19 0 3.892-1.042 5.316-2.849l5.177 4.621c-2.397 3.127-6.15 4.865-10.91 4.865z"/><path fill="url(#paint2_linear_4105_1718)" d="M65.577 25.572l-1.633-4.239h-9.242l-1.633 4.239H44.73L55.362 1.25h8.06l10.633 24.32h-8.478zM56.96 15.426h4.725l-2.362-6.115-2.363 6.115z"/><path fill="url(#paint3_linear_4105_1718)" d="M26.282 25.572V1.25h8.2v17.962h10.98v6.359h-19.18z"/><path fill="url(#paint4_linear_4105_1718)" d="M19.561 12.855c2.85 1.043 4.517 3.127 4.517 6.011 0 4.239-3.579 6.706-10.284 6.706H.104V1.25H13.1c6.706 0 9.832 2.675 9.832 6.358 0 2.258-1.146 4.134-3.37 5.246zm-7.574-5.872H8.165v3.614h3.822c1.841 0 2.675-.626 2.675-1.807s-.834-1.807-2.675-1.807zm1.112 12.856c1.806 0 2.71-.66 2.71-1.911 0-1.25-.903-1.911-2.71-1.911H8.165v3.822h4.934z"/><mask id="a" width="63" height="33" x="30.78" y="3" fill="#000" maskUnits="userSpaceOnUse"><path fill="#fff" d="M30.78 3h63v33h-63z"/><path d="M77.612 34.707c.593-.018 1.134-.262 1.64-.767a17.593 17.593 0 003.103-3.959c1.15-1.918 2.005-3.54 2.563-4.865-3.487 2.302-5.946 4.708-7.358 7.184-.349.628-.523 1.221-.523 1.762 0 .035 0 .157.017.33.017.193.21.297.558.315zm-.035.54c-.645 0-.994-.244-1.064-.732a3.393 3.393 0 01-.034-.419c0-.592.192-1.237.558-1.952a16.637 16.637 0 013.592-4.43 51.514 51.514 0 014.533-3.504c.227-.157.401-.332.506-.524l1.5-2.667-.035-.018-.366.297c-1.849 1.743-3.017 2.598-3.523 2.598a.625.625 0 01-.418-.157c-.175-.157-.262-.384-.262-.68 0-.61.227-1.326.68-2.11.384-.68.767-1.203 1.169-1.57.261-.226.47-.33.645-.33.174 0 .261.104.261.33 0 .14-.122.384-.348.715a10.027 10.027 0 00-1.43 2.511 1.129 1.129 0 00-.053.384v.157c.523-.21 1.221-.68 2.093-1.413.4-.313 1.15-.993 2.267-2.04.157-.157.279-.226.4-.226.036 0 .07.017.123.052.052.018.087.07.122.157.017.087.035.157.035.21-.053.139-.366.766-.977 1.882-.366.646-.558 1.047-.593 1.238a24.506 24.506 0 015.476-2.86l.226-.052c.07.018.105.07.105.14 0 .087-.052.14-.122.174l-1.133.454c-1.082.47-2.738 1.517-4.953 3.104a1.258 1.258 0 00-.366.435c-.244.471-.47.907-.68 1.343-1.778 3.714-3.749 6.609-5.928 8.684a3.187 3.187 0 01-1.082.645 2.547 2.547 0 01-.924.174zM74.811 21.21c.035 0 .123-.034.245-.122.645-.383 1.133-.697 1.482-.959-.122-.157-.244-.226-.401-.226a.618.618 0 00-.175.035c-.453.192-.854.558-1.185 1.116v.087c0 .052 0 .07.034.07zm2.442 1.936c-.07 0-.21 0-.384-.017-.192-.018-.349-.157-.488-.419a1.422 1.422 0 01-.105-.54c0-.384.105-.785.332-1.221-.332.227-.715.453-1.169.715-.697.401-1.133.593-1.29.593a.523.523 0 01-.314-.105.364.364 0 01-.14-.279.71.71 0 01.088-.314c.209-.715.645-1.377 1.342-1.988.297-.244.628-.366.994-.366.018 0 .14 0 .349.018.21.017.401.157.593.436l.035.035c.017 0 .07-.018.122-.07a.34.34 0 01.21-.07c.104 0 .19.018.26.087.07.07.105.14.105.21 0 .07-.07.279-.244.61-.4.732-.61 1.29-.61 1.674 0 .279.14.436.436.436s.715-.105 1.255-.297a20.058 20.058 0 002.511-1.15c.837-.436 1.395-.768 1.674-.96.035-.035.07-.035.105-.035.087 0 .14.053.14.192 0 .035-.018.07-.053.105a13.48 13.48 0 01-1.848 1.238 15.8 15.8 0 01-2.337 1.116c-.593.244-1.116.366-1.57.366zm-10.148-.837a9.549 9.549 0 001.935-2.005c.244-.314.506-.628.767-.942l-.209-.017c-.087 0-.192.07-.349.192-.889.68-1.586 1.5-2.075 2.493l-.07.279zm1.028 5.353c-.226 0-.348-.174-.348-.523 0-.244.052-.593.157-1.046a27.835 27.835 0 011.778-4.656c.436-.907.645-1.43.645-1.552a.349.349 0 00-.034-.157c-.21.244-.436.523-.698.855-.366.47-.767.941-1.186 1.412-.732.802-1.29 1.203-1.674 1.203a.578.578 0 01-.436-.174.82.82 0 01-.174-.523c0-.61.401-1.465 1.186-2.564a5.894 5.894 0 011.482-1.464c.261-.175.506-.28.767-.28.332 0 .663.175.994.506l.523-.732a22.846 22.846 0 001.64-2.493 22.605 22.605 0 012.754-3.941c.123-.175.28-.262.436-.262a.57.57 0 01.35.122c.086.087.139.175.139.244 0 .14-.123.297-.332.489a1.5 1.5 0 00-.226.244c-2.738 3.522-5.075 7.777-6.993 12.746-.157.401-.279.82-.383 1.238-.105.21-.157.471-.157.768 0 .122.017.244.035.366v.035c0 .087-.07.14-.245.14zm-8.423-3.906c-.663 0-.994-.296-1.012-.906.035-.593.227-1.22.593-1.884.349-.645.715-1.185 1.116-1.604.227-.209.436-.314.593-.314.262 0 .384.123.384.367 0 .104-.157.4-.454.872l-1.01 1.603c-.192.314-.28.593-.28.82 0 .296.14.435.42.435.453 0 1.307-.296 2.563-.907 1.011-.47 1.796-.889 2.336-1.238.192-.122.366-.244.54-.348l.803-.523c.087 0 .14.017.174.052 0 .07-.017.122-.07.157-.697.645-1.69 1.308-2.981 2.005-1.29.697-2.407 1.15-3.33 1.378-.14.034-.262.034-.384.034zm2.668-6.277c-.314 0-.471-.157-.471-.47 0-.262.104-.437.314-.541.157-.07.296-.122.453-.122.157 0 .331.035.54.104.21.053.314.157.314.297 0 .244-.313.488-.906.697l-.244.035zm-12.077 9.224c-.47 0-.802-.174-1.01-.523a1.65 1.65 0 01-.262-.889c0-.384.087-.785.244-1.22.174-.436.488-1.03.906-1.744l.35-.628c.139-.262.208-.453.208-.558 0-.174-.122-.279-.348-.366-.244-.07-.384-.122-.436-.122a.038.038 0 00-.035.035c-.017.017-.017.017-.035.017-1.203 1.5-2.424 2.529-3.662 3.07a.282.282 0 01-.157.051c-.087 0-.122-.052-.122-.157 0-.052.035-.104.122-.14a9.721 9.721 0 001.36-.993c.855-.732 1.465-1.412 1.831-2.058l.018-.034c0-.035-.018-.053-.053-.07a.45.45 0 01-.157-.366.91.91 0 01.07-.366 2.38 2.38 0 01.523-.872.59.59 0 01.436-.175c.087 0 .192.018.314.053a.73.73 0 01.28.191.382.382 0 01.121.297c0 .157-.07.314-.192.47.436.175.75.35.96.541.191.175.278.419.296.698 0 .226-.087.47-.244.732l-.907 1.36c-.61.924-.907 1.691-.907 2.337 0 .505.262.75.785.75.105 0 .261-.018.453-.053.436-.087 1.081-.453 1.97-1.116.611-.453 1.483-1.186 2.634-2.18l1.604-1.36c.837-.662 1.447-1.063 1.83-1.203.018 0 .053 0 .07.018.018.017.018.052.018.07v.017l-.017.035c-.576.331-1.483 1.08-2.72 2.25-1.988 1.848-3.47 3.05-4.412 3.609-.663.4-1.238.592-1.727.592zm-16.393 3.819c.54-.54 1.15-1.796 1.848-3.732.663-1.813 1.046-3.173 1.15-4.08-.156.28-.662 1.482-1.516 3.627l-.192.506c-.837 1.988-1.255 3.208-1.29 3.679zm4.742-13.043l.088-.14a148.672 148.672 0 012.842-5.405c1.203-2.18 2.215-3.94 3.052-5.266.034-.035.07-.087.07-.14l-.018-.07c-1.343 0-2.808.07-4.412.227a45.465 45.465 0 00-4.411.628l-.14.018c-1.482.278-2.284.453-2.406.523a.737.737 0 00-.349.14l1.081.19a16.37 16.37 0 011.657.332c1.36.349 2.232.994 2.65 1.918.419.925.628 2.145.628 3.645v.244c-.035 1.482-.14 2.546-.331 3.156zM33.525 32.23a.877.877 0 01-.244-.053c-.105-.034-.21-.139-.297-.313-.087-.157-.122-.367-.122-.628 0-.21.053-.54.157-.977.227-.941.558-1.988.994-3.103.384-.977.715-1.762 1.012-2.355 1.203-2.737 1.935-4.306 2.197-4.707.296-.576.453-1.012.505-1.273-.017-.053-.052-.088-.122-.088l-.104.018c-.297.052-.855.105-1.657.14l-.349.017c-.279 0-.4-.122-.4-.366 0-.384.889-.663 2.685-.855.07 0 .14-.052.191-.174.07-.14.157-.715.262-1.709.052-.663.087-1.308.087-1.936 0-2.075-.418-3.4-1.255-3.975a7.123 7.123 0 00-1.395-.645 13.11 13.11 0 00-2.459-.471c-.645-.053-1.029-.14-1.186-.244-.174-.122-.244-.297-.244-.54 0-.193.053-.35.157-.472.244-.209.733-.383 1.448-.558l.732-.14c4.01-.784 7.585-1.185 10.724-1.185h.279c.157 0 .261-.052.331-.157.07-.104.174-.261.331-.453l.471-.576c.157-.174.28-.296.366-.348A.419.419 0 0146.881 4c.035 0 .087.017.174.035.087.017.157.087.21.174a.579.579 0 01.087.28c0 .226-.227.662-.663 1.29.122.052.227.07.331.07.454 0 1.692.07 3.732.191 3.034.21 5.562.558 7.55 1.081l.192.105c.087.07.122.174.122.296 0 .227-.157.332-.506.332-.122 0-.209-.018-.261-.053a5.9 5.9 0 00-1.064-.366l-.296-.07c-.785-.226-1.918-.4-3.4-.523a4.55 4.55 0 01-.646-.07l-2.93-.244c-1.272-.104-2.283-.157-2.998-.157-.244 0-.436.123-.593.367-.087.14-.244.314-.453.523-.157.157-.489.715-1.03 1.639a307.35 307.35 0 00-3.626 6.521c-.698 1.29-1.064 1.953-1.081 1.97-.018.036-.018.07-.018.123v.052h.14c.209 0 .645-.035 1.307-.104 1.622-.14 2.913-.227 3.871-.227.768 0 1.308.07 1.622.174.227.087.349.192.349.297 0 .157-.401.244-1.186.244-1.813 0-3.923.157-6.347.453-.174.018-.296.087-.366.244s-.157.332-.244.489c-.314.645-.523 1.133-.593 1.482-.419 1.953-.89 3.731-1.43 5.335a40.926 40.926 0 01-2.04 4.918c-.105.192-.244.4-.419.662-.313.471-.593.698-.854.698z"/></mask><path fill="#fff" d="M77.612 34.707c.593-.018 1.134-.262 1.64-.767a17.593 17.593 0 003.103-3.959c1.15-1.918 2.005-3.54 2.563-4.865-3.487 2.302-5.946 4.708-7.358 7.184-.349.628-.523 1.221-.523 1.762 0 .035 0 .157.017.33.017.193.21.297.558.315zm-.035.54c-.645 0-.994-.244-1.064-.732a3.393 3.393 0 01-.034-.419c0-.592.192-1.237.558-1.952a16.637 16.637 0 013.592-4.43 51.514 51.514 0 014.533-3.504c.227-.157.401-.332.506-.524l1.5-2.667-.035-.018-.366.297c-1.849 1.743-3.017 2.598-3.523 2.598a.625.625 0 01-.418-.157c-.175-.157-.262-.384-.262-.68 0-.61.227-1.326.68-2.11.384-.68.767-1.203 1.169-1.57.261-.226.47-.33.645-.33.174 0 .261.104.261.33 0 .14-.122.384-.348.715a10.027 10.027 0 00-1.43 2.511 1.129 1.129 0 00-.053.384v.157c.523-.21 1.221-.68 2.093-1.413.4-.313 1.15-.993 2.267-2.04.157-.157.279-.226.4-.226.036 0 .07.017.123.052.052.018.087.07.122.157.017.087.035.157.035.21-.053.139-.366.766-.977 1.882-.366.646-.558 1.047-.593 1.238a24.506 24.506 0 015.476-2.86l.226-.052c.07.018.105.07.105.14 0 .087-.052.14-.122.174l-1.133.454c-1.082.47-2.738 1.517-4.953 3.104a1.258 1.258 0 00-.366.435c-.244.471-.47.907-.68 1.343-1.778 3.714-3.749 6.609-5.928 8.684a3.187 3.187 0 01-1.082.645 2.547 2.547 0 01-.924.174zM74.811 21.21c.035 0 .123-.034.245-.122.645-.383 1.133-.697 1.482-.959-.122-.157-.244-.226-.401-.226a.618.618 0 00-.175.035c-.453.192-.854.558-1.185 1.116v.087c0 .052 0 .07.034.07zm2.442 1.936c-.07 0-.21 0-.384-.017-.192-.018-.349-.157-.488-.419a1.422 1.422 0 01-.105-.54c0-.384.105-.785.332-1.221-.332.227-.715.453-1.169.715-.697.401-1.133.593-1.29.593a.523.523 0 01-.314-.105.364.364 0 01-.14-.279.71.71 0 01.088-.314c.209-.715.645-1.377 1.342-1.988.297-.244.628-.366.994-.366.018 0 .14 0 .349.018.21.017.401.157.593.436l.035.035c.017 0 .07-.018.122-.07a.34.34 0 01.21-.07c.104 0 .19.018.26.087.07.07.105.14.105.21 0 .07-.07.279-.244.61-.4.732-.61 1.29-.61 1.674 0 .279.14.436.436.436s.715-.105 1.255-.297a20.058 20.058 0 002.511-1.15c.837-.436 1.395-.768 1.674-.96.035-.035.07-.035.105-.035.087 0 .14.053.14.192 0 .035-.018.07-.053.105a13.48 13.48 0 01-1.848 1.238 15.8 15.8 0 01-2.337 1.116c-.593.244-1.116.366-1.57.366zm-10.148-.837a9.549 9.549 0 001.935-2.005c.244-.314.506-.628.767-.942l-.209-.017c-.087 0-.192.07-.349.192-.889.68-1.586 1.5-2.075 2.493l-.07.279zm1.028 5.353c-.226 0-.348-.174-.348-.523 0-.244.052-.593.157-1.046a27.835 27.835 0 011.778-4.656c.436-.907.645-1.43.645-1.552a.349.349 0 00-.034-.157c-.21.244-.436.523-.698.855-.366.47-.767.941-1.186 1.412-.732.802-1.29 1.203-1.674 1.203a.578.578 0 01-.436-.174.82.82 0 01-.174-.523c0-.61.401-1.465 1.186-2.564a5.894 5.894 0 011.482-1.464c.261-.175.506-.28.767-.28.332 0 .663.175.994.506l.523-.732a22.846 22.846 0 001.64-2.493 22.605 22.605 0 012.754-3.941c.123-.175.28-.262.436-.262a.57.57 0 01.35.122c.086.087.139.175.139.244 0 .14-.123.297-.332.489a1.5 1.5 0 00-.226.244c-2.738 3.522-5.075 7.777-6.993 12.746-.157.401-.279.82-.383 1.238-.105.21-.157.471-.157.768 0 .122.017.244.035.366v.035c0 .087-.07.14-.245.14zm-8.423-3.906c-.663 0-.994-.296-1.012-.906.035-.593.227-1.22.593-1.884.349-.645.715-1.185 1.116-1.604.227-.209.436-.314.593-.314.262 0 .384.123.384.367 0 .104-.157.4-.454.872l-1.01 1.603c-.192.314-.28.593-.28.82 0 .296.14.435.42.435.453 0 1.307-.296 2.563-.907 1.011-.47 1.796-.889 2.336-1.238.192-.122.366-.244.54-.348l.803-.523c.087 0 .14.017.174.052 0 .07-.017.122-.07.157-.697.645-1.69 1.308-2.981 2.005-1.29.697-2.407 1.15-3.33 1.378-.14.034-.262.034-.384.034zm2.668-6.277c-.314 0-.471-.157-.471-.47 0-.262.104-.437.314-.541.157-.07.296-.122.453-.122.157 0 .331.035.54.104.21.053.314.157.314.297 0 .244-.313.488-.906.697l-.244.035zm-12.077 9.224c-.47 0-.802-.174-1.01-.523a1.65 1.65 0 01-.262-.889c0-.384.087-.785.244-1.22.174-.436.488-1.03.906-1.744l.35-.628c.139-.262.208-.453.208-.558 0-.174-.122-.279-.348-.366-.244-.07-.384-.122-.436-.122a.038.038 0 00-.035.035c-.017.017-.017.017-.035.017-1.203 1.5-2.424 2.529-3.662 3.07a.282.282 0 01-.157.051c-.087 0-.122-.052-.122-.157 0-.052.035-.104.122-.14a9.721 9.721 0 001.36-.993c.855-.732 1.465-1.412 1.831-2.058l.018-.034c0-.035-.018-.053-.053-.07a.45.45 0 01-.157-.366.91.91 0 01.07-.366 2.38 2.38 0 01.523-.872.59.59 0 01.436-.175c.087 0 .192.018.314.053a.73.73 0 01.28.191.382.382 0 01.121.297c0 .157-.07.314-.192.47.436.175.75.35.96.541.191.175.278.419.296.698 0 .226-.087.47-.244.732l-.907 1.36c-.61.924-.907 1.691-.907 2.337 0 .505.262.75.785.75.105 0 .261-.018.453-.053.436-.087 1.081-.453 1.97-1.116.611-.453 1.483-1.186 2.634-2.18l1.604-1.36c.837-.662 1.447-1.063 1.83-1.203.018 0 .053 0 .07.018.018.017.018.052.018.07v.017l-.017.035c-.576.331-1.483 1.08-2.72 2.25-1.988 1.848-3.47 3.05-4.412 3.609-.663.4-1.238.592-1.727.592zm-16.393 3.819c.54-.54 1.15-1.796 1.848-3.732.663-1.813 1.046-3.173 1.15-4.08-.156.28-.662 1.482-1.516 3.627l-.192.506c-.837 1.988-1.255 3.208-1.29 3.679zm4.742-13.043l.088-.14a148.672 148.672 0 012.842-5.405c1.203-2.18 2.215-3.94 3.052-5.266.034-.035.07-.087.07-.14l-.018-.07c-1.343 0-2.808.07-4.412.227a45.465 45.465 0 00-4.411.628l-.14.018c-1.482.278-2.284.453-2.406.523a.737.737 0 00-.349.14l1.081.19a16.37 16.37 0 011.657.332c1.36.349 2.232.994 2.65 1.918.419.925.628 2.145.628 3.645v.244c-.035 1.482-.14 2.546-.331 3.156zM33.525 32.23a.877.877 0 01-.244-.053c-.105-.034-.21-.139-.297-.313-.087-.157-.122-.367-.122-.628 0-.21.053-.54.157-.977.227-.941.558-1.988.994-3.103.384-.977.715-1.762 1.012-2.355 1.203-2.737 1.935-4.306 2.197-4.707.296-.576.453-1.012.505-1.273-.017-.053-.052-.088-.122-.088l-.104.018c-.297.052-.855.105-1.657.14l-.349.017c-.279 0-.4-.122-.4-.366 0-.384.889-.663 2.685-.855.07 0 .14-.052.191-.174.07-.14.157-.715.262-1.709.052-.663.087-1.308.087-1.936 0-2.075-.418-3.4-1.255-3.975a7.123 7.123 0 00-1.395-.645 13.11 13.11 0 00-2.459-.471c-.645-.053-1.029-.14-1.186-.244-.174-.122-.244-.297-.244-.54 0-.193.053-.35.157-.472.244-.209.733-.383 1.448-.558l.732-.14c4.01-.784 7.585-1.185 10.724-1.185h.279c.157 0 .261-.052.331-.157.07-.104.174-.261.331-.453l.471-.576c.157-.174.28-.296.366-.348A.419.419 0 0146.881 4c.035 0 .087.017.174.035.087.017.157.087.21.174a.579.579 0 01.087.28c0 .226-.227.662-.663 1.29.122.052.227.07.331.07.454 0 1.692.07 3.732.191 3.034.21 5.562.558 7.55 1.081l.192.105c.087.07.122.174.122.296 0 .227-.157.332-.506.332-.122 0-.209-.018-.261-.053a5.9 5.9 0 00-1.064-.366l-.296-.07c-.785-.226-1.918-.4-3.4-.523a4.55 4.55 0 01-.646-.07l-2.93-.244c-1.272-.104-2.283-.157-2.998-.157-.244 0-.436.123-.593.367-.087.14-.244.314-.453.523-.157.157-.489.715-1.03 1.639a307.35 307.35 0 00-3.626 6.521c-.698 1.29-1.064 1.953-1.081 1.97-.018.036-.018.07-.018.123v.052h.14c.209 0 .645-.035 1.307-.104 1.622-.14 2.913-.227 3.871-.227.768 0 1.308.07 1.622.174.227.087.349.192.349.297 0 .157-.401.244-1.186.244-1.813 0-3.923.157-6.347.453-.174.018-.296.087-.366.244s-.157.332-.244.489c-.314.645-.523 1.133-.593 1.482-.419 1.953-.89 3.731-1.43 5.335a40.926 40.926 0 01-2.04 4.918c-.105.192-.244.4-.419.662-.313.471-.593.698-.854.698z"/><path stroke="#fff" stroke-width=".6" d="M77.612 34.707c.593-.018 1.134-.262 1.64-.767a17.593 17.593 0 003.103-3.959c1.15-1.918 2.005-3.54 2.563-4.865-3.487 2.302-5.946 4.708-7.358 7.184-.349.628-.523 1.221-.523 1.762 0 .035 0 .157.017.33.017.193.21.297.558.315zm-.035.54c-.645 0-.994-.244-1.064-.732a3.393 3.393 0 01-.034-.419c0-.592.192-1.237.558-1.952a16.637 16.637 0 013.592-4.43 51.514 51.514 0 014.533-3.504c.227-.157.401-.332.506-.524l1.5-2.667-.035-.018-.366.297c-1.849 1.743-3.017 2.598-3.523 2.598a.625.625 0 01-.418-.157c-.175-.157-.262-.384-.262-.68 0-.61.227-1.326.68-2.11.384-.68.767-1.203 1.169-1.57.261-.226.47-.33.645-.33.174 0 .261.104.261.33 0 .14-.122.384-.348.715a10.027 10.027 0 00-1.43 2.511 1.129 1.129 0 00-.053.384v.157c.523-.21 1.221-.68 2.093-1.413.4-.313 1.15-.993 2.267-2.04.157-.157.279-.226.4-.226.036 0 .07.017.123.052.052.018.087.07.122.157.017.087.035.157.035.21-.053.139-.366.766-.977 1.882-.366.646-.558 1.047-.593 1.238a24.506 24.506 0 015.476-2.86l.226-.052c.07.018.105.07.105.14 0 .087-.052.14-.122.174l-1.133.454c-1.082.47-2.738 1.517-4.953 3.104a1.258 1.258 0 00-.366.435c-.244.471-.47.907-.68 1.343-1.778 3.714-3.749 6.609-5.928 8.684a3.187 3.187 0 01-1.082.645 2.547 2.547 0 01-.924.174zM74.811 21.21c.035 0 .123-.034.245-.122.645-.383 1.133-.697 1.482-.959-.122-.157-.244-.226-.401-.226a.618.618 0 00-.175.035c-.453.192-.854.558-1.185 1.116v.087c0 .052 0 .07.034.07zm2.442 1.936c-.07 0-.21 0-.384-.017-.192-.018-.349-.157-.488-.419a1.422 1.422 0 01-.105-.54c0-.384.105-.785.332-1.221-.332.227-.715.453-1.169.715-.697.401-1.133.593-1.29.593a.523.523 0 01-.314-.105.364.364 0 01-.14-.279.71.71 0 01.088-.314c.209-.715.645-1.377 1.342-1.988.297-.244.628-.366.994-.366.018 0 .14 0 .349.018.21.017.401.157.593.436l.035.035c.017 0 .07-.018.122-.07a.34.34 0 01.21-.07c.104 0 .19.018.26.087.07.07.105.14.105.21 0 .07-.07.279-.244.61-.4.732-.61 1.29-.61 1.674 0 .279.14.436.436.436s.715-.105 1.255-.297a20.058 20.058 0 002.511-1.15c.837-.436 1.395-.768 1.674-.96.035-.035.07-.035.105-.035.087 0 .14.053.14.192 0 .035-.018.07-.053.105a13.48 13.48 0 01-1.848 1.238 15.8 15.8 0 01-2.337 1.116c-.593.244-1.116.366-1.57.366zm-10.148-.837a9.549 9.549 0 001.935-2.005c.244-.314.506-.628.767-.942l-.209-.017c-.087 0-.192.07-.349.192-.889.68-1.586 1.5-2.075 2.493l-.07.279zm1.028 5.353c-.226 0-.348-.174-.348-.523 0-.244.052-.593.157-1.046a27.835 27.835 0 011.778-4.656c.436-.907.645-1.43.645-1.552a.349.349 0 00-.034-.157c-.21.244-.436.523-.698.855-.366.47-.767.941-1.186 1.412-.732.802-1.29 1.203-1.674 1.203a.578.578 0 01-.436-.174.82.82 0 01-.174-.523c0-.61.401-1.465 1.186-2.564a5.894 5.894 0 011.482-1.464c.261-.175.506-.28.767-.28.332 0 .663.175.994.506l.523-.732a22.846 22.846 0 001.64-2.493 22.605 22.605 0 012.754-3.941c.123-.175.28-.262.436-.262a.57.57 0 01.35.122c.086.087.139.175.139.244 0 .14-.123.297-.332.489a1.5 1.5 0 00-.226.244c-2.738 3.522-5.075 7.777-6.993 12.746-.157.401-.279.82-.383 1.238-.105.21-.157.471-.157.768 0 .122.017.244.035.366v.035c0 .087-.07.14-.245.14zm-8.423-3.906c-.663 0-.994-.296-1.012-.906.035-.593.227-1.22.593-1.884.349-.645.715-1.185 1.116-1.604.227-.209.436-.314.593-.314.262 0 .384.123.384.367 0 .104-.157.4-.454.872l-1.01 1.603c-.192.314-.28.593-.28.82 0 .296.14.435.42.435.453 0 1.307-.296 2.563-.907 1.011-.47 1.796-.889 2.336-1.238.192-.122.366-.244.54-.348l.803-.523c.087 0 .14.017.174.052 0 .07-.017.122-.07.157-.697.645-1.69 1.308-2.981 2.005-1.29.697-2.407 1.15-3.33 1.378-.14.034-.262.034-.384.034zm2.668-6.277c-.314 0-.471-.157-.471-.47 0-.262.104-.437.314-.541.157-.07.296-.122.453-.122.157 0 .331.035.54.104.21.053.314.157.314.297 0 .244-.313.488-.906.697l-.244.035zm-12.077 9.224c-.47 0-.802-.174-1.01-.523a1.65 1.65 0 01-.262-.889c0-.384.087-.785.244-1.22.174-.436.488-1.03.906-1.744l.35-.628c.139-.262.208-.453.208-.558 0-.174-.122-.279-.348-.366-.244-.07-.384-.122-.436-.122a.038.038 0 00-.035.035c-.017.017-.017.017-.035.017-1.203 1.5-2.424 2.529-3.662 3.07a.282.282 0 01-.157.051c-.087 0-.122-.052-.122-.157 0-.052.035-.104.122-.14a9.721 9.721 0 001.36-.993c.855-.732 1.465-1.412 1.831-2.058l.018-.034c0-.035-.018-.053-.053-.07a.45.45 0 01-.157-.366.91.91 0 01.07-.366 2.38 2.38 0 01.523-.872.59.59 0 01.436-.175c.087 0 .192.018.314.053a.73.73 0 01.28.191.382.382 0 01.121.297c0 .157-.07.314-.192.47.436.175.75.35.96.541.191.175.278.419.296.698 0 .226-.087.47-.244.732l-.907 1.36c-.61.924-.907 1.691-.907 2.337 0 .505.262.75.785.75.105 0 .261-.018.453-.053.436-.087 1.081-.453 1.97-1.116.611-.453 1.483-1.186 2.634-2.18l1.604-1.36c.837-.662 1.447-1.063 1.83-1.203.018 0 .053 0 .07.018.018.017.018.052.018.07v.017l-.017.035c-.576.331-1.483 1.08-2.72 2.25-1.988 1.848-3.47 3.05-4.412 3.609-.663.4-1.238.592-1.727.592zm-16.393 3.819c.54-.54 1.15-1.796 1.848-3.732.663-1.813 1.046-3.173 1.15-4.08-.156.28-.662 1.482-1.516 3.627l-.192.506c-.837 1.988-1.255 3.208-1.29 3.679zm4.742-13.043l.088-.14a148.672 148.672 0 012.842-5.405c1.203-2.18 2.215-3.94 3.052-5.266.034-.035.07-.087.07-.14l-.018-.07c-1.343 0-2.808.07-4.412.227a45.465 45.465 0 00-4.411.628l-.14.018c-1.482.278-2.284.453-2.406.523a.737.737 0 00-.349.14l1.081.19a16.37 16.37 0 011.657.332c1.36.349 2.232.994 2.65 1.918.419.925.628 2.145.628 3.645v.244c-.035 1.482-.14 2.546-.331 3.156zM33.525 32.23a.877.877 0 01-.244-.053c-.105-.034-.21-.139-.297-.313-.087-.157-.122-.367-.122-.628 0-.21.053-.54.157-.977.227-.941.558-1.988.994-3.103.384-.977.715-1.762 1.012-2.355 1.203-2.737 1.935-4.306 2.197-4.707.296-.576.453-1.012.505-1.273-.017-.053-.052-.088-.122-.088l-.104.018c-.297.052-.855.105-1.657.14l-.349.017c-.279 0-.4-.122-.4-.366 0-.384.889-.663 2.685-.855.07 0 .14-.052.191-.174.07-.14.157-.715.262-1.709.052-.663.087-1.308.087-1.936 0-2.075-.418-3.4-1.255-3.975a7.123 7.123 0 00-1.395-.645 13.11 13.11 0 00-2.459-.471c-.645-.053-1.029-.14-1.186-.244-.174-.122-.244-.297-.244-.54 0-.193.053-.35.157-.472.244-.209.733-.383 1.448-.558l.732-.14c4.01-.784 7.585-1.185 10.724-1.185h.279c.157 0 .261-.052.331-.157.07-.104.174-.261.331-.453l.471-.576c.157-.174.28-.296.366-.348A.419.419 0 0146.881 4c.035 0 .087.017.174.035.087.017.157.087.21.174a.579.579 0 01.087.28c0 .226-.227.662-.663 1.29.122.052.227.07.331.07.454 0 1.692.07 3.732.191 3.034.21 5.562.558 7.55 1.081l.192.105c.087.07.122.174.122.296 0 .227-.157.332-.506.332-.122 0-.209-.018-.261-.053a5.9 5.9 0 00-1.064-.366l-.296-.07c-.785-.226-1.918-.4-3.4-.523a4.55 4.55 0 01-.646-.07l-2.93-.244c-1.272-.104-2.283-.157-2.998-.157-.244 0-.436.123-.593.367-.087.14-.244.314-.453.523-.157.157-.489.715-1.03 1.639a307.35 307.35 0 00-3.626 6.521c-.698 1.29-1.064 1.953-1.081 1.97-.018.036-.018.07-.018.123v.052h.14c.209 0 .645-.035 1.307-.104 1.622-.14 2.913-.227 3.871-.227.768 0 1.308.07 1.622.174.227.087.349.192.349.297 0 .157-.401.244-1.186.244-1.813 0-3.923.157-6.347.453-.174.018-.296.087-.366.244s-.157.332-.244.489c-.314.645-.523 1.133-.593 1.482-.419 1.953-.89 3.731-1.43 5.335a40.926 40.926 0 01-2.04 4.918c-.105.192-.244.4-.419.662-.313.471-.593.698-.854.698z" mask="url(#a)"/><defs><linearGradient id="paint0_linear_4105_1718" x1="56" x2="58" y1="1" y2="51" gradientUnits="userSpaceOnUse"><stop stop-color="#4E3DB5"/><stop offset=".762" stop-color="#402FA9"/></linearGradient><linearGradient id="paint1_linear_4105_1718" x1="56" x2="58" y1="1" y2="51" gradientUnits="userSpaceOnUse"><stop stop-color="#4E3DB5"/><stop offset=".762" stop-color="#402FA9"/></linearGradient><linearGradient id="paint2_linear_4105_1718" x1="56" x2="58" y1="1" y2="51" gradientUnits="userSpaceOnUse"><stop stop-color="#4E3DB5"/><stop offset=".762" stop-color="#402FA9"/></linearGradient><linearGradient id="paint3_linear_4105_1718" x1="56" x2="58" y1="1" y2="51" gradientUnits="userSpaceOnUse"><stop stop-color="#4E3DB5"/><stop offset=".762" stop-color="#402FA9"/></linearGradient><linearGradient id="paint4_linear_4105_1718" x1="56" x2="58" y1="1" y2="51" gradientUnits="userSpaceOnUse"><stop stop-color="#4E3DB5"/><stop offset=".762" stop-color="#402FA9"/></linearGradient></defs></svg>
                                    </span>

                                    <div class="banner-text">
                                        🔥 Black Friday Special: Claim A Huge <span class="highlighted-text">40% OFF</span> On Mail Mint! - Limited Time Deal!
                                    </div>

                                    <a href="<?php echo esc_url($btn_link); ?>" class="cta-button" role="button" aria-label="get special discount " target="_blank">
                                        <?php
                                            echo __('Get 40% OFF', 'getwpfunnels');
                                        ?>
                                        <svg width="11" height="11" fill="none" viewBox="0 0 11 11" xmlns="http://www.w3.org/2000/svg"><path fill="#3C1F7D" stroke="#3C1F7D" stroke-width=".2" d="M9.419.1a.88.88 0 01.88.881V9.42a.88.88 0 11-1.761 0V3.11l-6.934 6.933A.88.88 0 01.358 8.796l6.934-6.934H.982A.88.88 0 11.981.1h8.437z"></path></svg>
                                    </a>

                                </div>
                            </div>
                        </div>
                    </section>
                </div>

                <button class="close-promotional-banner" type="button" aria-label="close banner">
                    <svg width="12" height="13" fill="none" viewBox="0 0 12 13" xmlns="http://www.w3.org/2000/svg"><path stroke="#7A8B9A" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 1.97L1 11.96m0-9.99l10 9.99"/></svg>
                </button>
            </div>
        </div>

        <script>
            var timeRemaining = <?php echo esc_js($time_remaining); ?>;

            function updateCountdown() {
                var endDate = new Date("2025-12-11 23:59:59").getTime();
                var now = new Date().getTime();
                var timeLeft = endDate - now;

                var days = Math.floor(timeLeft / (1000 * 60 * 60 * 24));
                var hours = Math.floor((timeLeft % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                var minutes = Math.floor((timeLeft % (1000 * 60 * 60)) / (1000 * 60));
                var seconds = Math.floor((timeLeft % (1000 * 60)) / 1000);

                var daysElement = document.getElementById('mint-blackfriday-days');
                var hoursElement = document.getElementById('mint-blackfriday-hours');
                var minsElement = document.getElementById('mint-blackfriday-mins');
                var secsElement = document.getElementById('mint-blackfriday-secs');

                if (daysElement) {
                    daysElement.innerHTML = days;
                }

                if (hoursElement) {
                    hoursElement.innerHTML = hours;
                }

                if (minsElement) {
                    minsElement.innerHTML = minutes;
                }
                if (secsElement) {
                    secsElement.innerHTML = seconds;
                }
            }

            document.addEventListener('DOMContentLoaded', function() {
                updateCountdown();
                setInterval(updateCountdown, 1000); // Update every minute
            });
        </script>
    <?php
    }

    /**
     * Adds internal CSS styles for the special occasion banners.
     */
    public function add_styles()
    {
    ?>
        <style id="mailmint-promotional-banner-style">

            .mailmint-promotional-notice .gwpf-tb__notification,
            .mailmint-promotional-notice .gwpf-tb__notification * {
                box-sizing: border-box;
            }

            .mailmint-promotional-notice .gwpf-tb__notification {
                width: 100%;
                margin: 20px 0 20px;
                position: relative;
                border: none;
                box-shadow: none;
                display: block;
                max-height: 110px;
                border-radius: 6px;
            }
            .mailmint-promotional-notice .toplevel_page_mrm-admin .gwpf-tb__notification {
                width: calc(100% - 20px);
            }

            .mailmint-promotional-notice .gwpf-tb__notification .banner-overflow {
                position: relative;
                width: 100%;
                z-index: 1;
            }

            .gwpf-promotional-notice.notice {
                border: none;
                padding: 0;
                display: block;
                background: transparent;
                margin: 0;
                box-shadow: none !important;
            }

            .mailmint-promotional-notice .gwpf-tb__notification .close-promotional-banner {
                position: absolute;
                top: -10px;
                right: -9px;
                background: #fff;
                border: none;
                padding: 0;
                border-radius: 50%;
                cursor: pointer;
                z-index: 9;
                width: 30px;
                height: 30px;
                display: flex;
                align-items: center;
                justify-content: center;
            }

            .mailmint-promotional-notice .gwpf-tb__notification .close-promotional-banner svg {
                width: 22px;
                display: block;
            }

            /* ---banner style start--- */
            .mailmint-promotional-notice .gwpf-promotional-banner {
                position: relative;
                background-color: #150A53;
                /* background: linear-gradient(96deg, #573BFF 23.39%, #3216DA 49.94%); */
                z-index: 1111;
                /* padding: 6px 0; */
            }
            .gwpf-promotional-banner .banner-content .blackfriday-logo svg {
                display: block;
                position: relative;
                top: 5px;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .promotional-banner {
                color: white;
                padding: 12px 20px;
                text-align: center;
                font-size: 14px;
                line-height: 1.4;
                position: relative;
                display: flex;
                flex-flow: column;
                align-items: center;
                justify-content: center;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .banner-content {
                max-width: 1200px;
                margin: 0 auto;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                flex-wrap: wrap;
                gap: 20px;
                row-gap: 8px;
                position: relative;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .banner-text {
                display: flex;
                align-items: center;
                flex-wrap: wrap;
                gap: 6px;
                row-gap: 0;
                justify-content: center;
                font-size: 16px;
                color: #fff;
                font-weight: 400;
                line-height: 1.4;
                text-transform: capitalize;
                letter-spacing: 0;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .banner-text svg {
                display: block;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .banner-text .highlighted-text {
                color: #02C4FB;
                font-weight: 700;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .cta-button {
                color: #3c1f7d;
                font-size: 14px;
                font-style: normal;
                font-weight: 700;
                line-height: 1;
                letter-spacing: -0.084px;
                display: inline-flex;
                align-items: center;
                gap: 8px;
                transition: all .3s ease;
                border-radius: 100px;
                background: #fff;
                box-shadow: 0 4px 10px 0 rgba(110, 66, 211, .3);
                padding: 10px 15px;
                text-decoration: none;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .cta-button svg {
                transform: translate(0, 0);
                transition: all .3s ease;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .cta-button:focus, 
            .mailmint-promotional-notice .gwpf-promotional-banner .cta-button:visited {
                color: #3c1f7d !important;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .cta-button:hover {
                color: #3c1f7d !important;
            }
            .mailmint-promotional-notice .gwpf-promotional-banner .cta-button:hover svg {
                transform: translate(3px, -4px);
            }

            .mailmint-promotional-notice .gwpf-tb__notification .close-promotional-banner {
                position: absolute;
                top: -10px;
                right: -9px;
                background: #fff;
                border: none;
                padding: 0;
                border-radius: 50%;
                cursor: pointer;
                z-index: 9;
                width: 30px;
                height: 30px;
                display: flex;
                align-items: center;
                justify-content: center;
            }

            .mailmint-promotional-notice .gwpf-tb__notification .close-promotional-banner svg {
                width: 22px;
                display: block;
            }

            @media only screen and (max-width: 991px) {
                .mailmint-promotional-notice .promotional-banner {
                    padding: 16px 20px;
                }

                .mailmint-promotional-notice .gwpf-tb__notification {
                    margin: 60px 0 20px;
                }

                .mailmint-promotional-notice .gwpf-tb__notification .close-promotional-banner {
                    width: 25px;
                    height: 25px;
                }
            }

            @media only screen and (max-width: 767px) {
                .mailmint-promotional-notice .wpvr-promotional-banner {
                    padding-top: 20px;
                    padding-bottom: 30px;
                    max-height: none;
                }

                .mailmint-promotional-notice .wpvr-promotional-banner {
                    max-height: none;
                }
            }

            @media only screen and (max-width: 575px) {
                .mailmint-promotional-notice .promotional-banner {
                    padding: 16px 55px;
                    font-size: 13px;
                }
            }
        </style>

<?php
    }
}
