<?php
/**
 * Helper functions
 *
 * @package MintMailPro
 */

namespace MintMailPro;

use DOMDocument;
use DOMXPath;
use MailMintPro\App\Utilities\Helper\FieldConditions;
use MailMintPro\Mint\DataBase\Tables\LeadMagnet;
use Mint\MRM\DataBase\Models\CustomFieldModel;
use WP_Query;
use Mint\MRM\Constants;

/**
 * Helper functions
 *
 * @package MintMailPro;
 */
class Mint_Pro_Helper { //phpcs:ignore
	/**
	 * Check if edd is installed.
	 *
	 * @return bool
	 * @since  1.0.0
	 * @since  1.9.5 Added support for EDD Pro.
	 */
	public static function is_edd_active() {
		$active_plugins = apply_filters( 'active_plugins', get_option( 'active_plugins' ) );
		if ( in_array('easy-digital-downloads/easy-digital-downloads.php', $active_plugins ) || //phpcs:ignore
			in_array( 'easy-digital-downloads-pro/easy-digital-downloads.php', $active_plugins ) ) { //phpcs:ignore
			return true;
		} elseif ( function_exists( 'is_plugin_active' ) ) {
			if ( is_plugin_active( 'easy-digital-downloads/easy-digital-downloads.php' ) ||
				is_plugin_active( 'easy-digital-downloads-pro/easy-digital-downloads.php' ) ) {
				return true;
			}
		}
		return false;
	}


	/**
	 * Check if gravity form is installed.
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_gform_active() {
		if ( in_array( 'gravityforms/gravityforms.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) { //phpcs:ignore
			return true;
		} elseif ( function_exists( 'is_plugin_active' ) ) {
			if ( is_plugin_active( 'gravityforms/gravityforms.php' ) ) {
				return true;
			}
		}
		return false;
	}


	/**
	 * Check if tutor lms is installed.
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_tutor_active() {
		if ( in_array( 'tutor/tutor.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) { //phpcs:ignore
			return true;
		} elseif ( function_exists( 'is_plugin_active' ) ) {
			if ( is_plugin_active( 'tutor/tutor.php' ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Check if edd is installed.
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_jetform_active() {
        if ( in_array( 'jetformbuilder/jet-form-builder.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) { //phpcs:ignore
			return true;
		} elseif ( function_exists( 'is_plugin_active' ) ) {
			if ( is_plugin_active( 'jetformbuilder/jet-form-builder.php' ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Check if the Fluent Forms plugin is active on the WordPress site.
	 *
	 * @return bool Returns true if the Fluent Forms plugin is active, false otherwise.
	 * @since  1.2.7
	 */
	public static function is_fluentform_active() {
		if ( defined( 'FLUENTFORM' ) ) {
			return true;
		}
		return false;
	}

		/**
	 * Check if the Fluent booking plugin is active on the WordPress site.
	 *
	 * @return bool Returns true if the Fluent booking plugin is active, false otherwise.
	 * @since  1.18.2
	 */
	public static function is_fluent_booking_active() {

		if (in_array('fluent-booking/fluent-booking.php', apply_filters('active_plugins', get_option('active_plugins')))) {
			return true;
		}
	
		return false;
	}

	/**
	 * Check if LearnDash LMS is active.
	 *
	 * @return bool True if LearnDash is active, false otherwise.
	 * @since 1.7.1
	 */
	public static function is_learndash_lms_active() {
		// Check if LearnDash version constant is defined.
		if ( defined( 'LEARNDASH_VERSION' ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Check if the Contact Form 7 plugin is active on the WordPress site.
	 *
	 * @return bool Returns true if the Contact Form 7 plugin is active, false otherwise.
	 * @since  1.5.12
	 */
	public static function is_contact_form_7_active() {
		if ( defined( 'WPCF7_VERSION' ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Get all gravity forms.
	 */
	public static function get_all_gform_forms() {
		if ( self::is_gform_active() ) {
			if ( class_exists( 'GFFormsModel' ) ) {
				$forms = \GFFormsModel::get_forms( true, 'title', 'ASC', false );
				if ( is_array( $forms ) ) {
					$formatted_forms = array(
						array(
							'value' => '',
							'label' => 'Select Form',
						),
					);
					foreach ( $forms as $form ) {
						if ( isset( $form->id, $form->title ) ) {
							$array = array(
								'value' => $form->id,
								'label' => $form->title,
							);
							array_push( $formatted_forms, $array );
						}
					}
					return $formatted_forms;
				}
			}
		}
		return false;
	}


	/**
	 * Encrypt key with AES.
	 *
	 * @param string $key key.
	 * @return string
	 */
	public static function encrypt_key( $key ) {
		$encrypted_key = \MintMailPro\Mint_AES_Encription\Mint_Aes_Ctr::encrypt( $key, '', 256 );
		return $encrypted_key;
	}


	/**
	 * Decrypt a key with AES.
	 *
	 * @param string $key Key.
	 * @return string
	 */
	public static function decrypt_key( $key ) {
		$encrypted_key = \MintMailPro\Mint_AES_Encription\Mint_Aes_Ctr::decrypt( $key, '', 256 );
		return $encrypted_key;
	}


	/**
	 * Fetch product by term.
	 *
	 * @param string $term Search term.
	 * @param string $type Search type.
	 * @return array
	 * @since 1.0.0
	 */
	public static function retrieve_product( $term = '', $type = 'wc' ) {
		$products = array();
		if ( $term && $type ) {
			$function_name = 'get_' . $type . '_product';
			$products      = self::$function_name( $term );
		}
		return $products;
	}

	/**
	 * Retrieve LMS courses based on the search term and type.
	 * 
	 * @param string $term The search term to filter the courses by.
	 * @param string $type The type of LMS courses to retrieve.
	 *
	 * @return array The formatted LMS courses.
	 * @since 1.17.2
	 */
	public static function retrieve_lms_courses($term = '', $type = 'sfwd-courses'){
		$posts = get_posts(
			array(
				'post_type'   => $type,
				'numberposts' => -1,
				'orderby'     => 'created_at',
				'order'       => 'DESC',
				'post_status' => 'publish',
				's'           => $term,
			)
		);

		$formatted_courses[] = array(
			'value'   => 0,
			'label'   => __('Select', 'mrm'),
		);

		if (is_array($posts)) {			
			foreach ($posts as $post) {
				$formatted_courses[] = array(
					'value'   => $post->ID,
					'label'   => $post->post_title,
				);
			}
		}

		return $formatted_courses;
	}

	/**
	 * Retrieve LMS groups based on the search term and type.
	 * 
	 * @param string $term The search term to filter the groups by.
	 * @param string $type The type of LMS groups to retrieve.
	 *
	 * @return array The formatted LMS groups.
	 * @since 1.17.2
	 */
	public static function retrieve_lms_groups($term = '', $type = 'groups'){
		$posts = get_posts(
			array(
				'post_type'   => $type,
				'numberposts' => -1,
				'orderby'     => 'created_at',
				'order'       => 'DESC',
				'post_status' => 'publish',
				's'           => $term,
			)
		);

		$formatted_groups = array();

		if (is_array($posts) && !empty($posts)) {			
			foreach ($posts as $post) {
				$formatted_groups[] = array(
					'value'   => $post->ID,
					'label'   => $post->post_title,
				);
			}
		}

		return $formatted_groups;
	}

	/**
	 * Get Wc products.
	 *
	 * @param string $term Get WC Term.
	 * @return array
	 */
	public static function get_wc_product( $term ) {
		$products = array();
		if ( $term ) {
			$data_store      = \WC_Data_Store::load( 'product' );
			$ids             = $data_store->search_products( $term, '', false, false, 10 );
			$product_objects = array_filter( array_map( 'wc_get_product', $ids ), 'wc_products_array_filter_readable' );
			if ( is_array( $product_objects ) ) {
				foreach ( $product_objects as $product_object ) {
					if ( $product_object && ( ( $product_object->managing_stock() && $product_object->get_stock_quantity() > 0 ) || ( !$product_object->managing_stock() && $product_object->get_stock_status() !== 'outofstock' ) ) ) {
						$formatted_name = $product_object->get_name();

						if ( $product_object->get_type() === 'variable' || $product_object->get_type() === 'variable-subscription' ) {
							$variations = $product_object->get_available_variations();
							$parent_id  = $product_object->get_id();
							$products[] = array(
								'value' => $parent_id,
								'label' => $formatted_name,
							);

							if ( !empty( $variations ) ) {
								foreach ( $variations as $variation ) {
									$variation_product = wc_get_product( $variation['variation_id'] );

									if ( $variation_product ) {
										if ( ( $variation_product->managing_stock() && $variation_product->get_stock_quantity() > 0 ) || ( !$variation_product->managing_stock() && $variation_product->get_stock_status() !== 'outofstock' ) ) {
											$products[] = array(
												'value' => $variation['variation_id'],
												'label' => self::get_formated_product_name( $variation_product ),
											);
										}
									}
								}
							}
						} else {
							$products[] = array(
								'value' => $product_object->get_id(),
								'label' => rawurldecode( $formatted_name ),
							);
						}
					}
				}
			}
		}
		return $products;
	}

	/**
	 * Get Wc products.
	 *
	 * @param string $term Term for EDD product.
	 * @return array
	 */
	public static function get_edd_product( $term ) {
		$formatted_products = array();
		$posts              = get_posts(
			array(
				'numberposts' => -1,
				'post_type'   => 'download',
				'post_status' => 'publish',
				'fields'      => 'ID',
				's'           => $term,
				'orderby'     => 'date',
				'order'       => 'ASC',
			)
		);
		if ( is_array( $posts ) ) {
			foreach ( $posts as $product ) {
				if ( isset( $product->ID ) ) {
					$formatted_products[] = array(
						'value' => $product->ID,
						'label' => $product->post_title,
					);
				}
			}
		}

		return $formatted_products;
	}


	/**
	 * Get formated product name.
	 *
	 * @param Object $product Get product object.
	 * @param array  $formatted_attr Get formatted array.
	 *
	 * @return String
	 */
	public static function get_formated_product_name( $product, $formatted_attr = array() ) {
		$_product        = wc_get_product( $product );
		$each_child_attr = array();
		$_title          = '';
		if ( $_product ) {
			if ( !$formatted_attr ) {
				if ( 'variable' === $_product->get_type() || 'variation' === $_product->get_type() || 'subscription_variation' === $_product->get_type() || 'variable-subscription' === $_product->get_type() ) {
					$attr_summary = $_product->get_attribute_summary();
					$attr_array   = explode( ',', $attr_summary );

					foreach ( $attr_array as $ata ) {
						$attr              = strpbrk( $ata, ':' );
						$each_child_attr[] = $attr;
					}
				}
			} else {
				foreach ( $formatted_attr as $attr ) {
					$each_child_attr[] = ucfirst( $attr );
				}
			}
			if ( $each_child_attr ) {
				$each_child_attr_two = array();
				foreach ( $each_child_attr as $eca ) {
					$each_child_attr_two[] = str_replace( ': ', ' ', $eca );
				}
				$_title = $_product->get_title() . ' - ';
				$_title = $_title . implode( ', ', $each_child_attr_two );
			} else {
				$_title = $_product->get_title();
			}
		}

		return $_title;
	}


	/**
	 * Check if wpfunnels is installed
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public static function is_wpf_free_pro_active() {
		$is_wpf_pro_activated  = apply_filters( 'is_wpf_pro_active', false );
		$is_wpf_free_activated = false;
        if ( in_array( 'wpfunnels/wpfnl.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) { //phpcs:ignore
			$is_wpf_free_activated = true;
		} elseif ( function_exists( 'is_plugin_active' ) ) {
			if ( is_plugin_active( 'wpfunnels/wpfnl.php' ) ) {
				$is_wpf_free_activated = true;
			}
		}
		if ( $is_wpf_pro_activated && $is_wpf_free_activated ) {
			return true;
		}
		return false;
	}

	/**
	 * Retrieves group IDs associated with a specific email and type.
	 *
	 * @param string $email The email address to search for.
	 * @param string $type The type of contact group to filter by.
	 *
	 * @return array An array of group IDs associated with the given email and type.
	 * @since 1.3.1
	 */
	public static function get_group_ids_by_email( $email, $type ) {
		global $wpdb;

		$query = $wpdb->prepare(
			"SELECT distinct(cg.id)
			FROM {$wpdb->prefix}mint_contacts AS c
			INNER JOIN {$wpdb->prefix}mint_contact_group_relationship AS cgr ON c.id = cgr.contact_id
			INNER JOIN {$wpdb->prefix}mint_contact_groups AS cg ON cgr.group_id = cg.id
			WHERE cg.type IN (%s) and c.email = %s",
			$type,
			$email
		);

		return array_values( array_column( $wpdb->get_results( $query, ARRAY_A ), 'id' ) ); //phpcs:ignore
	}

	/**
	 * Retrieves the name associated with a specific email and type.
	 *
	 * @param string $email The email address to search for.
	 * @param string $type The type of contact to filter by.
	 *
	 * @return string The name associated with the given email and type. Returns an empty string if not found.
	 * @since 1.3.1
	 */
	public static function get_name_by_email( $email, $type ) {
		global $wpdb;
		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$query  = $wpdb->prepare(
			"SELECT {$type}
			FROM {$wpdb->prefix}mint_contacts WHERE email = %s",
			$email
		);
		$result = $wpdb->get_row( $query, ARRAY_A ); //phpcs:ignore
		return isset( $result[ $type ] ) ? $result[ $type ] : '';
		// phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Fetch product by term.
	 *
	 * @param string $term Search term.
	 * @param string $type Search type.
	 * @return array
	 * @since 1.0.0
	 */
	public static function retrieve_category( $term = '', $type = 'wc' ) {
		$category = array();
		if ( $term && $type ) {
			$function_name = 'get_' . $type . '_category';
			$category      = self::$function_name( $term );
		}
		return $category;
	}
	/**
	 * Fetch WC by term.
	 *
	 * @param string $cat_name Search term.
	 * @return array
	 * @since 1.5.5
	 */
	public static function get_wc_category( $cat_name ) {
		global $wpdb;
		$data       = array();
		$cat_args   = "SELECT * 
             FROM $wpdb->terms AS t 
             INNER JOIN $wpdb->term_taxonomy AS tx ON t.term_id = tx.term_id 
             WHERE tx.taxonomy = 'product_cat' 
             AND t.name LIKE '%" . $cat_name . "%' "; //phpcs:ignore
		$categories = $wpdb->get_results( $cat_args, OBJECT ); //phpcs:ignore
		if ( !empty( $categories ) && is_array( $categories ) ) {
			foreach ( $categories as $category ) {
				$data[] = array(
					'value' => $category->term_id,
					'label' => rawurldecode( $category->name ),
				);
			}
		}
		return $data;
	}

	/**
	 * Fetch EDD by term.
	 *
	 * @param string $cat_name Search term.
	 * @return array
	 * @since 1.5.5
	 */
	public static function get_edd_category( $cat_name ) {
		global $wpdb;
		$data       = array();
		$cat_args   = "SELECT * 
             FROM $wpdb->terms AS t 
             INNER JOIN $wpdb->term_taxonomy AS tx ON t.term_id = tx.term_id 
             WHERE tx.taxonomy = 'download_category' 
             AND t.name LIKE '%" . $cat_name . "%' "; //phpcs:ignore
		$categories = $wpdb->get_results( $cat_args, OBJECT ); //phpcs:ignore
		if ( !empty( $categories ) && is_array( $categories ) ) {
			foreach ( $categories as $category ) {
				$data[] = array(
					'value' => $category->term_id,
					'label' => rawurldecode( $category->name ),
				);
			}
		}
		return $data;
	}

	/**
	 * Retrieve WooCommerce product tags based on the term and type.
	 *
	 * @param string $term The search term.
	 * @param string $type The type of product tags to retrieve.
	 * @return array The retrieved product tags.
	 * 
	 * @since 1.15.5
	 */
	public static function retrieve_tags($term, $type){
		// Query WooCommerce product tags
		$args = array(
			'taxonomy'   =>  'edd' === $type ? 'download_tag' : 'product_tag',
			'hide_empty' => false,
			'search'     => $term,
		);

		$tags = get_terms($args);

		// Map the tags to value-label format.
		$data = array();
		if (!empty($tags) && is_array($tags)) {
			foreach ($tags as $tag) {
				$data[] = array(
					'value' => $tag->term_id,
					'label' => rawurldecode($tag->name),
				);
			}
		}

		// Return the formatted tags.
		return $data;
	}

	/**
	 * Check if MemberPress is active.
	 *
	 * @return bool True if MemberPress is active, false otherwise.
	 * @since 1.8.0
	 */
	public static function is_memberpress_active() {
		// Check if LearnDash version constant is defined.
		if ( defined( 'MEPR_PLUGIN_NAME' ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Calculates the starting timestamp for an action scheduler based on a scheduled date.
	 *
	 * @param string $schedule_date The scheduled date in 'Y-m-d H:i:s' format.
	 * @return int The starting timestamp for the action scheduler.
	 *
	 * @since 1.11.0
	 */
	public static function get_action_scheduler_starting_timestamp( $schedule_date ) {
		$current_date  = new \DateTime( 'now', wp_timezone() );
		$schedule_date = date_create( gmdate( 'Y-m-d H:i:s', strtotime( $schedule_date ) ), wp_timezone() );

		// Handle the case where $schedule_date is in the past.
		if ( $schedule_date < $current_date ) {
			return time();
		}

		$date_diff = date_diff( $schedule_date, $current_date );
		$date_diff = '+' . $date_diff->y . 'year' . $date_diff->m . 'month' . $date_diff->d . 'day' . $date_diff->h . 'hour' . $date_diff->i . 'minute' . ( $date_diff->s + 1 ) . 'second';
		return strtotime( $date_diff );
	}

	/**
	 * Check if LifterLMS is active.
	 *
	 * @return bool True if LifterLMS is active, false otherwise.
	 * @since 1.12.0
	 */
	public static function is_lifter_lms_active() {
		// Check if LearnDash version constant is defined.
		if ( defined( 'LLMS_PLUGIN_FILE' ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Retrieves the latest content for the email body.
	 * This function is used to replace the content of the latest content block in the email body.
	 * 
	 * @param string $email_body The email body.
	 * @return string The email body with the latest content block replaced.
	 * @since 1.13.0
	 */
	public static function replace_automatic_latest_content( $email_body, $post_type = 'post' ) {
		if ( !class_exists( 'DOMDocument' ) ) {
			return $email_body;
		}

		// Initialize variables.
		$output = '';

		// Load the email body into a DOMDocument for manipulation.
		$dom = new DOMDocument();

		@$dom->loadHTML( $email_body );

		$xpath = new DOMXPath( $dom );

		$blocks = $xpath->query( '//div[contains(@class, "mint-latest-content-block")]' );

		if ( 0 === $blocks->length ) {
			return $email_body;
		}

		// Initialize the total number of child elements.
		$total_children      = 0;
		$first_child_content = '';

		if ( $blocks->length > 0 ) {
			$block = $blocks->item( 0 ); // Get the first node.

			// Query for the direct children of the block.
			$children = $xpath->query( 'div', $block );

			$total_children = $children->length; // This is the total number of children.

			if ( $total_children > 0 ) {
				$first_child         = $children->item( 0 ); // This is the first child.
				$first_child_content = $dom->saveHTML( $first_child ); // This is the HTML content of the first child.
			}
		}

		$args = array(
			'post_type'      => $post_type,
			'post_status'    => 'publish',
			'posts_per_page' => $total_children,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);
		$posts_query = new WP_Query( $args );
		$posts       = array();
		if ( $posts_query->have_posts() ) {
			while ( $posts_query->have_posts() ) {
				$posts_query->the_post();
				$post = array(
					'title'     => get_the_title(),
					'excerpt'   => strip_shortcodes( get_the_excerpt() ),
					'content'   => strip_shortcodes( get_the_content() ),
					'permalink' => get_the_permalink(),
					'thumbnail' => empty( get_the_post_thumbnail_url() ) ? MRM_DIR_URL . 'admin/assets/images/mint-placeholder.png' : get_the_post_thumbnail_url(),
				);
				array_push( $posts, $post );
			}
			wp_reset_postdata();
		}
		array_walk(
			$posts,
			function( &$post ) {
				$content         = preg_replace( '/<img[^>]+\>/i', ' ', $post['content'] );
				$content         = apply_filters( 'the_content', $content );
				$content         = str_replace( ']]>', ']]>', $content );
				$post['content'] = $content;
			}
		);
		// Process each cart item and replace placeholders in the email body.
		foreach ( $posts as $post ) {
			$doc = new DOMDocument();
			$doc->loadHTML( $first_child_content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD | LIBXML_NOERROR );
			$doc->substituteEntities = false; // phpcs:ignore
			$elements                = $doc->getElementsByTagName( 'td' );

			// Update the content for product title, price, and image.
			foreach ( $elements as $element ) {
				if ( $element->hasAttribute( 'class' ) && strpos( $element->getAttribute( 'class' ), 'mint-latest-content-title' ) !== false ) {
					$inner_div = $element->getElementsByTagName( 'div' )->item( 0 );
					if ( $inner_div ) {
						$inner_div->nodeValue = $post['title']; // phpcs:ignore
					}
				}

				if ( $element->hasAttribute( 'class' ) && strpos( $element->getAttribute( 'class' ), 'mint-latest-content-excerpt' ) !== false ) {
					$inner_div = $element->getElementsByTagName( 'div' )->item( 0 );
					if ( $inner_div ) {
						$inner_div->nodeValue = $post['excerpt']; // phpcs:ignore
					}
				}

				if ( $element->hasAttribute( 'class' ) && strpos( $element->getAttribute( 'class' ), 'mint-latest-content-content' ) !== false ) {
					$inner_div = $element->getElementsByTagName( 'div' )->item( 0 );
					if ( $inner_div ) {
						$inner_div->nodeValue = $post['content']; // phpcs:ignore
					}
				}

				if ( $element->hasAttribute( 'class' ) && strpos( $element->getAttribute( 'class' ), 'mint-latest-content-permalink' ) !== false ) {
					// Find anchor tags within the current element
					$anchors = $element->getElementsByTagName( 'a' );

					foreach ( $anchors as $anchor ) {
						// Set the href attribute for each anchor tag
						$anchor->setAttribute( 'href', $post['permalink'] );
					}
				}

				if ( $element->hasAttribute( 'class' ) && strpos( $element->getAttribute( 'class' ), 'mint-latest-content-image' ) !== false ) {
					$imgs = $element->getElementsByTagName( 'img' );
					foreach ( $imgs as $img ) {
						$img->setAttribute( 'src', $post['thumbnail'] );
					}
				}
			}

			$current_content = $doc->saveHTML();
			$output         .= $current_content;
		}

		// Replace the content of cart blocks in the email body with updated cart items.
		foreach ( $blocks as $index => $order_block ) {
			// Remove the original content
			while ( $order_block->firstChild ) {
				$order_block->removeChild( $order_block->firstChild );
			}

			// Create a new DOMDocument for the updated content.
			$new_doc = new DOMDocument();
			@$new_doc->loadHTML( $output );

			// Import the nodes of the updated content to the original DOMDocument.
			foreach ( $new_doc->getElementsByTagName( 'body' )->item( 0 )->childNodes as $node ) {
				$order_block->appendChild( $dom->importNode( $node, true ) );
			}
		}

		// Clean up and return the updated email body.
		$email_body = $dom->saveHTML();
		$email_body = str_replace( '%7B%7B', '{{', $email_body );
		$email_body = str_replace( '%7D%7D', '}}', $email_body );
		return $email_body;
	}

	/**
	 * Check if Bricks is active.
	 *
	 * @return bool True if Bricks is active, false otherwise.
	 * @since 1.14.0
	 */
	public static function is_bricks_active() {
        return wp_get_theme()->get_template() === 'bricks';
    }

	/**
	 * Check if WooCommerce Subscription is active.
	 *
	 * @return bool True if WooCommerce Subscription is active, false otherwise.
	 * @since 1.15.0
	 */
	public static function is_woocommerce_subscription_active() {
		if ( class_exists( 'WC_Subscriptions' ) && function_exists( 'wcs_get_subscription_statuses' ) ) {
			return true;
		}
		$active_plugins = (array) get_option( 'active_plugins', array() );
		return in_array( 'woocommerce-subscriptions/woocommerce-subscriptions.php', $active_plugins, true ) || array_key_exists( 'woocommerce-subscriptions/woocommerce-subscriptions.php', $active_plugins );
	}
	/**
	 * Get WooCommerce Subscription order statuses.
	 *
	 * @return array An array of WooCommerce Subscription order statuses.
	 * @since 1.15.0
	 */
	public static function get_wcs_order_statuses() {
		if ( ! self::is_woocommerce_subscription_active() ) {
			return false;
		}

		$order_statuses = wcs_get_subscription_statuses();

		$formatted_statuses[] = array(
			'value'   => 'wc-any',
			'label'   => 'Any',
		);
		foreach ( $order_statuses as $key => $value ) {
			$formatted_statuses[] = array(
				'value' => $key,
				'label' => $value,
			);
		}

		return $formatted_statuses;
	}

	/**
	 * Check if WooCommerce Membership is active.
	 *
	 * @return bool True if WooCommerce Membership is active, false otherwise.
	 * @since 1.15.0
	 */
	public static function is_woocommerce_membership_active() {
		$active_plugins = (array) get_option( 'active_plugins', array() );
		return in_array( 'woocommerce-memberships/woocommerce-memberships.php', $active_plugins, true ) || array_key_exists( 'woocommerce-memberships/woocommerce-memberships.php', $active_plugins );
	}

	/**
	 * Get WooCommerce Membership plans.
	 *
	 * @return array An array of WooCommerce Membership plans.
	 * @since 1.15.0
	 */
	public static function get_wcm_plans() {
		if ( ! self::is_woocommerce_membership_active() ) {
			return false;
		}

		$options = array();

		foreach( wc_memberships_get_membership_plans() as $plan ) {
			$options[] = array(
				'value' => (int)$plan->get_id(),
				'label' => $plan->get_name(),
			);
		}

		return $options;
	}

	/**
	 * Get WooCommerce Membership plan statuses.
	 *
	 * @return array An array of WooCommerce Membership order statuses.
	 * @since 1.15.0
	 */
	public static function get_wcm_plan_statuses() {
		if ( ! self::is_woocommerce_membership_active() ) {
			return false;
		}

		$formatted_statuses[] = array(
			'value'   => 'any',
			'label'   => 'Any',
		);

		foreach ( wc_memberships_get_user_membership_statuses() as $status => $value ) {
			$status = 0 === strpos( $status, 'wcm-' ) ? substr( $status, 4 ) : $status;
			$formatted_statuses[] = array(
				'value' => $status,
				'label' => $value['label'],
			);
		}

		return $formatted_statuses;
	}

	/**
	 * Check if WooCommerce Wishlist is active.
	 *
	 * @return bool True if WooCommerce Wishlist is active, false otherwise.
	 * @since 1.15.0
	 */
	public static function is_woocommerce_wishlist_active() {
		if ( class_exists( 'WC_Wishlists_Plugin' ) ) {
			return true;
		}

		$active_plugins = (array) get_option( 'active_plugins', array() );
		return in_array( 'woocommerce-wishlists/woocommerce-wishlists.php', $active_plugins, true ) || array_key_exists( 'woocommerce-wishlists/woocommerce-wishlists.php', $active_plugins );
	}

	/**
	 * Check if WooCommerce Shipment Tracking is active.
	 *
	 * @return bool True if WooCommerce Shipment Tracking is active, false otherwise.
	 * @since 1.18.6
	 */
	public static function is_wc_shipment_tracking_active(){
		$active_plugins = (array) get_option('active_plugins', array());
		return in_array('woocommerce-shipment-tracking/woocommerce-shipment-tracking.php', $active_plugins, true) || array_key_exists('woocommerce-shipment-tracking/woocommerce-shipment-tracking.php', $active_plugins);
	}

	/**
	 * Check if WooCommerce Advanced Shipment Tracking is active.
	 *
	 * @return bool True if WooCommerce Shipment Tracking is active, false otherwise.
	 * @since 1.18.6
	 */
	public static function is_wc_advanced_shipment_tracking_active(){
		$active_plugins = (array) get_option('active_plugins', array());
		return in_array('woo-advanced-shipment-tracking/woocommerce-advanced-shipment-tracking.php', $active_plugins, true) || array_key_exists('woo-advanced-shipment-tracking/woocommerce-advanced-shipment-tracking.php', $active_plugins);
	}

	/**
	 * Processes the email body for lead-magnet tracking.
	 *
	 * This function searches for URLs in the email body containing the route=lead-magnet parameter,
	 * generates a unique token for each URL, replaces the 'lead_token' placeholder with the generated token,
	 * and inserts tracking information into the database.
	 *
	 * @param string $body The email body content.
	 * @return string The modified email body with lead-magnet tracking applied.
	 * 
	 * @since 1.15.1
	 */
	public static function process_lead_magnet_tracking($body, $email) {
		// Check if the URL contains route=lead-magnet.
		if ( strpos( $body, 'route=lead-magnet' ) !== false ) {
			$token = wp_generate_uuid4();
			$body  = str_replace('lead_token', $token, $body);

			global $wpdb;
			$table = $wpdb->prefix . LeadMagnet::$tracking_table_name;
			$args  = array(
				'email'      => $email,
				'token'      => $token,
				'created_at' => current_time('mysql'),
			);

			$wpdb->insert(
				$table,
				$args
			); // db call ok. ; no-cache ok.
		}

		return $body;
	}

	/**
	 * Prepare and return conditional fields for automation.
	 * 
	 * @return array An array of conditional fields for automation.
	 * @since 1.16.1
	 */
	public static function get_automation_condition_fields() {
		$custom_fields = CustomFieldModel::get_custom_fields_to_segment();
		$output = [
			'action' => __('Custom Fields', 'mint-mail-pro'),
			'values' => []
		];

		foreach ($custom_fields as $custom_field) {
			$conditions = FieldConditions::get_conditions_by_field_type($custom_field['type']);
			$meta_fields = isset($custom_field['meta']) ? maybe_unserialize($custom_field['meta']) : [];

			$field_data = [
				'name'       => $custom_field['label'],
				'param'      => $custom_field['value'],
				'relation'   => ['Custom Fields', $custom_field['value']],
				'conditions' => $conditions,
			];

			if (!empty($meta_fields['options'])) {
				$field_data['options'] = [];
				foreach ($meta_fields['options'] as $option) {
					$field_data['options'][] = [
						'value' => $option,
						'label' => $option
					];
				}
			}

			$output['values'][] = $field_data;
		}

		return array($output);
	}

	/**
	 * Check if WPForms is active.
	 * 
	 * @return bool
	 * @since 1.17.1
	 */
	public static function is_wp_form_active() {
        if ( defined( 'WPFORMS_VERSION' ) ) {
            return true;
        }

		$active_plugins = (array) get_option( 'active_plugins', array() );
        return in_array( 'wpforms-lite/wpforms.php', $active_plugins, true ) || array_key_exists( 'wpforms-lite/wpforms.php', $active_plugins );
	}

	/**
	 * Get text type fields.
	 *
	 * This function retrieves fields of type 'text', 'textArea', and 'number' from the primary contact fields
	 * and custom fields. It merges these fields into a single array and returns them.
	 *
	 * @return array An array of fields with 'name', 'slug', and 'type' keys.
	 * @since 1.17.2
	 */
	public static function get_text_type_fields(){
		$fields = get_option('mint_contact_primary_fields', Constants::$primary_contact_fields);
		$fields = array_merge(...array_values($fields));

		$text_fields = array();
		foreach ($fields as $field) {
			if ($field['type'] === 'text') {
				$text_fields[] = [
					'name' => $field['title'],
					'slug' => $field['slug'],
					'type' => $field['type']
				];
			}
		}

		$custom_fields = CustomFieldModel::get_custom_fields_to_map();

		if (!is_array($custom_fields) || empty($custom_fields)) {
			return $text_fields;
		}

		$custom_fields = array_filter(
			array_map(
				function ($custom_field) {
					if (!in_array($custom_field['type'], ['text', 'textArea'])) {
						return null;
					}
					$meta = isset($custom_field['meta']) ? maybe_unserialize($custom_field['meta']) : array();
					$custom_field['name'] = isset($meta['label']) ? $meta['label'] : '';
					unset($custom_field['meta']);
					return $custom_field;
				},
				$custom_fields
			)
		);

		return array_merge($text_fields, $custom_fields);
	}
}


