<?php
/**
 * Automatic Coupon Code Helper.
 *
 * @package MailMintPro\App\Utilities\Helper
 * @namespace MailMintPro\App\Utilities\Helper
 * @author [getwpfunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2023-12-28 11:03:17
 * @modify date 2023-12-28 11:03:17
 */

namespace MailMintPro\App\Utilities\Helper;

use MailMintPro\Mint\Internal\AbandonedCart\Helper\Common;

/**
 * MintAutomaticCoupon class
 *
 * MintAutomaticCoupon helper class.
 *
 * @package MailMintPro\App\Utilities\Helper
 * @namespace MailMintPro\App\Utilities\Helper
 *
 * @version 1.7.1
 */
class MintAutomaticCoupon {

	/**
	 * Summary: Get WooCommerce coupons.
	 *
	 * Description: This static method retrieves WooCommerce coupons by querying the 'shop_coupon' post type.
	 * It fetches all published coupons, orders them by creation date in descending order, and formats the results into an array of coupon values and labels.
	 *
	 * @access public
	 *
	 * @return array Returns an array of WooCommerce coupons with 'value' (coupon ID) and 'label' (coupon title) if coupons are found; otherwise, an empty array.
	 *
	 * @since 1.7.1
	 */
	public static function get_woocommerce_coupons() {
		$formatted_coupons = array();

		$posts = get_posts(
			array(
				'post_type'   => 'shop_coupon',
				'numberposts' => -1,
				'orderby'     => 'created_at',
				'order'       => 'DESC',
				'post_status' => 'publish',
			)
		);

		if ( $posts && is_array( $posts ) ) {
			foreach ( $posts as $post ) {
				$formatted_coupons[] = array(
					'value' => $post->ID,
					'label' => $post->post_title,
				);
			}
		}

		return $formatted_coupons;
	}

	/**
	 * Prepare coupon data for creating a WooCommerce coupon.
	 *
	 * Description: Formats and organizes coupon data based on provided settings.
	 *
	 * @access public
	 *
	 * @param string $email    The email associated with the coupon.
	 * @param array  $settings Coupon settings.
	 *
	 * @return array An array containing organized coupon data.
	 * @since 1.8.0
	 */
	public static function prepare_coupon_data( $email, $settings ) {
		$data_to_set          = array();
		$data_to_set['email'] = $email;

		$data_to_set['coupon']                  = array();
		$data_to_set['coupon']['discount_type'] = isset( $settings['general_settings']['type']['value'] ) ? $settings['general_settings']['type']['value'] : 'percent';
		$data_to_set['coupon']['prefix']        = isset( $settings['general_settings']['prefix'] ) ? $settings['general_settings']['prefix'] : '';
		$data_to_set['coupon']['coupon_amount'] = isset( $settings['general_settings']['amount'] ) ? $settings['general_settings']['amount'] : 0;
		$data_to_set['coupon']['free_shipping'] = isset( $settings['general_settings']['is_free_shipping'] ) ? $settings['general_settings']['is_free_shipping'] : 0;

		$no_expiry_date                  = isset( $settings['general_settings']['no_expiry_date'] ) ? $settings['general_settings']['no_expiry_date'] : 0;
		$data_to_set['coupon']['expiry'] = !$no_expiry_date ? gmdate( 'Y-m-d', strtotime( $settings['general_settings']['expiry_date'] ) ) : '';

		$data_to_set['coupon']['minimum_amount'] = isset( $settings['restrictions_settings']['minimum_spend'] ) ? $settings['restrictions_settings']['minimum_spend'] : 0;
		$data_to_set['coupon']['maximum_amount'] = isset( $settings['restrictions_settings']['maximum_spend'] ) ? $settings['restrictions_settings']['maximum_spend'] : 0;

		$data_to_set['coupon']['exclude_sale_items']     = isset( $settings['restrictions_settings']['exclude_sale'] ) ? $settings['restrictions_settings']['exclude_sale'] : 0;
		$data_to_set['coupon']['individual_use']         = isset( $settings['restrictions_settings']['individual_use'] ) ? $settings['restrictions_settings']['individual_use'] : 0;
		$data_to_set['coupon']['restrict_contact_email'] = isset( $settings['restrictions_settings']['restrict_contact_email'] ) ? $settings['restrictions_settings']['restrict_contact_email'] : 0;
		$data_to_set['coupon']['allowed_emails']         = isset( $settings['restrictions_settings']['allowed_emails'] ) ? $settings['restrictions_settings']['allowed_emails'] : '';

		if ( isset( $settings['restrictions_settings']['products'] ) && is_array( $settings['restrictions_settings']['products'] ) ) {
			$data_to_set['coupon']['product_ids'] = array_map(
				function ( $product ) {
					return isset( $product['value'] ) ? $product['value'] : 0;
				},
				$settings['restrictions_settings']['products']
			);
		}

		if ( isset( $settings['restrictions_settings']['exclude_products'] ) && is_array( $settings['restrictions_settings']['exclude_products'] ) ) {
			$data_to_set['coupon']['exclude_product_ids'] = array_map(
				function ( $product ) {
					return isset( $product['value'] ) ? $product['value'] : 0;
				},
				$settings['restrictions_settings']['exclude_products']
			);
		}

		if ( isset( $settings['restrictions_settings']['categories'] ) && is_array( $settings['restrictions_settings']['categories'] ) ) {
			$data_to_set['coupon']['product_categories'] = array_map(
				function ( $category ) {
					return isset( $category['value'] ) ? $category['value'] : 0;
				},
				$settings['restrictions_settings']['categories']
			);
		}

		if ( isset( $settings['restrictions_settings']['exclude_categories'] ) && is_array( $settings['restrictions_settings']['exclude_categories'] ) ) {
			$data_to_set['coupon']['exclude_product_categories'] = array_map(
				function ( $category ) {
					return isset( $category['value'] ) ? $category['value'] : 0;
				},
				$settings['restrictions_settings']['exclude_categories']
			);
		}

		$data_to_set['coupon']['usage_limit']          = isset( $settings['usage_limits_settings']['limit_per_coupon'] ) ? $settings['usage_limits_settings']['limit_per_coupon'] : '';
		$data_to_set['coupon']['usage_limit_per_user'] = isset( $settings['usage_limits_settings']['limit_per_user'] ) ? $settings['usage_limits_settings']['limit_per_user'] : '';
		return $data_to_set;
	}

	/**
	 * Create a coupon based on provided details.
	 *
	 * Description: Creates a coupon post with relevant metadata, including MailMint-specific details.
	 *
	 * @access public
	 * @param array $coupon_details Details for creating the coupon.
	 * @param int   $automation_id   ID of the associated automation.
	 * @param int   $step_id         ID of the associated automation step.
	 * @return bool True if the coupon is successfully created; otherwise, false.
	 * @since 1.8.0
	 */
	public static function create_coupon( $coupon_details, $automation_id, $step_id ) {
		// Apply filter to allow custom modification of the coupon metadata.
		$coupon_details = apply_filters( 'mailmint_before_create_woocommerce_coupon', $coupon_details, $automation_id );
		// Extract necessary details from the input.
		$coupon_prefix  = $coupon_details['coupon']['prefix'];
		$dynamic_string = Common::create_abandoned_cart_token( 6 );
		$coupon_name    = $coupon_prefix . $dynamic_string;

		// Prepare coupon metadata.
		$new_coupon_meta['is_mailmint_coupon']      = 1;
		$new_coupon_meta['mailmint_automation_id']  = $automation_id;
		$new_coupon_meta['mailmint_step_id']        = $step_id;
		$new_coupon_meta['expiry_date']             = '';
		$new_coupon_meta['date_expires']            = '';
		$new_coupon_meta['usage_count']             = isset( $data['coupon']['usage_limit'] ) ? $data['coupon']['usage_limit'] : '';
		$new_coupon_meta['customer_email']          = array();
		$new_coupon_meta['mailmint_customer_email'] = $coupon_details['email'];

		// Set coupon expiry date if provided.
		if ( isset( $coupon_details['coupon']['expiry'] ) && ! empty( $coupon_details['coupon']['expiry'] ) ) {
			$timestamp                       = strtotime( strval( $coupon_details['coupon']['expiry'] ) );
			$new_coupon_meta['expiry_date']  = $coupon_details['coupon']['expiry'];
			$new_coupon_meta['date_expires'] = $timestamp;
		}

		// Set up coupon post data.
		$args = array(
			'post_type'   => 'shop_coupon',
			'post_status' => 'publish',
			'post_title'  => $coupon_name,
		);
		$coupon_id = wp_insert_post( $args );

		// Check if coupon post insertion was successful.
		if ( ! is_wp_error( $coupon_id ) ) {
			$new_coupon_meta['usage_count'] = 0;
			if ( is_array( $new_coupon_meta ) && count( $new_coupon_meta ) > 0 ) {
				foreach ( $new_coupon_meta as $key => $val ) {
					update_post_meta( $coupon_id, $key, $val );
				}
			}
		}

		// Handle email restrictions for the created coupon.
		self::handle_coupon_emails_restriction( $coupon_id, $coupon_details );

		// Handle other coupon restrictions.
		self::handle_other_coupon_restriction( $coupon_id, $coupon_details['coupon'] );

		return true;
	}

	/**
	 * Handle coupon email restrictions.
	 *
	 * Description: Sets email restrictions for the given coupon based on provided details.
	 *
	 * @access public
	 * @param int   $coupon_id      ID of the coupon.
	 * @param array $coupon_details Details for handling email restrictions.
	 * @return \WC_Coupon|null WC_Coupon object if coupon is successfully updated; otherwise, null.
	 * @since 1.8.0
	 */
	public static function handle_coupon_emails_restriction( $coupon_id, $coupon_details ) {
		$emails = array();

		// Check if email restriction is enabled for the coupon.
		if ( isset( $coupon_details['coupon']['restrict_contact_email'] ) && 1 === absint( $coupon_details['coupon']['restrict_contact_email'] ) ) {
			$emails[] = $coupon_details['email'];
		}

		// Get allowed emails from coupon details.
		$allowed_emails = isset( $coupon_details['coupon']['allowed_emails'] ) ? $coupon_details['coupon']['allowed_emails'] : '';
		$allowed_emails = preg_split( '/,\s*|\s*,\s*/', $allowed_emails );

		$coupon = new \WC_Coupon( $coupon_id );
		if ( 0 === $coupon->get_id() ) {
			return;
		}

		// Set email restrictions for the coupon.
		$coupon->set_email_restrictions( array_merge( $emails, $allowed_emails ) );
		$coupon->save();

		return $coupon;
	}

	/**
	 * Handle other coupon restrictions.
	 *
	 * Description: Sets various coupon restrictions for the given coupon based on provided data.
	 *
	 * @access public
	 * @param int   $coupon_id   ID of the coupon.
	 * @param array $coupon_data Details for handling other coupon restrictions.
	 * @return \WC_Coupon|null WC_Coupon object if coupon is successfully updated; otherwise, null.
	 * @since 1.8.0
	 */
	public static function handle_other_coupon_restriction( $coupon_id, $coupon_data ) {
		// Check if coupon data is empty.
		if ( empty( $coupon_data ) ) {
			return;
		}

		$coupon = new \WC_Coupon( $coupon_id );

		// Set various coupon restrictions based on provided data.
		$coupon->set_description( __( 'Automated personalized coupon by Mail Mint', 'mailmint-pro' ) );
		$coupon->set_discount_type( $coupon_data['discount_type'] );
		$coupon->set_amount( $coupon_data['coupon_amount'] );
		$coupon->set_free_shipping( $coupon_data['free_shipping'] );

		if ( isset( $coupon_data['minimum_amount'] ) ) {
			$coupon->set_minimum_amount( $coupon_data['minimum_amount'] );
		}

		if ( isset( $coupon_data['maximum_amount'] ) ) {
			$coupon->set_maximum_amount( $coupon_data['maximum_amount'] );
		}

		if ( isset( $coupon_data['exclude_sale_items'] ) ) {
			$coupon->set_exclude_sale_items( $coupon_data['exclude_sale_items'] );
		}

		if ( isset( $coupon_data['individual_use'] ) ) {
			$coupon->set_individual_use( $coupon_data['individual_use'] );
		}

		if ( isset( $coupon_data['product_ids'] ) ) {
			$coupon->set_product_ids( $coupon_data['product_ids'] );
		}

		if ( isset( $coupon_data['exclude_product_ids'] ) ) {
			$coupon->set_excluded_product_ids( $coupon_data['exclude_product_ids'] );
		}

		if ( isset( $coupon_data['product_categories'] ) ) {
			$coupon->set_product_categories( $coupon_data['product_categories'] );
		}

		if ( isset( $coupon_data['exclude_product_categories'] ) ) {
			$coupon->set_excluded_product_categories( $coupon_data['exclude_product_categories'] );
		}

		if ( isset( $coupon_data['usage_limit'] ) ) {
			$coupon->set_usage_limit( $coupon_data['usage_limit'] );
		}

		if ( isset( $coupon_data['usage_limit_per_user'] ) ) {
			$coupon->set_usage_limit_per_user( $coupon_data['usage_limit_per_user'] );
		}

		$coupon->save();

		return $coupon;
	}

	/**
	 * Retrieve coupon titles associated with given step IDs.
	 *
	 * This function queries the WordPress database to fetch coupon titles based on
	 * the provided step IDs associated with the 'mailmint_step_id' post meta key.
	 *
	 * @param array $step_ids An array of step IDs for which coupon titles are requested.
	 * @return array An array of coupon titles associated with the given step IDs.
	 *
	 * @since 1.8.0
	 */
	public static function get_coupon_title( $step_ids ) {
		global $wpdb;

		$post_table      = $wpdb->prefix . 'posts';
		$post_meta_table = $wpdb->prefix . 'postmeta';
		$placeholders    = "'" . implode( "','", $step_ids ) . "'";

		$query = $wpdb->prepare( "SELECT wp_posts.post_title FROM {$post_table} AS wp_posts INNER JOIN {$post_meta_table} AS wp_postmeta ON wp_posts.ID = wp_postmeta.post_id WHERE wp_postmeta.meta_key = %s AND wp_postmeta.meta_value IN ($placeholders)", 'mailmint_step_id' ); //phpcs:ignore

		return $wpdb->get_results( $query, ARRAY_A ); //phpcs:ignore
	}
}
