<?php
/**
 * Analytics Helper.
 *
 * @package MailMintPro\App\Utilities\Helper
 * @namespace MailMintPro\App\Utilities\Helper
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-01-29 11:03:17
 * @modify date 2024-01-29 11:03:17
 */

namespace MailMintPro\App\Utilities\Helper;

use Mint\MRM\DataBase\Models\CampaignModel;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataBase\Tables\EmailSchema;
use Mint\MRM\DataBase\Models\EmailModel;
use Mint\MRM\DataBase\Tables\CampaignEmailBuilderSchema;
use Mint\MRM\DataBase\Tables\CampaignSchema;
use Mint\MRM\DataBase\Tables\EmailMetaSchema;
use MRM\Common\MrmCommon;

/**
 * Analytics class
 *
 * Analytics helper class.
 *
 * @package MailMintPro\App\Utilities\Helper
 * @namespace MailMintPro\App\Utilities\Helper
 *
 * @version 1.9.0
 */
class Analytics {

	/**
	 * Prepare a summary of campaign metrics.
	 *
	 * Calculates and retrieves various metrics summarizing the performance of a specific campaign.
	 *
	 * @param int $campaign_id The ID of the campaign for which to prepare the summary.
	 * @return array An associative array containing various campaign metrics.
	 *
	 * @since 1.9.0
	 */
	public static function prepare_campaign_summery( $campaign_id ) {
		$total_sent    = self::calculate_total_email_sent( $campaign_id );
		$total_success = EmailModel::count_delivered_status_on_campaign( $campaign_id, 'sent' );
		$success_rate  = self::calculate_email_delivery_success_rate( $total_sent, $total_success );
		$total_bounced = EmailModel::count_delivered_status_on_campaign( $campaign_id, 'failed' );
		$unsubscribe   = EmailModel::count_unsubscribe_on_campaign( $campaign_id );
		$total_open    = EmailModel::calculate_open_rate_on_campaign( $campaign_id );
		$total_click   = EmailModel::calculate_click_rate_on_campaign( $campaign_id );
		$ctor          = self::calculate_click_to_open_rate( $total_open, $total_click );
		$revenue       = self::calculate_campaign_total_revenue( $campaign_id );

		return array(
			'total_sent'    => $total_sent,
			'total_success' => $total_success,
			'success_rate'  => $success_rate,
			'total_bounced' => $total_bounced,
			'unsubscribe'   => $unsubscribe,
			'total_open'    => $total_open,
			'total_click'   => $total_click,
			'ctor'          => $ctor,
			'revenue'       => MrmCommon::price_format_with_wc_currency( $revenue ),
		);
	}

	/**
	 * Calculate the total number of emails sent for a specific campaign.
	 *
	 * This function queries the database to count the number of emails
	 * that have been sent for a given campaign, identified by its ID.
	 *
	 * @param int $campaign_id The ID of the campaign for which to calculate the total emails sent.
	 * @return int The total number of emails sent for the specified campaign.
	 *
	 * @since 1.9.0
	 */
	public static function calculate_total_email_sent( $campaign_id ) {
		global $wpdb;
		$table_name = $wpdb->prefix . EmailSchema::$table_name;
		return $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(id) as total_sent FROM $table_name WHERE campaign_id = %d", array( $campaign_id ) ) ); //phpcs:ignore
	}

	/**
	 * Calculate the success rate of email delivery.
	 *
	 * This function calculates the success rate as a percentage of the total number of emails sent
	 * that were successfully delivered. If no emails were sent, it returns 0.
	 *
	 * @param int $total_sent The total number of emails sent.
	 * @param int $total_success The number of emails that were successfully delivered.
	 * @return string The success rate formatted as a percentage with two decimal places.
	 *                If no emails were sent, returns '0.00'.
	 *
	 * @since 1.9.0
	 */
	public static function calculate_email_delivery_success_rate( $total_sent, $total_success ) {
		// Ensure that $total_sent is not zero to avoid division by zero.
		if ( $total_sent > 0 ) {
			return number_format( ( $total_success / $total_sent ) * 100, 2, '.', '' );
		} else {
			return number_format( 0, 2, '.', '' );
		}
	}

	/**
	 * Calculate the click-to-open rate for emails.
	 *
	 * This function calculates the click-to-open rate as a percentage of the total number of emails opened
	 * that were clicked. If no emails were opened, it returns 0.
	 *
	 * @param int $open The total number of emails opened.
	 * @param int $click The number of opened emails that were clicked.
	 * @return string The click-to-open rate formatted as a percentage with two decimal places.
	 *                If no emails were opened, returns '0.00'.
	 *
	 * @since 1.9.0
	 */
	public static function calculate_click_to_open_rate( $open, $click ) {
		// Ensure that $total_sent is not zero to avoid division by zero.
		if ( $open > 0 ) {
			return number_format( ( $click / $open ) * 100, 2, '.', '' );
		} else {
			return number_format( 0, 2, '.', '' );
		}
	}

	/**
	 * Calculate the total revenue generated by a specific campaign.
	 *
	 * This function queries the database to get the order IDs associated with emails sent as part of a given campaign.
	 * It then calculates the total revenue from these orders.
	 *
	 * @param int $campaign_id The ID of the campaign for which to calculate the total revenue.
	 * @return string The total revenue generated by the campaign, formatted as a price with the WooCommerce currency.
	 *
	 * @since 1.9.0
	 */
	public static function calculate_campaign_total_revenue( $campaign_id ) {
		global $wpdb;
		$email_table      = $wpdb->prefix . EmailSchema::$table_name;
		$email_meta_table = $wpdb->prefix . EmailMetaSchema::$table_name;

		$condition = "AND mint_email_id IN (SELECT id FROM {$email_table} WHERE campaign_id = '{$campaign_id}') ";

		$query  = 'SELECT meta_value ';
		$query .= "FROM {$email_meta_table} ";
		$query .= "WHERE meta_key =  'order_id' ";
		$query .= $condition;

		$order_ids = $wpdb->get_col( $query ); //phpcs:ignore

		if ( empty( $order_ids ) ) {
			return 0;
		}

		return EmailModel::get_total_revenue_from_email( $order_ids );
	}

	/**
	 * Prepare a summary of automation sequence metrics.
	 *
	 * Calculates and retrieves various metrics summarizing the performance of a specific campaign.
	 *
	 * @param int $campaign_id The ID of the campaign for which to prepare the summary.
	 * @return array An associative array containing various campaign metrics.
	 *
	 * @since 1.9.0
	 */
	public static function prepare_automation_summery( $campaign_id ) {
		$total_sent    = self::calculate_total_email_sent_from_automation_sequence( $campaign_id );
		$total_success = EmailModel::count_delivered_status_on_automation_sequence( $campaign_id, 'sent' );
		$success_rate  = self::calculate_email_delivery_success_rate( $total_sent, $total_success );
		$total_bounced = EmailModel::count_delivered_status_on_automation_sequence( $campaign_id, 'failed' );
		$unsubscribe   = EmailModel::count_email_metrics_on_automation_sequence( $campaign_id, 'is_unsubscribe' );
		$total_open    = EmailModel::count_email_metrics_on_automation_sequence( $campaign_id, 'is_open' );
		$total_click   = EmailModel::count_email_metrics_on_automation_sequence( $campaign_id, 'is_click' );
		$ctor          = self::calculate_click_to_open_rate( $total_open, $total_click );
		$revenue       = self::calculate_automation_sequence_total_revenue( $campaign_id );

		return array(
			'total_sent'    => $total_sent,
			'total_success' => $total_success,
			'success_rate'  => $success_rate,
			'total_bounced' => $total_bounced,
			'unsubscribe'   => $unsubscribe,
			'total_open'    => $total_open,
			'total_click'   => $total_click,
			'ctor'          => $ctor,
			'revenue'       => MrmCommon::price_format_with_wc_currency( $revenue ),
		);
	}

	/**
	 * Calculate the total number of emails sent from a specific campaign.
	 *
	 * This function queries the database to get the count of emails associated with a given campaign.
	 *
	 * @param int    $campaign_id The ID of the campaign for which to count the emails.
	 * @param string $email_status The status of the email.
	 *
	 * @return int The count of emails sent from the given campaign.
	 *
	 * @since 1.9.0
	 */
	public static function calculate_total_email_sent_from_automation_sequence( $campaign_id, $email_status = '' ) {
		global $wpdb;
		$email_table    = $wpdb->prefix . EmailSchema::$table_name;
		$campaign_table = $wpdb->prefix . CampaignSchema::$campaign_emails_table;

		// Prepare SQL query to count emails.
		$query = "SELECT COUNT(id) FROM {$email_table} WHERE email_id IN (SELECT id FROM {$campaign_table} WHERE campaign_id = %d)";

		// Add email status condition to the query if $email_status is not empty.
		if ( !empty( $email_status ) ) {
			$query .= $wpdb->prepare( ' AND status = %s', $email_status );
		}

		return $wpdb->get_var( $wpdb->prepare( $query, $campaign_id ) ); //phpcs:ignore
	}

	/**
	 * Calculate the total revenue generated by a specific campaign.
	 *
	 * This function queries the database to get the order IDs associated with emails sent as part of a given campaign.
	 * It then calculates the total revenue from these orders.
	 *
	 * @param int $campaign_id The ID of the campaign for which to calculate the total revenue.
	 * @return float The total revenue generated by the campaign. Returns 0 if no orders are associated with the campaign.
	 *
	 * @since 1.9.0
	 */
	public static function calculate_automation_sequence_total_revenue( $campaign_id ) {
		global $wpdb;
		$email_table      = $wpdb->prefix . EmailSchema::$table_name;
		$email_meta_table = $wpdb->prefix . EmailMetaSchema::$table_name;
		$campaign_table   = $wpdb->prefix . CampaignSchema::$campaign_emails_table;

		$query     = $wpdb->prepare( "SELECT meta_value FROM {$email_meta_table} WHERE meta_key =  'order_id' AND mint_email_id IN (SELECT id FROM {$email_table} WHERE email_id IN ( SELECT id FROM {$campaign_table} WHERE campaign_id = %d ) )", $campaign_id ); //phpcs:ignore
		$order_ids = $wpdb->get_col( $query ); //phpcs:ignore

		if ( empty( $order_ids ) ) {
			return 0;
		}

		return EmailModel::get_total_revenue_from_email( $order_ids );
	}

	/**
	 * Retrieves campaign email recipients.
	 *
	 * This method retrieves the campaign email recipients from the database using the provided campaign ID, offset, and per page values.
	 * It then enriches the emails with additional data such as contact name, open time, and click time.
	 *
	 * @param int    $campaign_id The ID of the campaign.
	 * @param int    $offset The offset for the query.
	 * @param int    $per_page The number of items per page.
	 * @param string $email_status The status of the email.
	 *
	 * @return array An array of emails with additional data.
	 *
	 * @since 1.9.0
	 */
	public static function retrieve_campaign_email_recipients( $campaign_id, $offset, $per_page, $email_status = '' ) {
		global $wpdb;
		// Define table names.
		$email_table    = $wpdb->prefix . EmailSchema::$table_name;
		$campaign_table = $wpdb->prefix . CampaignSchema::$campaign_emails_table;
		$builder_table  = $wpdb->prefix . CampaignEmailBuilderSchema::$table_name;

		// Prepare SQL query to retrieve emails.
		$query = "SELECT BET.*, BET.id AS broadcast_email_id, CET.sender_email, CET.sender_name, CET.email_subject, CET.email_preview_text, CEB.email_body as body_data
                    FROM {$email_table} AS BET
                    LEFT JOIN {$campaign_table} AS CET
                    ON BET.email_id = CET.id
                    LEFT JOIN {$builder_table} AS CEB
                    ON BET.email_id = CEB.email_id
                    WHERE BET.campaign_id = %d";

		// Add email status condition to the query if $email_status is not empty.
		if ( !empty( $email_status ) ) {
			$query .= $wpdb->prepare( ' AND BET.status = %s', $email_status );
		}

		$query .= $wpdb->prepare( ' LIMIT %d, %d', $offset, $per_page );

		// Execute the query and get the results.
		$emails = $wpdb->get_results( $wpdb->prepare( $query, $campaign_id ), ARRAY_A ); //phpcs:ignore

		// Enrich emails with contact name.
		$emails = array_map(
			function( $email ) {
				$email_address         = isset( $email['email_address'] ) ? $email['email_address'] : '';
				$email['contact_name'] = ContactModel::get_contact_full_name( $email_address );
				$email['sending_time'] = MrmCommon::date_time_format_with_core( $email['scheduled_at'] );
				return $email;
			},
			$emails
		);

		// Enrich emails with open time and click time.
		$emails = EmailModel::get_broadcast_emails_open_time( '', $emails );
		$emails = EmailModel::get_broadcast_emails_click_time( '', $emails );
		return $emails;
	}

	/**
	 * Retrieves campaign email recipients.
	 *
	 * This method retrieves the campaign email recipients from the database using the provided campaign ID, offset, and per page values.
	 * It then enriches the emails with additional data such as contact name, open time, and click time.
	 *
	 * @param int    $campaign_id The ID of the campaign.
	 * @param int    $offset The offset for the query.
	 * @param int    $per_page The number of items per page.
	 * @param string $email_status The status of the email.
	 *
	 * @return array An array of emails with additional data.
	 *
	 * @since 1.9.0
	 */
	public static function retrieve_automation_email_recipients( $campaign_id, $offset, $per_page, $email_status = '' ) {
		global $wpdb;
		// Define table names.
		$email_table    = $wpdb->prefix . EmailSchema::$table_name;
		$campaign_table = $wpdb->prefix . CampaignSchema::$campaign_emails_table;
		$builder_table  = $wpdb->prefix . CampaignEmailBuilderSchema::$table_name;

		// Prepare SQL query to retrieve emails.
		$query = "SELECT BET.*, BET.id AS broadcast_email_id, CET.sender_email, CET.sender_name, CET.email_subject, CET.email_preview_text, CEB.email_body as body_data
                    FROM {$email_table} AS BET
                    LEFT JOIN {$campaign_table} AS CET
                    ON BET.email_id = CET.id
                    LEFT JOIN {$builder_table} AS CEB
                    ON BET.email_id = CEB.email_id
                    WHERE BET.email_id IN ( SELECT id FROM {$campaign_table} WHERE campaign_id = %d )";

		// Add email status condition to the query if $email_status is not empty.
		if ( !empty( $email_status ) ) {
			$query .= $wpdb->prepare( ' AND BET.status = %s', $email_status );
		}

		$query .= $wpdb->prepare( ' LIMIT %d, %d', $offset, $per_page );

		// Execute the query and get the results.
		$emails = $wpdb->get_results( $wpdb->prepare( $query, $campaign_id ), ARRAY_A ); //phpcs:ignore

		// Enrich emails with contact name.
		$emails = array_map(
			function( $email ) {
				$email_address         = isset( $email['email_address'] ) ? $email['email_address'] : '';
				$email['contact_name'] = ContactModel::get_contact_full_name( $email_address );
				$email['sending_time'] = MrmCommon::date_time_format_with_core( $email['scheduled_at'] );
				return $email;
			},
			$emails
		);

		// Enrich emails with open time and click time.
		$emails = EmailModel::get_broadcast_emails_open_time( '', $emails );
		$emails = EmailModel::get_broadcast_emails_click_time( '', $emails );

		return $emails;
	}

	/**
	 * Retrieves campaign activity email list.
	 *
	 * This method retrieves the campaign activity email list from the database using the provided campaign ID, offset, and per page values.
	 * It then enriches the emails with additional data such as contact name, open time, and click time.
	 *
	 * @param int    $campaign_id The ID of the campaign.
	 * @param int    $offset The offset for the query.
	 * @param int    $per_page The number of items per page.
	 * @param string $type The type of the email.
	 *
	 * @return array An array of emails with additional data.
	 *
	 * @since 1.9.0
	 */
	public static function retrieve_campaign_activity_email_list( $campaign_id, $offset, $per_page, $type ) {
		global $wpdb;
		// Define table names.
		$email_table      = $wpdb->prefix . EmailSchema::$table_name;
		$email_meta_table = $wpdb->prefix . EmailMetaSchema::$table_name;

		// Prepare SQL query to retrieve emails.
		$query = "SELECT BET.*, BET.id AS broadcast_email_id
                    FROM {$email_table} AS BET
                    LEFT JOIN {$email_meta_table} AS BEMT
                    ON BET.id = BEMT.mint_email_id
                    WHERE BET.campaign_id = %d AND BEMT.meta_key = %s";

		$query .= $wpdb->prepare( ' LIMIT %d, %d', $offset, $per_page );

		// Execute the query and get the results.
		$emails = $wpdb->get_results( $wpdb->prepare( $query, $campaign_id, $type ), ARRAY_A ); //phpcs:ignore

		// Enrich emails with contact name.
		$emails = array_map(
			function( $email ) {
				$email_address         = isset( $email['email_address'] ) ? $email['email_address'] : '';
				$email['contact_name'] = ContactModel::get_contact_full_name( $email_address );
				$email['sending_time'] = MrmCommon::date_time_format_with_core( $email['scheduled_at'] );
				return $email;
			},
			$emails
		);

		return $emails;
	}

	/**
	 * Retrieves automation sequence activity email list.
	 *
	 * This method retrieves the campaign activity email list from the database using the provided campaign ID, offset, and per page values.
	 * It then enriches the emails with additional data such as contact name, open time, and click time.
	 *
	 * @param int    $campaign_id The ID of the campaign.
	 * @param int    $offset The offset for the query.
	 * @param int    $per_page The number of items per page.
	 * @param string $type The type of the email.
	 *
	 * @return array An array of emails with additional data.
	 *
	 * @since 1.9.0
	 */
	public static function retrieve_automation_sequence_activity_email_list( $campaign_id, $offset, $per_page, $type ) {
		global $wpdb;
		// Define table names.
		$email_table      = $wpdb->prefix . EmailSchema::$table_name;
		$email_meta_table = $wpdb->prefix . EmailMetaSchema::$table_name;
		$campaign_table   = $wpdb->prefix . CampaignSchema::$campaign_emails_table;

		// Prepare SQL query to retrieve emails.
		$query = "SELECT BET.*, BET.id AS broadcast_email_id
                    FROM {$email_table} AS BET
                    LEFT JOIN {$email_meta_table} AS BEMT
                    ON BET.id = BEMT.mint_email_id
                    WHERE BET.email_id IN ( SELECT id FROM {$campaign_table} WHERE campaign_id = %d ) AND BEMT.meta_key = %s";

		$query .= $wpdb->prepare( ' LIMIT %d, %d', $offset, $per_page );

		// Execute the query and get the results.
		$emails = $wpdb->get_results( $wpdb->prepare( $query, $campaign_id, $type ), ARRAY_A ); //phpcs:ignore

		// Enrich emails with contact name.
		$emails = array_map(
			function( $email ) {
				$email_address         = isset( $email['email_address'] ) ? $email['email_address'] : '';
				$email['contact_name'] = ContactModel::get_contact_full_name( $email_address );
				$email['sending_time'] = MrmCommon::date_time_format_with_core( $email['scheduled_at'] );
				return $email;
			},
			$emails
		);

		return $emails;
	}

	/**
	 * Retrieve campaign email link click performance.
	 *
	 * This method retrieves the open performance data for a specific campaign from the database.
	 *
	 * @param int $campaign_id The ID of the campaign for which to retrieve the open performance data.
	 * @return array An array of open performance data.
	 *
	 * @since 1.9.0
	 */
	public static function retrieve_campaign_click_performance( $campaign_id ) {
		$click_performance = CampaignModel::get_campaign_meta_value( $campaign_id, 'click_performance' );
		$click_performance = maybe_unserialize( $click_performance );

		if ( empty( $click_performance ) ) {
			return array();
		}

		$response = array();

		foreach ( $click_performance as $url => $data ) {
			$response[] = array(
				'url'          => $url,
				'total_clicks' => $data['count'],
				'last_clicked' => MrmCommon::date_time_format_with_core( $data['last_clicked'] ),
			);
		}

		return $response;
	}
}
