<?php

/**
 * LinkTriggerHandler class for handling the processing of link triggers in the MailMint plugin.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-08-10 11:03:17
 * @modify date 2024-08-10 11:03:17
 * @package MailMintPro\Mint\Internal\LinkTrigger
 */

namespace MailMintPro\Mint\Internal\LinkTrigger;

use MailMint\App\Helper;
use MailMintPro\Mint\DataBase\Models\LinkTriggerModel;
use Mint\MRM\DataBase\Models\ContactGroupModel;
use Mint\MRM\DataBase\Models\ContactGroupPivotModel;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\Internal\Parser\Parser;
use Mint\MRM\DataBase\Tables\AutomationLogSchema;
use Mint\MRM\DataBase\Models\EmailModel;
use MintMail\App\Internal\Automation\Action\AutomationAction;
Use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\MRM\Utilites\Helper\Campaign;
use Mint\Utilities\Arr;

/**
 * Handles the processing of link triggers in the MailMint plugin.
 *
 * The `LinkTriggerHandler` class manages the scheduling and execution of actions 
 * when a contact clicks on a tracked link. It allows for asynchronous processing 
 * of link triggers and performs various actions based on the link's configuration.
 *
 * @since 1.16.0
 */
class LinkTriggerHandler {

    /**
     * Constructs the `LinkTriggerHandler` object and hooks into WordPress actions.
     * 
     * @since 1.16.0
     */
    public function __construct(){
        add_action( 'mailmint_run_link_trigger_async', array( $this, 'link_trigger_async_cb'), 10, 3 );
    }

    /**
     * Handles the click event for a link trigger.
     *
     * @param array  $data      The data associated with the link trigger.
     * @param int    $contact_id The ID of the contact who clicked the link.
     * @param int    $email_id   The ID of the email containing the clicked link.
     * 
     * @since 1.16.0
     */
    public function handle_click( $data, $contact_id, $email_id ){
        $slug = $data['slug'] ?? '';
        


        $link_data    = LinkTriggerModel::get_by_slug( $slug );
        $redirect_url = Arr::get($link_data, 'url');

        if ( ! isset( $link_data['status'] ) || $link_data['status'] != 1 || empty( $redirect_url ) ) {
            return;
        }

        $link_id = isset( $link_data['id'] ) ? $link_data['id'] : 0;
        $run_async = apply_filters('mailmint_link_trigger_async', true, $link_id, $link_data);

        if ( true === $run_async ) {
            // Schedule link trigger action.
            $this->schedule_link_trigger_async( $link_id, $contact_id, $email_id );
        } else {
            // Direct process link trigger.
            $this->link_trigger_processing( $link_data, $contact_id);
        }


        // Update click count.
		$this->update_link_clicks( $link_id, $link_data );

        wp_redirect($redirect_url, 307);
        exit;
    }

    /**
     * Adds the contact to a specific list.
     *
     * @param int|string $value      The ID or name of the list to add the contact to.
     * @param int        $contact_id The ID of the contact to be added to the list.
     * 
     * @since 1.16.0
     */
    private function add_to_list( $value, $contact_id ){
        ContactGroupModel::set_lists_to_contact( $value, $contact_id );
    }

    /**
     * Assigns a tag to the contact.
     *
     * @param int|string $value      The ID or name of the tag to assign.
     * @param int        $contact_id The ID of the contact to be assigned the tag.
     * 
     * @since 1.16.0
     */
    private function assign_to_tag( $value, $contact_id ){
        ContactGroupModel::set_tags_to_contact($value, $contact_id);
    }

    /**
     * Removes the contact from a specific list.
     *
     * @param int|string $value      The ID or name of the list to remove the contact from.
     * @param int        $contact_id The ID of the contact to be removed from the list.
     * 
     * @since 1.16.0
     */
    private function remove_from_list( $value, $contact_id ){
        ContactGroupPivotModel::remove_groups_from_contacts( $value, array( $contact_id ) );
    }

    /**
     * Removes a tag from the contact.
     *
     * @param int|string $value      The ID or name of the tag to remove.
     * @param int        $contact_id The ID of the contact to be removed from the tag.
     * @since 1.16.0
     */
    private function remove_from_tag( $value, $contact_id ){
        ContactGroupPivotModel::remove_groups_from_contacts( $value, array( $contact_id ) );
    }


    /**
     * Permanently Delete single contact from DB.
     *
     * @param int|string $value      The ID or name of the tag to remove.
     * @param int        $contact_id The ID of the contact to be removed 
     * @since 1.17.2
     */
    private function delete_contact( $value, $contact_id ){
        ContactModel::destroy( $contact_id );
    }

    /**
     *  changes status to subscribed for a contact
     *
     * @param int|string $value      The ID or name of the tag to remove.
     * @param int        $contact_id The ID of the contact to be subscribed.
     * @since 1.17.2
     */
    private function subscribe_contact( $value, $contact_id ){
        ContactModel::update_subscription_status( $contact_id, 'subscribed' );
    
    }

    /**
     * changes status to unsubscribed for a contact
     *
     * @param int|string $value      The ID or name of the tag to remove.
     * @param int        $contact_id The ID of the contact to be unsubscribed 
     * @since 1.17.2
     */
    private function unsubscribe_contact( $value, $contact_id ){
        ContactModel::update_subscription_status( $contact_id, 'unsubscribed' );
    }

    /**
     * Trigger an automation for selected trigger.
     *
     * @param array $value array of automation id.
     * @param int   $contact_id The ID of the contact 
     * @since 1.17.2
     */
    private function trigger_automation( $value, $contact_id ){
        if (empty($value) || !is_array($value)) {
            return;
        }

        $automation_id = isset($value[0]['value']) ? $value[0]['value'] : '';
        $step_data     = HelperFunctions::get_next_step( $automation_id );

        if ( is_array( $step_data ) ) {
            $user_email = ContactMOdel::get_contact_email_by_id( $contact_id );

            if ( isset( $step_data['step_type'], $step_data['step_id'] ) && 'trigger' === $step_data['step_type'] ) {
                $identifier = uniqid();
                $log_data   = array(
                    'status'     => 'processing',
                    'email'      => $user_email,
                    'identifier' => $identifier,
                );
                HelperFunctions::update_automation_steps_status( $automation_id, $log_data );
                $payload = array(
                    'automation_id' => $automation_id,
                    'step_id'       => $step_data['step_id'],
                    'status'        => 'completed',
                    'email'         => $user_email,
                    'identifier'    => $identifier,
                );
                HelperFunctions::update_log( $payload );

                $next_step_data = HelperFunctions::get_next_step( $automation_id, $step_data['step_id'] );
                if ( $next_step_data && is_array( $next_step_data ) ) {
                    $_data = array(
                        'step_id'       => $next_step_data['step_id'],
                        'automation_id' => $automation_id,
                        'identifier'    => $identifier,
                        'data'          => array(
                            'user_email' => $user_email,
                        ),
                    );

                    $key = $next_step_data['key'];
                    AutomationAction::get_instance()->init( $key, $_data );
                }
            }
        }
    }

    /**
     * End the selected trigger automation.
     *
     * @param array $value selected trigger automation information   
     * @param int   $contact_id The ID of the contact
     * @since 1.17.2
     */
    private function end_automation($value, $contact_id) {
        if (empty($value) || !is_array($value)) {
            return;
        }

        $automation_id = isset($value[0]['value']) ? $value[0]['value'] : '';
        $user_email    = ContactMOdel::get_contact_email_by_id( $contact_id );

        HelperFunctions::update_job( $automation_id, null, 'completed' );
        global $wpdb;
        $table_name = $wpdb->prefix . AutomationLogSchema::$table_name;
        // Update the rows.
        $updated_rows = $wpdb->update(
            $table_name,
            [ 'status' => 'completed' ],
            [
                'automation_id' => $automation_id,
                'email'         => $user_email,
            ]
        );
    }


    /**
     * Schedules the asynchronous processing of a link trigger.
     *
     * This method schedules an action to process the link trigger 30 seconds
     * after the contact clicks the link.
     *
     * @param array $value    Action Information.
     * @param int $contact_id The ID of the contact who clicked the link.
     * @since 1.17.0
     */
    public function update_fields($value, $contact_id) {
        $updateValue = $value[0]['update_value'] ?? '';
        $slug = $value[0]['slug'] ?? '';
        $contact = ContactModel::get($contact_id);
        if (!empty($contact['meta_fields']) && is_array($contact['meta_fields'])) {
            $contact = array_merge($contact, $contact['meta_fields']);
            unset($contact['meta_fields']);
        }
        $parsed_returned = Parser::parse($updateValue, $contact, '', '', []);
        if (empty($parsed_returned) || $parsed_returned === 'WPFunnels') {
            return;
        }
        $primary_columns = ['email', 'first_name', 'last_name'];
        $is_meta_field = !in_array($slug, $primary_columns);
        $params = $is_meta_field ? ['meta_fields' => [$slug => $parsed_returned]]: [$slug => $parsed_returned];
        ContactModel::update($params, $contact_id);
    }
    

       /**
     * Schedules the asynchronous processing of a link trigger.
     *
     * This method schedules an action to process the link trigger 30 seconds
     * after the contact clicks the link.
     *
     * @param int $link_id    The ID of the link trigger.
     * @param int $contact_id The ID of the contact who clicked the link.
     * @param int $email_id   The ID of the email containing the clicked link.
     * 
     * @since 1.16.0
     */

    protected function schedule_link_trigger_async( $link_id, $contact_id, $email_id ){
        if ( ! $contact_id ) {
            return;
        }

        $args = array(
            'contact_id' => $contact_id,
            'link_id'    => $link_id,
            'email_id'   => $email_id,
        );

        // Schedule if not scheduled.
        $group = 'mailmint-link-trigger-' . $contact_id;
        if (! as_has_scheduled_action( 'mailmint_run_link_trigger_async', $args, $group) ) {
            as_schedule_single_action( time() + 30, 'mailmint_run_link_trigger_async', $args, $group );
        }
    }

    /**
     * Callback function for the asynchronous link trigger action.
     *
     * This method is called when the scheduled action for processing a link trigger
     * is executed. It tracks the click performance and processes the link trigger.
     *
     * @param int $contact_id The ID of the contact who clicked the link.
     * @param int $link_id    The ID of the link trigger.
     * @param int $email_id   The ID of the email containing the clicked link.
     *
     * @since 1.16.0
     */
    public function link_trigger_async_cb( $contact_id, $link_id, $email_id ){

        $link_data = LinkTriggerModel::get( $link_id );

        if ( $link_data && isset( $link_data['data'] ) ) {
			$data = unserialize($link_data['data']);
			unset($link_data['data']);
			$link_data = array_merge( $link_data, $data );
		}

        Campaign::track_email_link_click_performance($email_id, $link_data['url']);
        EmailModel::insert_or_update_email_meta('is_click', 1, $email_id);
        EmailModel::insert_or_update_email_meta('user_click_agent', Helper::get_user_agent(), $email_id);
        $is_ip_store = get_option('_mint_compliance');
        $is_ip_store = isset($is_ip_store['anonymize_ip']) ? $is_ip_store['anonymize_ip'] : 'no';
        if ('no' === $is_ip_store) {
            EmailModel::insert_or_update_email_meta('user_click_ip', Helper::get_user_ip(), $email_id);
        }

        $this->link_trigger_processing( $link_data, $contact_id );
    }

    /**
     * Processes the actions associated with a link trigger.
     *
     * This method executes the configured actions when a link trigger is clicked,
     * such as adding the contact to a list, assigning tags, or removing them from lists or tags.
     *
     * @param array $link_data  The data associated with the link trigger.
     * @param int   $contact_id The ID of the contact who clicked the link.
     * 
     * @since 1.16.0
     */
    protected function link_trigger_processing( $link_data, $contact_id ) {
		// Actions to run.
		$actions = Arr::get( $link_data, 'actions' );

		// No contact found, just redirect.
		if ( ! $contact_id ) {
			return;
		}

		// Trigger hook - link trigger clicked.
		do_action( 'mailmint_link_triggered', $link_data, $contact_id );

		// Get contact link triggered array.
        $link_id              = isset( $link_data['id'] ) ? $link_data['id'] : 0;
		$contact_link_clicked = ContactModel::get_meta_value_by_key( 'link_trigger_click', $contact_id );
		$contact_clicked_ids  = maybe_unserialize( $contact_link_clicked );
		$contact_clicked_ids  = ( ! empty( $contact_clicked_ids ) && is_array( $contact_clicked_ids ) ) ? $contact_clicked_ids : array();

		if ( empty( $actions ) ) {
			// Update Contact link click.
			$this->update_contact_link_click( $contact_id, $link_id, $contact_clicked_ids );
			return;
		}

		// Check Action runs times.
		$actions_run_time = isset( $link_data['run_multiple_time'] ) ? $link_data['run_multiple_time'] : 0;

		// Filter to run link trigger action multiple times ( by default false ).
		if ( apply_filters( 'mailmint_link_trigger_multi_execution', false ) ) {
			$actions_run_time = 1;
		}

		// Clicked multiple times, just redirect.
		if ( !$actions_run_time && in_array( $link_id, $contact_clicked_ids ) ) {
			return;
		}

		// Run actions.
		$logs = $this->process_actions( $actions, $contact_id );

		// Maybe create logs contact note.
		// $this->create_contact_logs_note( $logs, $contact, $link_data );

		// Update Contact link click.
		$this->update_contact_link_click( $contact_id, $link_id, $contact_clicked_ids );
	}

    /**
     * Executes the actions for a link trigger.
     *
     * This method iterates over the actions and calls the appropriate methods to execute them.
     *
     * @param array $actions    The list of actions to be executed.
     * @param int   $contact_id The ID of the contact for whom the actions are executed.
     * @return array The logs generated during the action processing.
     * 
     * @since 1.16.0
     */
    protected function process_actions( $actions, $contact_id ) {
		$logs = array();

		// Check for empty action.
		if ( empty( $actions ) ) {
			return;
		}

		// Iterate over action.
        foreach ( $actions as $action ) {
            $action_type  = $action['actionType'];
            $action_value = $action['actionValue'];

            if ( method_exists( $this, $action_type ) ) {
                call_user_func( [$this, $action_type], $action_value, $contact_id );
            }
            
        }

		return $logs;
	}

    /**
     * Updates the metadata for the contact to record the link click.
     *
     * @param int   $contact_id          The ID of the contact.
     * @param int   $link_id             The ID of the link trigger.
     * @param array $contact_clicked_ids An array of link IDs the contact has previously clicked.
     * 
     * @since 1.16.0
     */
    protected function update_contact_link_click( $contact_id, $link_id, $contact_clicked_ids ) {
		// Check if link trigger id already exists.
		if ( ! in_array( $link_id, $contact_clicked_ids ) ) {

			$contact_clicked_ids[] = $link_id;
			$contact_clicked_ids   = maybe_serialize( array_unique( $contact_clicked_ids ) );

			// Set updated link-trigger-click value.
            $args['meta_fields'] = array(
                'link_trigger_click' => $contact_clicked_ids,
            );

            ContactModel::update_meta_fields( $contact_id, $args );
		}
	}

    /**
     * Updates the click count for a specific link trigger.
     *
     * @param int   $link_id   The ID of the link trigger to be updated.
     * @param array $link_data The data associated with the link trigger, including the current click count.
     * 
     * @since 1.16.0
     */
    protected function update_link_clicks( $link_id, $link_data ) {
		// Update click count.
		$clicked = isset( $link_data['total_clicked'] ) ? absint( $link_data['total_clicked'] ) : 0;
		$clicked ++;

		// Link trigger data.
		$values = array(
			'total_clicked' => $clicked,
			'updated_at'    => current_time( 'mysql', 1 ),
		);

		// Update Link Trigger DB data.
		LinkTriggerModel::update_link_trigger_data( $link_id, $values );
	}
}