<?php
/**
 * Handles the downloading of lead magnet files.
 *
 * Class LeadMagnetDownloader
 *
 * @package MailMintPro\Internal\LeadMagnet
 */

namespace MailMintPro\Internal\LeadMagnet;

use MailMintPro\Mint\DataBase\Models\LeadMagnetModel;

/**
 * Handles the downloading of lead magnet files based on provided tokens and lead hashes.
 *
 * This class is designed to facilitate the secure and controlled download of files
 * associated with lead magnets in the MailMintPro system. It ensures that files can
 * only be downloaded if the correct token is provided and the token has not been used before.
 * 
 * @since 1.11.0
 */
class LeadMagnetDownloader {

    /**
     * Constructor that initializes the download handling process.
     * 
     * @param array $get The GET request array containing 'leadHash' and 'token'.
     * @since 1.11.0
     */
    public function __construct( $get ) {
        $this->handle_download( $get );
    }

    /**
     * Handles the download process by validating the token and serving the file if valid.
     *
     * This method checks the token's validity, retrieves the corresponding attachment ID,
     * and proceeds to serve the file if all conditions are met. It also marks the token as used
     * after a successful download.
     * 
     * @param array $get The GET request array containing 'leadHash' and 'token'.
     * @return void
     * @since 1.11.0
     */
    private function handle_download( $get ) {
        $lead_hash = isset( $get['leadHash'] ) ? $get['leadHash'] : '';
        $token     = isset( $get['token'] ) ? $get['token'] : '';

        $track   = LeadMagnetModel::get_tracking_info( $token );
        $is_used = isset( $track['is_used'] ) ? $track['is_used'] : false;

        if ( true ) {
            $attachment_id = LeadMagnetModel::get_attachment_id( $lead_hash );
            if ( $attachment_id ) {
                $this->save_file( $attachment_id, $token );
            }
        } else {
            wp_redirect(home_url());
            exit();
        }
    }

    /**
     * Serves the file associated with the attachment ID after validating conditions.
     * 
     * This method sets the necessary headers to force the file download and serves the file.
     * It checks that the file exists and is readable before proceeding with the download.
     *
     * @param int $attachment_id The attachment ID of the file to be downloaded.
     * @param string $token Token for the lead magnet download tracking.
     * @return void
     * @since 1.11.0
     */
    private function save_file( $attachment_id, $token ) {
        $media_url = wp_get_attachment_url( (int) $attachment_id );
        $file_path = get_attached_file((int) $attachment_id);

        if (file_exists($file_path) && is_readable($file_path)) {
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . basename($media_url) . '"');
            header('Expires: 0');
            header('Cache-Control: must-revalidate');
            header('Pragma: public');
            header('Content-Length: ' . filesize($file_path));
            readfile($file_path);
            LeadMagnetModel::mark_as_used($token);
            wp_redirect(home_url());
            exit();
        } else {
            echo "Error: File not found or not accessible.";
            exit();
        }
    }
}
