<?php

/**
 * WooCommerce merge tag parser
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-09-13 09:03:17
 * @modify date 2024-03-03 09:03:17
 * @package 
 */

namespace MailMintPro\App\Internal\EmailCustomization\Parser;

use WC_Subscription;
use WC_Memberships_User_Membership;

/**
 * Class WCMergeTagParser
 *
 * This class is responsible for parsing WooCommerce merge tags.
 *
 * @since 1.15.4
 */
class WCMergeTagParser
{

    /**
     * Value key
     *
     * @var string
     * @since 1.15.4
     */
    protected $value_key;

    /**
     * Default value
     *
     * @var string
     * @since 1.15.4
     */
    protected $default_value;

    /**
     * Params
     *
     * @var array
     * @since 1.15.4
     */
    protected $params;

    /**
     * Subscription ID
     *
     * @var int
     * @since 1.15.4
     */
    protected $subscription_id;

    /**
     * Membership ID
     *
     * @var int
     * @since 1.17.2
     */
    protected $membership_id;

    /**
     * Wishlist ID
     *
     * @var int
     * @since 1.17.2
     */
    protected $wishlist_id;

    /**
     * WC Product ID
     *
     * @var int
     * @since 1.17.2
     */
    protected $product_id;

    /**
     * Order ID
     *
     * @var int
     * @since 1.17.2
     */
    protected $order_id;

    /**
     * Constructor for the WCMergeTagParser class.
     *
     * @param string $value_key The key for the value to be parsed.
     * @param mixed $default_value The default value to return if the key is not found.
     * @param array $params Additional parameters, including 'subscription_id'.
     * 
     * @since 1.15.4
     */
    public function __construct($value_key, $default_value, $params)
    {
        $this->value_key       = $value_key;
        $this->default_value   = $default_value;
        $this->params          = $params;
        $this->subscription_id = isset($params['subscription_id']) ? $params['subscription_id'] : 0;
        $this->wishlist_id     = isset($params['wishlist_id']) ? $params['wishlist_id'] : 0;
        $this->membership_id   = isset($params['user_membership_id']) ? $params['user_membership_id'] : 0;
        $this->product_id      = isset($params['product_id']) ? $params['product_id'] : 0;
        $this->order_id        = isset($params['order_id']) ? $params['order_id'] : 0;
    }

    /**
     * Parses the WC Subscriptions merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.15.4
     */
    public function parse_wc_subscription_merge_tag(){
        $subscription =  wcs_get_subscription($this->subscription_id);

        if (! $subscription instanceof WC_Subscription) {
            return $this->default_value;
        }

        // Parse the merge tag based on the value key.
        switch ($this->value_key) {
            case 'id':
                return $this->subscription_id ? $this->subscription_id : $this->default_value;
            case 'status':
                $subscription_status_slug = 'wc-' . $subscription->get_status();
                $subscription_statuses    = wcs_get_subscription_statuses();
                $subscription_status      = $subscription_statuses[$subscription_status_slug];
                return $subscription_status ? $subscription_status : $this->default_value;
            case 'product_name':
                $items = $subscription->get_items();
                $product_names = [];
                foreach ($items as $item) {
                    $product_names[] = $item->get_name();
                }
                return !empty($product_names) ? implode(', ', $product_names) : $this->default_value;
            case 'product_price':
                $items = $subscription->get_items();
                $product_prices = [];
                foreach ($items as $item) {
                    $product_prices[] = $item->get_total();
                }
                return !empty($product_prices) ? implode(', ', $product_prices) : $this->default_value;
            case 'start_date':
                return $subscription->get_date_created() ? date_i18n(get_option('date_format'), strtotime($subscription->get_date_created())) : $this->default_value;
            case 'end_date':
                return $subscription->get_date( 'end', 'site' ) ? date_i18n(get_option('date_format'), strtotime($subscription->get_date( 'end', 'site' ))) : $this->default_value;
            case 'trial_end_date':
                return $subscription->get_date('trial_end') ? date_i18n(get_option('date_format'), strtotime($subscription->get_date('trial_end'))) : $this->default_value;
            case 'last_payment_date':
                return $subscription->get_date('last_order_date_created') ? date_i18n(get_option('date_format'), strtotime($subscription->get_date('last_order_date_created'))) : $this->default_value;
            case 'next_payment_date':
                return $subscription->get_date('next_payment', 'site') ? date_i18n(get_option('date_format'), strtotime($subscription->get_date('next_payment', 'site'))) : $this->default_value;
            case 'view_url':
                return $subscription->get_view_order_url() ? $subscription->get_view_order_url() : $this->default_value;
            case 'billing_company':
                return $subscription->get_billing_company() ? $subscription->get_billing_company() : $this->default_value;
            case 'billing_address':
                return $subscription->get_formatted_billing_address() ? $subscription->get_formatted_billing_address() : $this->default_value;
            case 'payment_method':
                return $subscription->get_payment_method_to_display() ? $subscription->get_payment_method_to_display() : $this->default_value;
            case 'total':
                return $subscription->get_total() ? wc_price($subscription->get_total()) : $this->default_value;
            case 'order_id':
                return $subscription->get_parent_id() ? $subscription->get_parent_id() : $this->default_value;
            default:
                return $this->default_value;
        }
    }

    /**
     * Parses the WC Product merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.17.2
     */
    public function parse_wc_product_merge_tag() {
        $product = wc_get_product($this->product_id);

        if (! $product) {
            return $this->default_value;
        }

        // Parse the merge tag based on the value key.
        switch ($this->value_key) {
            case 'id':
                return $this->product_id ? $this->product_id : $this->default_value;
            case 'name':
                return $product->get_name() ? $product->get_name() : $this->default_value;
            case 'regular_price':
                return $product->get_regular_price() ? wc_price($product->get_regular_price()) : $this->default_value;
            case 'sale_price':
                return $product->get_sale_price() ? wc_price($product->get_sale_price()) : $this->default_value;
            case 'description':
                if (is_callable([$product, 'get_description'])) {
                    return $product->get_description() ? $product->get_description() : $this->default_value;
                } else {
                    if( $product->is_type( [ 'variation', 'subscription_variation' ] ) ){
                        $variation_description = get_post_meta($this->product_id, '_variation_description', true);
                        return $variation_description ? $variation_description : $this->default_value;
                    }else{
                        return $product->post->post_content ? $product->post->post_content : $this->default_value;
                    }
                }
            case 'short_description':
                return $product->get_short_description() ? $product->get_short_description() : $this->default_value;
            case 'sku':
                return $product->get_sku() ? $product->get_sku() : $this->default_value;
            case 'parent_sku':
                $parent_id = $product->get_parent_id();
                if (!$parent_id) {
                    return $this->default_value;
                }
                $parent = wc_get_product($parent_id);
                return $parent->get_sku() ? $parent->get_sku() : $this->default_value;
            case 'featured_image':
                $product_image = $product->get_image('shop_catalog');
                return $product_image ? $product_image : $this->default_value;
            case 'permalink':
                return $product->get_permalink() ? $product->get_permalink() : $this->default_value;
            case 'add_to_cart_url':
                return $product->add_to_cart_url() ? $product->add_to_cart_url() : $this->default_value;
            default:
                return $this->default_value;
        }
    }

    /**
     * Parses the WC Review merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.17.2
     */
    public function parse_wc_review_merge_tag(){
        $product = wc_get_product($this->product_id);

        if (! $product) {
            return $this->default_value;
        }

        // Parse the merge tag based on the value key.
        switch ($this->value_key) {
            case 'rating':
                return $product->get_average_rating() ? $product->get_average_rating() : $this->default_value;
            case 'content':
                return $product->get_description() ? $product->get_description() : $this->default_value;
            default:
                return $this->default_value;
        }
    }

    /**
     * Parses the WC Membership merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.17.2
     */
    public function parse_wc_membership_merge_tag() {
        $membership = wc_memberships_get_user_membership($this->membership_id);

        if (!$membership instanceof WC_Memberships_User_Membership) {
            return $this->default_value;
        }

        $user = get_userdata($membership->get_user_id());
        $plan = $membership->get_plan();

        switch ($this->value_key) {
            case 'first_name':
                return $user ? $user->first_name : $this->default_value;
            case 'last_name':
                return $user ? $user->last_name : $this->default_value;
            case 'end_date':
                return $membership->get_end_date() ? date_i18n(get_option('date_format'), strtotime($membership->get_end_date())) : $this->default_value;
            case 'id':
                return $this->membership_id ?? $this->default_value;
            case 'plan_id':
                return $plan ? $plan->get_id() : $this->default_value;
            case 'plan_name':
                return $plan ? $plan->get_name() : $this->default_value;
            case 'start_date':
                return $membership->get_start_date() ? date_i18n(get_option('date_format'), strtotime($membership->get_start_date())) : $this->default_value;
            case 'status':
                $status   = $membership->get_status();
                $statuses = wc_memberships_get_user_membership_statuses();
                $status   = $statuses['wcm-'.$status];
                return isset($status['label']) ? $status['label'] : $this->default_value;
            case 'renewal_url':
                $next_bill_date = $user->get_next_bill_on_local_date(wc_date_format());
                if (! empty($next_bill_date)) {
                    $next_bill_link = $membership->get_renew_membership_url();
                }
                return $next_bill_link ?? $this->default_value;
            default:
                return $this->default_value;
        }
    }

    /**
     * Parses the WC Wishlist merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.17.2
     */
    public function parse_wc_wish_list_merge_tag(){
        // Fetch the wishlist object
        $wishlist = wc_get_wishlist($this->wishlist_id);

        if (!$wishlist) {
            return $this->default_value;
        }

        $wishlist_post = $wishlist->post;
        $items = $wishlist->get_items();


        switch ($this->value_key) {
            case 'items_count':
                return count($items) > 0 ? count($items) + 1 : $this->default_value;

            case 'items':
                $item_names = [];
                foreach ($items as $item) {
                    $product_id = $item['product_id'];
                    $product = wc_get_product($product_id);
                    if ($product) {
                        $item_names[] = $product->get_name();
                    }
                }
                return !empty($item_names) ? implode(', ', $item_names) : $this->default_value;

            case 'id':
                return $wishlist_post->ID ?? $this->default_value;

            case 'title':
                return $wishlist_post->post_title ?? $this->default_value;

            case 'view_link':
                return $wishlist_post->guid ?? $this->default_value;

            default:
                return $this->default_value;
        }
    }

    /**
     * Parses the WC Shipment Tracking merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.17.2
     */
    public function parse_wc_shipment_tracking_merge_tag(){
        $tracking_items = wc_shipment_tracking()->actions->get_tracking_items($this->order_id, true);
        
        if (count($tracking_items) < 1) {
            return $this->default_value;
        }

        if (count($tracking_items) > 1) {
            usort($tracking_items, function ($item1, $item2) {
                return absint($item1['date_shipped']) <= absint($item2['date_shipped']);
            });
        }

        switch ($this->value_key) {
            case 'tracking_number':
                return $tracking_items[0]['tracking_number'] > 0 ? $tracking_items[0]['tracking_number'] : $this->default_value;
            
            case 'formatted_tracking_provider':
                return $tracking_items[0]['formatted_tracking_provider'] > 0 ? $tracking_items[0]['formatted_tracking_provider'] : $this->default_value;
            
            case 'formatted_tracking_link':
                return $tracking_items[0]['formatted_tracking_link'] > 0 ? $tracking_items[0]['formatted_tracking_link'] : $this->default_value;

            case 'date_shipped':
                return $tracking_items[0]['formatted_date_shipped_i18n'] ? $tracking_items[0]['formatted_date_shipped_i18n'] : $this->default_value;

            default:
                return $this->default_value;
        }
    }

    /**
     * Parses the WC Advanced Shipment Tracking merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.17.2
     */
    public function parse_wc_adv_shipment_tracking_merge_tag(){
        $tracking_items = ast_get_tracking_items($this->order_id);

        if (count($tracking_items) < 1) {
            return $this->default_value;
        }

        if (count($tracking_items) > 1) {
            usort($tracking_items, function ($item1, $item2) {
                return absint($item1['date_shipped']) <= absint($item2['date_shipped']);
            });
        }

        switch ($this->value_key) {
            case 'tracking_number':
                return $tracking_items[0]['tracking_number'] > 0 ? $tracking_items[0]['tracking_number'] : $this->default_value;

            case 'formatted_tracking_provider':
                return $tracking_items[0]['formatted_tracking_provider'] > 0 ? $tracking_items[0]['formatted_tracking_provider'] : $this->default_value;

            case 'formatted_tracking_link':
                return $tracking_items[0]['formatted_tracking_link'] > 0 ? $tracking_items[0]['formatted_tracking_link'] : $this->default_value;

            case 'date_shipped':
                return $tracking_items[0]['formatted_date_shipped_i18n'] ? $tracking_items[0]['formatted_date_shipped_i18n'] : $this->default_value;

            default:
                return $this->default_value;
        }
    }
}
