<?php

/**
 * Edd merge tag parser
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-09-13 09:03:17
 * @modify date 2024-03-03 09:03:17
 * @package 
 */

namespace MailMintPro\App\Internal\EmailCustomization\Parser;

/**
 * Class EddMergeTagParser
 *
 * This class is responsible for parsing EDD merge tags.
 *
 * @since 1.15.4
 */
class LearnDashTagParser{

    /**
     * Value key
     *
     * @var string
     * @since 1.15.4
     */
    protected $value_key;

    /**
     * Default value
     *
     * @var string
     * @since 1.15.4
     */
    protected $default_value;

    /**
     * Params
     *
     * @var array
     * @since 1.15.4
     */
    protected $params;

    /**
     * quiz ID
     *
     * @var int
     * @since 1.18.1
     */
    protected $course_id;

        /**
     * Payment ID
     *
     * @var string
     * @since 1.18.1
     */
    protected $user_email;

    /**
     * Constructor for the EddMergeTagParser class.
     *
     * @param string $value_key The key for the value to be parsed.
     * @param mixed $default_value The default value to return if the key is not found.
     * @param array $params Additional parameters, including 'edd_payment_id'.
     * @param array $contact The contact array.
     * 
     * @since 1.15.4
     */
    public function __construct($value_key, $default_value, $params, $contact){
        $this->value_key     = $value_key;
        $this->default_value = $default_value;
        $this->params        = $params;
        $this->course_id     = isset( $params['ld_course_id'] ) ? $params['ld_course_id'] : 0;
        $this->user_email    = isset( $contact['email'] ) ? $contact['email'] : '';
    }

    /**
     * Parses the EDD merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.15.4
     */
    public function parse_learn_dash_merge_tag(){

        switch ($this->value_key) {
            // Replace candidate_name with user_name
            case 'candidate_name':
                $user           = get_user_by('email', $this->user_email);
                $candidate_name = (!empty($user->first_name) || !empty($user->last_name)) ? "$user->first_name $user->last_name" : $user->display_name;

                return $user ? $candidate_name : $this->default_value;

            // Replace enrolled_courses with courses
            case 'enrolled_courses':
                $user    = get_user_by( 'email', $this->user_email );
                $courses = \ld_get_mycourses( $user->ID );

                $courses_names = array();
                if ( ! empty( $courses ) && is_array( $courses ) ) {
                    foreach ( $courses as $course_id ) {
                        $courses_names[] = get_the_title( $course_id );
                    }
                }
                $courses_names = implode( ', ', $courses_names );

                return $user ? $courses_names : $this->default_value;

            case 'course_name':
                if ( empty( $this->course_id ) ) {
                    $this->course_id = ! empty( $this->params['ld_lesson_id'] ) ? get_post_meta( $this->params['ld_lesson_id'], 'course_id', true ) : 0;
                }

                if ( empty( $this->course_id ) ) {
                    $this->course_id = ! empty( $this->params['ld_topic_id'] ) ? get_post_meta( $this->params['ld_topic_id'], 'course_id', true ) : 0;
                }

                if ( empty( $this->course_id ) ) {
                    $quiz_id         = isset( $this->params['ld_quiz_id'] ) ? $this->params['ld_quiz_id'] : 0;
                    $this->course_id = ! empty($quiz_id) ? get_post_meta($quiz_id, 'course_id', true) : 0;
                }

                $course_name = ! empty( $this->course_id ) ? get_the_title( $this->course_id ) : '';
                return $course_name ? $course_name : $this->default_value;

            case 'quiz_highest_points':
                $user           = get_user_by( 'email', $this->user_email );
                $quizzes        = get_user_meta( $user->ID , '_sfwd-quizzes', true );
                $quiz_id        = isset($this->params['ld_quiz_id']) ? $this->params['ld_quiz_id'] : 0;
                $highest_points = 0;
                if ( ! empty( $quizzes ) && is_array( $quizzes ) ) {
                    foreach ( $quizzes as $quiz ) {
                        $points       = absint( $quiz['points'] );
                        $current_quiz = absint( $quiz['quiz'] );
                        $highest_points = ( ( $quiz_id === 0 || $quiz_id === $current_quiz ) && $points > $highest_points ) ? $points : $highest_points;
                    }
                }
                return $user ? $highest_points : $this->default_value;

            case 'quiz_lowest_points': 
                $user          = get_user_by( 'email', $this->user_email );
                $quizzes       = get_user_meta( $user->ID , '_sfwd-quizzes', true );
                $quiz_id       = isset($this->params['ld_quiz_id']) ? $this->params['ld_quiz_id'] : 0;
                $lowest_points = 0;
                if ( ! empty( $quizzes ) && is_array( $quizzes ) ) {
                    foreach ( $quizzes as $quiz ) {
                        $points       = absint( $quiz['points'] );
                        $current_quiz = absint( $quiz['quiz'] );
                        $lowest_points = ( 0 === $lowest_points ) ? $points : $lowest_points;
                        $lowest_points = ( ( $quiz_id  === 0 || $quiz_id  === $current_quiz ) && $points < $lowest_points ) ? $points : $lowest_points;
                    }
                }   
                return $user ? $lowest_points : $this->default_value;

            case 'quiz_name':
                $quiz_id   = isset( $this->params['ld_quiz_id'] ) ? $this->params['ld_quiz_id'] : 0;
                $quiz_name = get_the_title( $quiz_id );
                return $quiz_name ? $quiz_name : $this->default_value;
        
            case 'lesson_name':
                $lesson_id = ! empty( $this->params['ld_lesson_id'] ) ? $this->params['ld_lesson_id'] : 0;

                if ( empty( $this->params['ld_lesson_id'] ) ) {
                    $lesson_id = ! empty($this->params['ld_topic_id']) ? get_post_meta($this->params['ld_topic_id'], 'lesson_id', true) : 0;
                }

                if ( empty( $this->params['ld_lesson_id'] ) ) {
                    $topic_id  = ! empty($this->params['ld_quiz_id']) ? learndash_get_lesson_id($this->params['ld_quiz_id']) : 0;
                    $lesson_id = ! empty($topic_id) ? learndash_get_lesson_id($topic_id) : 0;
                }

                $lesson_name = ! empty( $lesson_id ) ? get_the_title( $lesson_id ) : '';
                return $lesson_name ? $lesson_name : $this->default_value;
        
            case 'quiz_percentage':
                $user       = get_user_by('email', $this->user_email);
                $quizzes    = get_user_meta($user->ID, '_sfwd-quizzes', true);
                $percentage = 0;
                if (!empty($quizzes) && is_array($quizzes)) {
                    foreach ($quizzes as $quiz) {
                        if ( $this->params['ld_quiz_id'] === $quiz['quiz'] ) {
                            $percentage = ( isset( $quiz['percentage'] ) ) ? $quiz['percentage'] : $percentage;
                        }
                    }
                }
                return $user ? $percentage : $this->default_value;
        
            case 'quiz_score':
                $user    = get_user_by('email', $this->user_email);
                $quizzes = get_user_meta($user->ID, '_sfwd-quizzes', true);
                $score   = 0;
                if ( !empty( $quizzes ) && is_array( $quizzes ) ) {
                    foreach ( $quizzes as $quiz ) {
                        if ( $this->params['ld_quiz_id'] === $quiz['quiz'] ) {
                            $score = (isset($quiz['points'])) ? $quiz['points'] : $score;
                        }
                    }
                }
                return $user ? $score : $this->default_value;
        
            case 'topic_name':
                $topic_id = isset($this->params['ld_topic_id']) ? $this->params['ld_topic_id'] : 0;

                if (empty($topic_id)) {
                    $quiz_id   = isset($this->params['ld_quiz_id']) ? $this->params['ld_quiz_id'] : 0;
                    $lesson_id = ! empty($quiz_id) ? get_post_meta($quiz_id, 'ld_lesson_id', true) : 0;
                    if (! empty($lesson_id)) {
                        if (learndash_get_post_type_slug('topic') === get_post_type($lesson_id)) {
                            $topic_id = absint($lesson_id);
                        }
                    }
                }

                $topic_name = ! empty($topic_id) ? get_the_title($topic_id) : '';
                return $topic_name ? $topic_name : $this->default_value;
        
            case 'group_name':
                $group_name = ! empty($this->params['ld_group_id']) ? get_the_title($this->params['ld_group_id']) : '';
                return $group_name ? $group_name : $this->default_value;

            case 'user_groups':
                $user   = get_user_by('email', $this->user_email);
                $groups = learndash_get_users_group_ids($user->ID);
                $names  = array();
                foreach ( $groups as $group_id ) {
                    $names[] = get_the_title($group_id);
                }
                return !empty($names) ? implode(', ', $names) : $this->default_value;
            
            case 'group_leader_emails':
                $group_leaders = learndash_get_groups_administrator_ids($this->params['ld_group_id']);
                $emails        = array();
                foreach ($group_leaders as $leader_id) {
                    $email = get_the_author_meta('user_email', $leader_id);
                    if ($email) {
                        $emails[] = $email;
                    }
                }
                return !empty($emails) ? implode(', ', $emails) : $this->default_value;
            
            case 'group_leaders':
                $group_leaders = learndash_get_groups_administrator_ids($this->params['ld_group_id']);
                $names         = array();
                foreach ($group_leaders as $leader_id) {
                    $name = get_the_author_meta('display_name', $leader_id);
                    if ($name) {
                        $names[] = $name;
                    }
                }
                return !empty($names) ? implode(', ', $names) : $this->default_value;
            
            default:
                return $this->default_value;
    
        }
    }
}
