<?php

/**
 * Fluent Booking merge tag parser
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2025-01-23 8:44:17
 * @modify date 2025-01-23 8:44:19
 * @package 
 */

namespace MailMintPro\App\Internal\EmailCustomization\Parser;

use FluentBooking\App\Services\EditorShortCodeParser;

/**
 * Class FBMergeTagParser
 *
 * This class is responsible for parsing WooCommerce merge tags.
 *
 * 1.18.4
 */
class FBMergeTagParser
{
	/**
	 * Value key
	 *
	 * @var string
	 * @since 1.18.4
	 */
	protected $value_key;
	/**
	 * Default value
	 *
	 * @var string
	 * 1.18.4
	 */
	protected $default_value;

	/**
	 * Params
	 *
	 * @var array
	 * 1.18.4
	 */
	protected $params;

	/**
	 * Subscription ID
	 *
	 * @var int
	 * 1.18.4
	 */
	protected $booking_id;


	/**
	 * Constructor for the Fluent Booking  class.
	 *
	 * @param string $value_key The key for the value to be parsed.
	 * @param mixed $default_value The default value to return if the key is not found.
	 * @param array $params Additional parameters, including 'booking_id'.
	 * 
	 * 1.18.4
	 */
	public function __construct($value_key, $default_value, $params)
	{
		$this->value_key       = $value_key;
		$this->default_value   = $default_value;
		$this->params          = $params;
		$this->booking_id      = isset($params['booking_id']) ? $params['booking_id'] : 0;
	}

	/**
	 * Parses the fluent booking merge tag based on the provided value key.
	 *
	 * @return mixed The parsed value based on the value key,
	 * or the default value if the key is not found.
	 * 
	 * @since 1.18.4
	 */
	public function parse_fluent_booking_merge_tag()
	{
		$booking = \FluentBooking\App\Models\Booking::with(['calendar_event'])->find($this->booking_id);
		if (!$booking) {
			return $this->default_value;
		}

		$calendar_event = $booking->calendar_event;
		$shortcodes = [
			'event_name'                         => '{{booking.event_name}}',
			'description'                        => $calendar_event->description ? strip_tags($calendar_event->description) : $this->default_value,
			'booking_title'                      => '{{booking.booking_title}}',
			'additional_guests'                  => '{{booking.additional_guests}}',
			'full_start_date_guest_timezone'     => '{{booking.start_date_time_for_attendee.format.Y-m-d}}',
			'full_start_date_host_timezone'      => '{{booking.start_date_time_for_host.format.Y-m-d}}',
			'full_start_end_date_guest_timezone' => '{{booking.full_start_end_guest_timezone}}',
			'full_start_end_date_host_timezone'  => '{{booking.full_start_end_host_timezone}}',
			'cancelation_url'                    => '{{booking.cancelation_url}}',
			'cancelation_reason'                 => '{{booking.cancel_reason}}',
			'reschedule_url'                     => '{{booking.reschedule_url}}',
			'admin_booking_url'                  => '{{booking.admin_booking_url}}',
			'booking_hash'                       => '{{booking.booking_hash}}',
			'reschedule_reason'                  => '{{booking.reschedule_reason}}',
		];
		if (isset($shortcodes[$this->value_key])) {
			$parsedContent = EditorShortCodeParser::parse($shortcodes[$this->value_key], $booking);
			return $parsedContent ? $parsedContent : $this->default_value;
		}

		return $this->default_value;
	}

	/**
	 * Parses the Fluent Booking guest merge tag.
	 *
	 * This method retrieves a booking using the booking ID and replaces merge tags with the corresponding booking details.
	 * If the booking is not found, it returns the default value.
	 *
	 * @return string The parsed merge tag value or the default value if the booking is not found or the merge tag is not set.
	 * 1.18.4
	 */
	public function parse_fluent_booking_guest_merge_tag()
	{
		$booking = \FluentBooking\App\Models\Booking::find($this->booking_id);
		if (!$booking) {
			return $this->default_value;
		}

		$shortcodes = [
			'first_name' => $booking->first_name ? $booking->first_name : $this->default_value,
			'last_name'  =>  $booking->last_name ? $booking->last_name : $this->default_value,
			'email'      => $booking->email ? $booking->email : $this->default_value,
			'note'       => '{{guest.note}}',
			'phone'      => $booking->phone ? $booking->phone : $this->default_value,
			'timezone'   => '{{guest.timezone}}',

		];

		if (isset($shortcodes[$this->value_key]) && !str_contains($shortcodes[$this->value_key], '{{')) {
			return $shortcodes[$this->value_key];
		}

		if (isset($shortcodes[$this->value_key])) {
			$parsedContent = EditorShortCodeParser::parse($shortcodes[$this->value_key], $booking);
			return $parsedContent ? $parsedContent : $this->default_value;
		}
		return $this->default_value;
	}

	/**
	 * Parses the Fluent Booking event merge tag and returns the corresponding value.
	 *
	 * This method retrieves the booking information along with its associated calendar event.
	 * It then constructs an array of shortcodes with their respective values. If the requested
	 * value key exists in the shortcodes array and does not contain a placeholder, it returns
	 * the value directly. Otherwise, it parses the content using the EditorShortCodeParser.
	 *
	 * @return string The parsed value of the merge tag or the default value if not found.
	 * 1.18.4
	 */
	public function parse_fluent_booking_event_merge_tag()
	{
		$booking = \FluentBooking\App\Models\Booking::with(['calendar_event'])->find($this->booking_id);
		if (!$booking) {
			return $this->default_value;
		}

		$calendar_event = $booking->calendar_event;
		$shortcodes = [
			'event_id'                     => $booking->event_id ? $booking->event_id : $this->default_value,
			'calendar_id'                  =>  $booking->calendar_id ? $booking->calendar_id : $this->default_value,
			'event_title'                  => '{{booking.event_name}}',
			'calendar_title'               => $calendar_event->title ? $calendar_event->title  : $this->default_value,
			'calendar_description'         => $calendar_event->description ? strip_tags($calendar_event->description) : $this->default_value,
			'start_date_time'              => '{{booking.start_date_time}}',
			'start_date_time_for_attendee' => '{{booking.start_date_time_for_attendee}}',
			'start_date_time_for_host'     => '{{booking.start_date_time_for_host}}',
			'location_details_text'        => '{{booking.location_details_text}}',

		];

		if (isset($shortcodes[$this->value_key]) && !str_contains($shortcodes[$this->value_key], '{{')) {
			return $shortcodes[$this->value_key];
		}
		if (isset($shortcodes[$this->value_key])) {
			$parsedContent = EditorShortCodeParser::parse($shortcodes[$this->value_key], $booking);
			return $parsedContent ? $parsedContent : $this->default_value;
		}
		return $this->default_value;
	}

	/**
	 * Parses the fluent booking host merge tag.
	 *
	 * This function retrieves the booking information using the booking ID and fetches the associated calendar event.
	 *
	 * @return string The parsed content or the default value if the booking or user data is not found.
	 * 1.18.4
	 */
	public function parse_fluent_booking_host_merge_tag()
	{
		$booking = \FluentBooking\App\Models\Booking::with(['calendar_event'])->find($this->booking_id);
		if (!$booking) {
			return $this->default_value;
		}
		$user = get_userdata($booking->host_user_id);
		$shortcodes = [
			'name'     => $user->display_name ? $user->display_name : $this->default_value,
			'email'    =>  $user->user_email ? $user->user_email : $this->default_value,
			'timezone' => '{{host.timezone}}'
		];

		if (isset($shortcodes[$this->value_key]) && !str_contains($shortcodes[$this->value_key], '{{')) {
			return $shortcodes[$this->value_key];
		}
		if (isset($shortcodes[$this->value_key])) {
			$parsedContent = EditorShortCodeParser::parse($shortcodes[$this->value_key], $booking);
			return $parsedContent ? $parsedContent : $this->default_value;
		}

		return $this->default_value;
	}
}
