<?php

/**
 * Edd merge tag parser
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-09-13 09:03:17
 * @modify date 2024-03-03 09:03:17
 * @package 
 */

namespace MailMintPro\App\Internal\EmailCustomization\Parser;

/**
 * Class EddMergeTagParser
 *
 * This class is responsible for parsing EDD merge tags.
 *
 * @since 1.15.4
 */
class EddMergeTagParser{

    /**
     * Value key
     *
     * @var string
     * @since 1.15.4
     */
    protected $value_key;

    /**
     * Default value
     *
     * @var string
     * @since 1.15.4
     */
    protected $default_value;

    /**
     * Params
     *
     * @var array
     * @since 1.15.4
     */
    protected $params;

    /**
     * Payment ID
     *
     * @var int
     * @since 1.15.4
     */
    protected $payment_id;

    /**
     * Constructor for the EddMergeTagParser class.
     *
     * @param string $value_key The key for the value to be parsed.
     * @param mixed $default_value The default value to return if the key is not found.
     * @param array $params Additional parameters, including 'edd_payment_id'.
     * 
     * @since 1.15.4
     */
    public function __construct($value_key, $default_value, $params){
        $this->value_key = $value_key;
        $this->default_value = $default_value;
        $this->params = $params;
        $this->payment_id = isset($params['edd_payment_id']) ? $params['edd_payment_id'] : 0;
    }

    /**
     * Parses the EDD merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.15.4
     */
    public function parse_edd_merge_tag(){
        $payment = edd_get_payment($this->payment_id);

        if (!is_object($payment) || !isset($payment->ID)) {
            return $this->default_value;
        }

        // Parse the merge tag based on the value key.
        switch ($this->value_key) {
            case 'order_number':
                return isset($payment->ID) ? $payment->ID : $this->default_value;
            case 'order_status':
                return isset($payment->status_nicename) ? $payment->status_nicename : $this->default_value;
            case 'order_currency':
                return isset($payment->currency) ? edd_currency_symbol($payment->currency) : $this->default_value;
            case 'order_total_amount':
                return isset($payment->total) ? edd_format_amount($payment->total) : $this->default_value;
            case 'order_payment_method':
                return isset($payment->gateway) ? $payment->gateway : $this->default_value;
            case 'order_date':
                return isset($payment->date) ? date_i18n(get_option('date_format'), strtotime($payment->date)) : $this->default_value;
            case 'order_items_count':
                return isset($payment->cart_details) ? count($payment->cart_details) : $this->default_value;
            case 'order_items_table':
                return $this->generate_items_table($payment, $this->default_value);
            case 'order_download_lists':
                return $this->get_download_list($payment, $this->default_value);
            case 'order_address':
                return isset($payment->address) ? implode(' ', $payment->address) : $this->default_value;
            default:
                return $this->default_value;
        }
    }

    /**
     * Parses the EDD Customer merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.15.4
     */
    public function parse_edd_customer_merge_tag(){
        $payment_meta  = edd_get_payment_meta($this->payment_id);
        $payment_email = isset($payment_meta['email']) ? $payment_meta['email'] : '';
        $customer      = new \EDD_Customer($payment_email);

        if (!is_array($payment_meta) || empty($payment_meta)) {
            return $this->default_value;
        }

        // Parse the merge tag based on the value key.
        switch ($this->value_key) {
            case 'first_name':
                return isset($payment_meta['user_info']['first_name']) ? $payment_meta['user_info']['first_name'] : $this->default_value;
            case 'last_name':
                return isset($payment_meta['user_info']['last_name']) ? $payment_meta['user_info']['last_name'] : $this->default_value;
            case 'email':
                return isset($customer->email) ? $customer->email : $this->default_value;
            case 'total_order_count':
                return isset($customer->purchase_count) ? $customer->purchase_count : $this->default_value;
            case 'total_spent':
                return isset($customer->purchase_value) ? edd_format_amount($customer->purchase_value) : $this->default_value;
            case 'first_order_date':
                $orders = $customer->get_orders();
                $first_payment = reset( $orders );
                return isset($first_payment->date_created) ? date_i18n(get_option('date_format'), strtotime($first_payment->date_created)) : $this->default_value;
            case 'last_order_date':
                $orders = $customer->get_orders();
                $last_payment = end( $orders );
                return isset($last_payment->date_created) ? date_i18n(get_option('date_format'), strtotime($last_payment->date_created)) : $this->default_value;
            default:
                return $this->default_value;
        }
    }

    /**
     * Parses the EDD Billing merge tag based on the provided value key.
     *
     * @return mixed The parsed value based on the value key,
     * or the default value if the key is not found.
     * 
     * @since 1.15.4
     */
    public function parse_edd_billing_merge_tag(){
        $payment_meta = edd_get_payment_meta($this->payment_id);

        if (!isset($payment_meta['user_info']['address']) || empty($payment_meta['user_info']['address'])) {
            return $this->default_value;
        }

        $address = $payment_meta['user_info']['address'];
        // Parse the merge tag based on the value key.
        switch ($this->value_key) {
            case 'address_line_1':
                return isset($address['line1']) ? $address['line1'] : $this->default_value;
            case 'address_line_2':
                return isset($address['line2']) ? $address['line2'] : $this->default_value;
            case 'city':
                return isset($address['city']) ? $address['city'] : $this->default_value;
            case 'postal':
                return isset($address['state']) ? $address['state'] : $this->default_value;
            case 'country':
                return isset($address['country']) ? $address['country'] : $this->default_value;
            case 'region':
                return isset($address['zip']) ? $address['zip'] : $this->default_value;
            default:
                return $this->default_value;
        }
    }

    /**
     * Generates the items table for the order.
     *
     * @param object $payment The payment object.
     * @param string $default The default value to return if the items table cannot be generated.
     * 
     * @return string The generated items table.
     * 
     * @since 1.15.4
     */
    private function generate_items_table($payment, $default = ''){
        $text_align  = is_rtl() ? 'right' : 'left';
        $order_items = edd_get_payment_meta_cart_details($payment->ID, true);

        if (!$order_items) {
            return $default;
        }

        ob_start(); ?>

        <div>
            <table class="td" cellspacing="0" cellpadding="6"
                style="color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;width: 100%;font-family: 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif" border="1">
                <thead>
                    <tr>
                        <th class="td" scope="col"
                            style="text-align:
							<?php
                            echo esc_attr($text_align);
                            ?>
							; color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;padding: 5px; min-width: 55px">
                            <?php
                            esc_html_e('Product', 'mailmint-pro');
                            ?>
                        </th>
                        <th class="td" scope="col"
                            style="text-align:
							<?php
                            echo esc_attr($text_align);
                            ?>
							; color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;padding: 5px; min-width: 85px">
                            <?php
                            esc_html_e('Quantity', 'mailmint-pro');
                            ?>
                        </th>
                        <th class="td" scope="col"
                            style="text-align:
							<?php
                            echo esc_attr($text_align);
                            ?>
							; color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;padding: 5px; min-width: 85px">
                            <?php
                            esc_html_e('Price', 'mailmint-pro');
                            ?>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    <?php

                    if (!is_array($order_items) || empty($order_items)) {
                        ob_end_clean();
                        return;
                    }

                    foreach ($order_items as $order_item) {
                        $item_id   = !empty($order_item['id']) ? (int) $order_item['id'] : 0;
                        $image_id  = get_post_thumbnail_id($item_id);
                        $image_url = !empty($image_id) ? wp_get_attachment_image_src($image_id, 'full')[0] : MRM_DIR_URL . 'admin/assets/images/mint-placeholder.png';
                    ?>
                        <tr class='cart_item'>
                            <td class="td"
                                style="text-align:
									<?php
                                    echo esc_attr($text_align);
                                    ?>
									; color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;padding: 5px; font-family: 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif;">
                                <img alt="<?php echo esc_html($order_item['name']); // phpcs:ignore 
                                            ?>" style="width: 60px;height: 60px;border: none;font-size: 14px;font-weight: bold;text-decoration: none;text-transform: capitalize;vertical-align: super;margin-right: 10px;max-width: 100%;"
                                    src="<?php echo $image_url; // phpcs:ignore 
                                            ?>">
                                <span style="display: inline-block; padding-top: 5px; width: 70%">
                                    <?php
                                    echo esc_html($order_item['name']);
                                    ?>
                                </span>
                            </td>
                            <td class="td"
                                style="text-align:
									<?php
                                    echo esc_attr($text_align);
                                    ?>
									; color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;padding: 5px; font-family: 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif;">
                                <?php
                                echo esc_html($order_item['quantity']);
                                ?>
                            </td>
                            <td class="td"
                                style="text-align:
									<?php
                                    echo esc_attr($text_align);
                                    ?>
									; color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;padding: 5px; font-family: 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif;">
                                <?php
                                echo edd_currency_filter(edd_format_amount($order_item['quantity'] * $order_item['item_price'])); // phpcs:ignore
                                ?>
                            </td>
                        </tr>
                    <?php
                    }


                    ?>
                </tbody>
                <tfoot>
                    <tr>
                        <th class="td" scope="row" colspan="2"
                            style="text-align:
						<?php
                        echo esc_attr($text_align);
                        ?>
						; color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;padding: 12px;">
                            <?php esc_html_e('Total:', 'mailmint-pro'); ?>
                        </th>
                        <td class="td"
                            style="text-align:
						<?php
                        echo esc_attr($text_align);
                        ?>
						; color: #636363;border: 1px solid #e5e5e5;vertical-align: middle;padding: 12px;">
                            <?php
                            echo edd_currency_filter(edd_format_amount($payment->total)); // phpcs:ignore
                            ?>
                        </td>
                    </tr>
                </tfoot>
            </table>
        </div>

        <?php
        $output = ob_get_contents();
        ob_end_clean();
        return $output;
    }

    /**
     * Get download list
     *
     * @param object $payment The payment object.
     * @param string $default The default value to return if the download list cannot be generated.
     * 
     * @return string The generated download list.
     * 
     * @since 1.15.4
     */
    private function get_download_list($payment, $default = ''){
        $order_items = edd_get_payment_meta_cart_details($payment->ID, true);

        if (!$order_items) {
            return $default;
        }

        $meta  = edd_get_payment_meta($payment->ID);
        $email = edd_get_payment_user_email($payment->ID);

        ob_start();

        echo '<ul>';
        foreach ($order_items as $key => $item) :
            if (!empty($item['in_bundle'])) {
                continue;
            }
            $price_id = edd_get_cart_item_price_id($item);
            $download_files = edd_get_download_files($item['id'], $price_id);
            if (edd_is_payment_complete($payment->ID) && !empty($download_files) && is_array($download_files)) :
                foreach ($download_files as $filekey => $file) :
        ?>
                    <li class="edd_download_file">
                        <a href="<?php echo esc_url(edd_get_download_file_url($meta['key'], $email, $filekey, $item['id'], $price_id)); ?>"
                            class="edd_download_file_link"><?php echo edd_get_file_name($file); ?></a>
                    </li>
        <?php endforeach;
            endif;
        endforeach;
        echo '</ul>';
        return ob_get_clean();
    }
}
