<?php
/**
 * WPFunnels triggers
 *
 * @package MintMail\App\Internal\Automation\Connector\trigger;
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use WPFunnels\Wpfnl_functions;
use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;
/**
 * WPFunnels triggers
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class WpfunnelsTriggers {

	use Singleton;



	/**
	 * Connector name
	 *
	 * @var $connector_name
	 */
	public $connector_name = 'Wpfunnels';

	/**
	 * Initialization of WPFunnels hooks
	 */
	public function init() {
		add_action( 'wpfunnels/after_optin_submit', array( $this, 'mrm_wpf_optin_submit' ), 10, 4 );
		add_action( 'wpfunnels/offer_accepted', array( $this, 'mrm_wpf_offer_accepted' ), 10, 2 );
		add_action( 'wpfunnels/offer_rejected', array( $this, 'mrm_wpf_offer_rejected' ), 10, 2 );
		add_action( 'wpfunnels/funnel_order_placed', array( $this, 'mrm_wpf_order_placed' ), 10, 3 );
		add_action( 'wpfunnels/trigger_cta', array( $this, 'mrm_wpf_cta_trigger' ), 10, 2 );
	}


	/**
	 * Validate trigger settings
	 *
	 * @param array $step_data Get Step data.
	 * @param array $data Get All data.
	 * @return bool
	 */
	public function validate_settings( $step_data, $data ) {
		$data = $data['data'];
		if ( isset( $step_data['automation_id'], $step_data['step_id'] ) ) {
			$settings = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			if ( isset( $data['funnel_id'], $data['step_id'], $settings['settings'], $settings['settings']['selectedStep'], $settings['settings']['selected_funnel_ids'] ) ) {
				if ( is_array( $settings['settings']['selected_funnel_ids'] ) && in_array( $data['funnel_id'], $settings['settings']['selected_funnel_ids'] ) ) { //phpcs:ignore
					$parent_step_id = get_post_meta( $data['step_id'], '_parent_step_id', true );
					$step_id        = $parent_step_id ? $parent_step_id : $data['step_id'];
					if ( $step_id == $settings['settings']['selectedStep'] ) { //phpcs:ignore
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * WPF after optin submit
	 *
	 * @param string $step_id Get Step id.
	 * @param object $post_action Get Request Action.
	 * @param object $action_type Get Request Action Type.
	 * @param object $record Get Record.
	 */
	public function mrm_wpf_optin_submit( $step_id, $post_action, $action_type, $record ) {
		$funnel_id = Wpfnl_functions::get_funnel_id_from_step( $step_id );
		if ( $funnel_id ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wpf_optin_submit',
				'data'           => array(
					'user_email'   => isset( $record->form_data['email'] ) ? $record->form_data['email'] : '',
					'first_name'   => isset( $record->form_data['first_name'] ) ? $record->form_data['first_name'] : '',
					'last_name'    => isset( $record->form_data['last_name'] ) ? $record->form_data['last_name'] : '',
					'phone_number' => isset( $record->form_data['phone'] ) ? $record->form_data['phone'] : '',
					'funnel_id'    => $funnel_id,
					'step_id'      => $step_id,
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * This will trigger after user accept the offer
	 *
	 * @param object $order Get Order.
	 * @param object $offer_product Get Offer Product.
	 */
	public function mrm_wpf_offer_accepted( $order, $offer_product ) {
		if ( HelperFunctions::is_wc_active() ) {
			$step_id   = isset( $offer_product['step_id'] ) ? $offer_product['step_id'] : '';
			$funnel_id = get_post_meta( $step_id, '_funnel_id', true );
			$step_type = get_post_meta( $step_id, '_step_type', true );
			$data      = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wpf_' . $step_type . '_accepted',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
					'funnel_id'  => $funnel_id,
					'step_id'    => $step_id,
					'action'   	 => 'accepted',
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
			$data['trigger_name'] = 'wpf_' . $step_type . '_trigger';
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * This will trigger after user reject the offer
	 *
	 * @param object $order Get Order.
	 * @param object $offer_product Get Offer Product.
	 */
	public function mrm_wpf_offer_rejected( $order, $offer_product ) {
		if ( HelperFunctions::is_wc_active() ) {
			$step_id   = isset( $offer_product['step_id'] ) ? $offer_product['step_id'] : '';
			$funnel_id = get_post_meta( $step_id, '_funnel_id', true );
			$step_type = get_post_meta( $step_id, '_step_type', true );
			$data      = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wpf_' . $step_type . '_rejected',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
					'funnel_id'  => $funnel_id,
					'step_id'    => $step_id,
					'action'   	 => 'rejected',
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
			$data['trigger_name'] = 'wpf_' . $step_type . '_trigger';
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * This will trigger after user reject the offer.
	 *
	 * @param object $order_id Get Order ID.
	 * @param object $funnel_id Get Funnel ID.
	 * @param object $step_id Get Funnel Step ID.
	 */
	public function mrm_wpf_order_placed( $order_id, $funnel_id, $step_id ) {
		if ( HelperFunctions::is_wc_active() ) {
			$order = new \WC_Order( $order_id );
			if ( $order ) {
				$order_bump_product = get_post_meta( $order_id, '_wpfunnels_order_bump_product', true );
				$data               = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'wpf_order_placed',
					'data'           => array(
						'user_email' => $order->get_billing_email(),
						'first_name' => $order->get_billing_first_name(),
						'last_name'  => $order->get_billing_last_name(),
						'funnel_id'  => $funnel_id,
						'step_id'    => $step_id,
					),
				);
				do_action( MINT_TRIGGER_AUTOMATION, $data );

				$items        = $order->get_items();
				$is_ob_accept = false;
				foreach ( $order->get_items() as $item_id => $item ) {
					$wpfunnels_order_bump = wc_get_order_item_meta( $item_id, '_wpfunnels_order_bump', true );
					if ( 'yes' === $wpfunnels_order_bump ) {
						$is_ob_accept = true;
					}
				}

				if ( $is_ob_accept ) {
					$data['trigger_name'] = 'wpf_orderbump_accepted';
					$data['data']['action']   	  = 'accepted';
					do_action( MINT_TRIGGER_AUTOMATION, $data );
				} else {
					$data['trigger_name'] = 'wpf_orderbump_rejected';
					$data['data']['action']   	  = 'rejected';
					do_action( MINT_TRIGGER_AUTOMATION, $data );
				}
				$data['trigger_name'] = 'wpf_orderbump_action';
				do_action( MINT_TRIGGER_AUTOMATION, $data );
			}
		}
	}


	/**
	 * WPFunnels CTA button trigger event.
	 *
	 * @param int $step_id Funnel step ID.
	 * @param int $funnel_id Funnel ID.
	 *
	 * @return void
	 *
	 * @since 1.0.0
	 */
	public function mrm_wpf_cta_trigger( $step_id, $funnel_id ) {
		if ( is_user_logged_in() ) {
			$user = wp_get_current_user();
			if ( $user && isset( $user->data->user_email ) ) {
				$data = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'wpf_cta_trigger',
					'data'           => array(
						'user_email' => $user->data->user_email,
						'first_name' => isset( $user->first_name ) ? $user->first_name : '',
						'last_name'  => isset( $user->last_name ) ? $user->last_name : '',
						'funnel_id'  => $funnel_id,
						'step_id'    => $step_id,
					),
				);
				do_action( MINT_TRIGGER_AUTOMATION, $data );
			}
		}
	}
}