<?php
/**
 * Woocommerce triggers
 *
 * @package MintMail\App\Internal\Automation\Connector\trigger
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use MailMintPro\Mint\Internal\AbandonedCart\Helper\Model;
use Mint\App\Internal\Cron\BackgroundProcessHelper;
use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;
use Mint\MRM\DataBase\Tables\AutomationSchema;
use Mint\MRM\DataBase\Tables\AutomationStepSchema;
use Mint\MRM\DataBase\Tables\ContactSchema;
use MRM\Common\MrmCommon;

/**
 * Woocommerce triggers
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class WoocommerceTriggers {

	use Singleton;


	/**
	 * Connector name
	 *
	 * @var $connector_name
	 */
	public $connector_name = 'Woocommerce';

	/**
	 * Automation ID for the currently running automation.
	 *
	 * @var int Holds the automation ID.
	 * @since 1.13.0
	 */
	private $automation_id;

	/**
	 * Initialization of Woocommerce hooks
	 */
	public function init() {
		add_action( 'woocommerce_thankyou', array( $this, 'mint_process_first_order_in_woocommerce_store' ), 10, 1 );
		add_action( 'woocommerce_new_order', array( $this, 'mrm_woocommerce_new_order' ), 10, 2 );
		add_action( 'woocommerce_new_order', array( $this, 'mrm_woocommerce_new_all_order' ), 10, 2 );
		add_action( 'woocommerce_order_status_completed', array( $this, 'mrm_order_status_completed' ), 10, 1 );
		add_action( 'woocommerce_order_status_changed', array( $this, 'mrm_order_status_changed' ), 10, 3 );
		add_action( 'woocommerce_order_status_failed', array( $this, 'mrm_order_status_failed' ), 10, 3 );
		add_action( 'woocommerce_order_note_added', array( $this, 'mrm_order_note_added' ), 10, 3 );
		add_action( 'woocommerce_order_refunded', array( $this, 'mrm_order_refunded' ), 10, 2 );
		add_action( 'mailmint_after_cart_abandoned', array( $this, 'mrm_mailmint_after_cart_abandoned' ), 10, 1 );
		add_action( 'mailmint_after_abandoned_cart_lost', array( $this, 'after_abandoned_cart_lost' ), 10, 1 );
		add_action( 'mailmint_after_abandoned_cart_recovered', array( $this, 'after_abandoned_cart_recovered' ), 10, 1 );
		add_action( 'mailmint_manually_run_abandoned_data', array( $this, 'manually_run_abandoned_data' ), 10, 2 );
		add_action( 'woocommerce_product_object_updated_props', array( $this, 'mint_woocommerce_product_price_drop' ), 10, 2 );
		add_action( 'comment_post', array( $this, 'mint_after_woocommerce_review_received' ), 10, 2 );
		add_action( 'transition_comment_status', array( $this, 'mint_after_woocommerce_review_approved' ), 20, 3 );
		add_action( 'mailmint_process_price_dropped_scheduler', array($this, 'mint_price_dropped_trigger'), 10, 5);
	}

	/**
	 * Validate trigger settings.
	 *
	 * @param array $step_data get Step data.
	 * @param array $data  all data form this trigger.
	 * @return bool
	 */
	public function validate_settings( $step_data, $data ) {
		if ( isset( $data['trigger_name'] ) && 'wc_order_status_changed' === $data['trigger_name'] ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}
			if ( isset( $step_data['settings']['status_settings']['status'] ) ) {
				if ( isset( $data['data']['new_status'] ) ) {
					return $data['data']['new_status'] === $step_data['settings']['status_settings']['status'];
				}
			}
		}

		$allowed_trigger_names = array( 'wc_price_dropped', 'wc_review_received' );
		if ( isset( $data['trigger_name'] ) && in_array( $data['trigger_name'], $allowed_trigger_names, true ) ) {
			$product_id = $data['data']['product_id'];
			if ( $product_id ) {
				if ( isset( $step_data['automation_id'], $step_data['step_id'] ) ) {
					$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
				}
				$option_type = !empty( $step_data['settings']['product_settings']['option_type'] ) ? $step_data['settings']['product_settings']['option_type'] : 'choose-all';
				return $this->is_product_included( $product_id, $option_type, $step_data['settings']['product_settings']['category'], $step_data['settings']['product_settings']['products'] );
			}
		}

		$allowed_trigger_names = array( 'wc_order_created', 'wc_order_completed' );
		if ( isset( $data['trigger_name'], $data['data']['order'] ) && in_array( $data['trigger_name'], $allowed_trigger_names, true ) && !empty( $data['data']['order'] ) ) {
			$order = $data['data']['order'];
			if ( $order ) {
				if ( isset( $step_data['automation_id'], $step_data['step_id'] ) ) {
					$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
				}
				$option_type = !empty( $step_data['settings']['product_settings']['option_type'] ) ? $step_data['settings']['product_settings']['option_type'] : 'choose-all';

				if ( !isset( $step_data['settings']['product_settings']['option_type'] ) && !empty( $step_data['settings']['product_settings']['products'] ) ) {
					return self::check_product_criteria( $step_data, $order );
				}
				if ( 'choose-product' === $option_type ) {
					return self::check_product_criteria( $step_data, $order );
				}
				if ( 'choose-category' === $option_type ) {
					return self::check_category_criteria( $step_data, $order );
				}
				return true;
			}
		}
		return true;
	}


	/**
	 * Process the first order in the WooCommerce store.
	 *
	 * @param object $order Get the Order Info.
	 *
	 * @return void
	 * @since 1.0.0
	 * @since 1.13.0 Added support for guest users.
	 */
	public function mint_process_first_order_in_woocommerce_store( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}
		// Get the customer ID and billing email.
		$customer_id = $order->get_user_id();
		$billing_email = $order->get_billing_email();
		
		// Check if this is the customer's first order.
		$args = array(
			'limit'    => -1,
			'return'   => 'ids',
		);

	
		if ( $customer_id ) {
			// For registered users, check by customer ID.
			$args['customer_id'] = $customer_id;
		} else {
			// For guest users, check by billing email.
			$args['billing_email'] = $billing_email;
		}
	

		$customer_orders = wc_get_orders( $args );
		
		// For registered users, trigger the automation only if it's their first order.
		// For guest users, always trigger the automation.
		if ( count( $customer_orders ) === 1 ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_first_order',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
					'order_id'   => $order_id,
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * WC new completed order created
	 *
	 * @param string $order_id Get Woocommerce Order ID.
	 */
	public function mrm_order_status_completed( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( $order ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_order_completed',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
					'order'      => $order,
					'order_id'   => $order_id,
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * WC new completed order created.
	 *
	 * @param string $order_id Get Woocommerce Order ID.
	 * @param object $order Get Woocommerce Order ID.
	 */
	public function mrm_woocommerce_new_order( $order_id, $order ) {
		if ( $order ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_order_created',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
					'order'      => $order,
					'order_id'   => $order_id,
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * WC new completed order created
	 *
	 * @param string $order_id Get Woocommerce Order ID.
	 * @param object $order Get Woocommerce Order ID.
	 */
	public function mrm_woocommerce_new_all_order( $order_id, $order ) {
		if ( $order ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_all_order_created',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
					'order'      => $order,
					'order_id'   => $order_id,
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Change wc order status
	 *
	 * @param string $order_id Get Woocommerce Order ID.
	 * @param string $old_status Get Woocommerce Order Status.
	 * @param string $new_status Get Set Woocommerce Order New status.
	 */
	public function mrm_order_status_changed( $order_id, $old_status, $new_status ) {
		$order = new \WC_Order( $order_id );
		if ( $order ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_order_status_changed',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
					'new_status' => $new_status,
					'order_id'   => $order_id,
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Failed wc order status
	 *
	 * @param string $order_id Get Woocommerce Order ID.
	 * @param object $order Get Woocommerce Order Info.
	 */
	public function mrm_order_status_failed( $order_id, $order ) {
		if ( $order ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_order_failed',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
					'order_id'   => $order_id,
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Added order note
	 *
	 * @param string $order_note_id Get Woocommerce Order Note ID.
	 * @param object $order Get Woocommerce Order.
	 */
	public function mrm_order_note_added( $order_note_id, $order ) {
		if ( $order ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_order_note_added',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Handle actions after a WooCommerce product is updated.
	 *
	 * This function is hooked into the WooCommerce 'woocommerce_update_product' action
	 * and checks if the product is on sale. If it is, it triggers an automation event
	 * indicating that the price has dropped.
	 *
	 * @param int    $product_id The ID of the updated product.
	 * @param object $product    The updated product object.
	 *
	 * @since 1.5.14
	 */
	public function mint_woocommerce_product_price_drop( $product, $updated_props ) {
		$product_id          = $product->get_id();
		$updating_product_id = 'update_product_' . $product_id;

		if ( false === ( get_transient( $updating_product_id ) ) ) {
			if ( $product->is_on_sale() && in_array( 'sale_price', $updated_props, true ) ) {
				// Get all the triggers for the price dropped event.
				global $wpdb;
				$step_table       = $wpdb->prefix . AutomationStepSchema::$table_name;
				$automation_table = $wpdb->prefix . AutomationSchema::$table_name;

				$steps = $wpdb->get_results($wpdb->prepare(
					"SELECT a.id AS automation_id, a.status, s.id AS step_id, s.step_id AS step_identifier, s.key, s.type, s.settings
					FROM 
						$automation_table a
					JOIN 
						$step_table s
					ON 
						a.id = s.automation_id
					WHERE 
						s.type = %s 
						AND s.key = %s
						AND a.status = %s",
							'trigger',
							'wc_price_dropped',
							'active'
						), ARRAY_A);  // phpcs:ignore.

				if (is_array($steps) && !empty($steps)) {
					foreach ($steps as $step) {
						$automation_id = isset($step['automation_id']) ? $step['automation_id'] : 0;
						$step_id       = isset($step['step_identifier']) ? $step['step_identifier'] : '';

						// Prepare arguments for the recurring action callback.
						$args = array(
							'automation_id' => $automation_id,
							'step_id'       => $step_id,
							'offset'        => 0,
							'per_page'      => 20,
							'product_id'    => $product_id,
						);

						$group = 'mailmint-process-price-dropped-' . $automation_id;
						if (as_has_scheduled_action('mailmint_process_price_dropped_scheduler', $args, $group)) {
							as_unschedule_all_actions('mailmint_process_price_dropped_scheduler', $args, $group);
						}

						/**
						 * Action: mailmint_process_price_dropped_scheduler
						 * 
						 * Summary: Fires when a product price is dropped.
						 * 
						 * Description: This action is used to process the triggers for the 'wc_price_dropped' event.
						 * 
						 * @param array $args An array containing information about the automation and step.
						 * @since 1.13.0
						 */
						as_schedule_single_action(time() + 30, 'mailmint_process_price_dropped_scheduler', $args, $group);
					}
					set_transient($updating_product_id, $product_id, 2);
				}
			}
		}
	}

	/**
	 * Trigger for handling price drop automation.
	 *
	 * This function processes a batch of contacts for a given automation step when a product price drops.
	 * It schedules the next batch if there are more contacts to process.
	 *
	 * @param int $automation_id The ID of the automation.
	 * @param int $step_id The ID of the step within the automation.
	 * @param int $offset The offset for the batch of contacts to process.
	 * @param int $per_batch The number of contacts to process per batch.
	 * @param int $product_id The ID of the product whose price has dropped.
	 *
	 * @return bool True if there are more contacts to process, false otherwise.
	 * @since 1.5.14
	 */
	public function mint_price_dropped_trigger($automation_id, $step_id, $offset, $per_batch, $product_id){
		$this->automation_id = $automation_id;

		// Get the step data.
		$step_data = HelperFunctions::get_step_data($automation_id, $step_id);
		$settings  = isset($step_data['settings']['product_settings']) ? $step_data['settings']['product_settings'] : array();
		$contacts  = $this->get_contacts($settings, $offset, $per_batch);

		if (!$contacts) {
			return false;
		}

		$start_time = time();
		$has_more   = true;
		$run        = true;
		if ($contacts && $run) {
			foreach ($contacts as $contact) {
				$email = isset($contact['email']) ? $contact['email'] : '';
				$data  = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'wc_price_dropped',
					'data'           => array(
						'user_email' => $email,
						'contact'    => $contact,
						'product_id' => $product_id,
					),
				);

				do_action(MINT_TRIGGER_AUTOMATION, $data);
			}

			if (BackgroundProcessHelper::memory_exceeded() || time() - $start_time > 40) {
				$run      = false;
				$has_more = true;
			} else {
				$contacts = $this->get_contacts($settings, $offset, $per_batch);
				if (!$contacts) {
					$run      = false;
					$has_more = false;
				}
			}
		}

		// Update the offset for the next batch.
		$offset += $per_batch;

		if ($has_more) {
			// run again after 120 seconds.
			$group = 'mailmint-process-price-dropped-' . $automation_id;
			$args  = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'offset'        => $offset,
				'per_page'      => $per_batch,
				'product_id'    => $product_id
			);
			as_schedule_single_action(time() + 30, 'mailmint_process_price_dropped_scheduler', $args, $group);
		}
		return $has_more;
	}

	/**
	 * Hook callback triggered after a WooCommerce review is received.
	 *
	 * This method is hooked into the action 'mint_after_woocommerce_review_received'. It checks if
	 * the review status is approved (status 1) and, if so, prepares the review data and triggers an
	 * automation using the 'MINT_TRIGGER_AUTOMATION' action.
	 *
	 * @param int $comment_id The ID of the comment.
	 * @param int $status     The status of the comment.
	 *
	 * @return void
	 * @since 1.5.14
	 */
	public function mint_after_woocommerce_review_received( $comment_id, $status ) {
		// Check if the comment status is approved.
		if ( 1 !== absint( $status ) ) {
			return;
		}

		// Retrieve the comment and associated product.
		$comment = get_comment( $comment_id );

		// Check if the comment has a valid post ID and if it's associated with a product.
		if ( ! $comment || ! isset( $comment->comment_post_ID ) ) {
			return;
		}

		// Check if the product exists before proceeding.
		$product = wc_get_product( $comment->comment_post_ID );

		if ( ! $product ) {
			return;
		}
		// Prepare the review data.
		$data = $this->prepare_review_data( $comment, $product );
		// Trigger an automation using the 'MINT_TRIGGER_AUTOMATION' action.
		do_action( MINT_TRIGGER_AUTOMATION, $data );
	}

	/**
	 * Hook callback triggered after a WooCommerce review is approved.
	 *
	 * This method is hooked into the action 'transition_comment_status'. It checks if
	 * the new status of the comment is 'approved', and if so, retrieves the associated
	 * product and prepares the review data. Finally, it triggers an automation using the
	 * 'MINT_TRIGGER_AUTOMATION' action.
	 *
	 * @param string $new_status The new status of the comment.
	 * @param string $old_status The old status of the comment.
	 * @param object $comment    The comment object.
	 *
	 * @return void
	 * @since 1.5.14
	 */
	public function mint_after_woocommerce_review_approved( $new_status, $old_status, $comment ) {
		// Check if the new status of the comment is 'approved'.
		if ( 'approved' !== $new_status ) {
			return;
		}

		// Retrieve the associated product.
		$product = wc_get_product( $comment->comment_post_ID );
		if ( ! $product ) {
			return;
		}

		$data = $this->prepare_review_data( $comment, $product );
		// Trigger an automation using the 'MINT_TRIGGER_AUTOMATION' action.
		do_action( MINT_TRIGGER_AUTOMATION, $data );
	}

	/**
	 * Order refunded
	 *
	 * @param string $order_id Get Woocommerce Order ID.
	 * @param object $refund_id Get Woocommerce Order Refund ID.
	 */
	public function mrm_order_refunded( $order_id, $refund_id ) {
		$order = new \WC_Order( $order_id );
		if ( $order ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_order_refunded',
				'data'           => array(
					'user_email' => $order->get_billing_email(),
					'first_name' => $order->get_billing_first_name(),
					'last_name'  => $order->get_billing_last_name(),
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Handles the action after a cart is marked as abandoned.
	 *
	 * Retrieves the cart details based on the abandoned ID and status, and triggers an automation event
	 * to process the abandoned cart data.
	 *
	 * @param int $abandoned_id The ID of the abandoned cart.
	 * @return void
	 * @since 1.5.0
	 */
	public function mrm_mailmint_after_cart_abandoned( $abandoned_id ) {
		// Get the cart details based on the ID and status.
		$cart_details = Model::get_cart_details_by_key( 'id', $abandoned_id, 'abandoned' );
		if ( !empty( $cart_details ) ) {
			$checkout_data = isset( $cart_details['checkout_data'] ) ? maybe_unserialize( $cart_details['checkout_data'] ) : array();

			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_abandoned_cart',
				'data'           => array(
					'user_email'   => !empty( $cart_details['email'] ) ? $cart_details['email'] : '',
					'session_key'  => !empty( $cart_details['session_key'] ) ? $cart_details['session_key'] : '',
					'items'        => maybe_unserialize( $cart_details['items'] ),
					'abandoned_id' => $abandoned_id,
					'first_name'   => isset($checkout_data['fields']['billing_first_name']) ? $checkout_data['fields']['billing_first_name'] : (isset($checkout_data['fields']['billing-first_name']) ? $checkout_data['fields']['billing-first_name'] : ''),
					'last_name'    => isset($checkout_data['fields']['billing_last_name']) ? $checkout_data['fields']['billing_last_name'] : (isset($checkout_data['fields']['billing-last_name']) ? $checkout_data['fields']['billing-last_name'] : ''),

				),
			);

			/**
			 * Trigger an automation based on the provided data.
			 *
			 * @param string $action_name The name of the action to trigger the automation.
			 * @param array  $data        The data associated with the automation trigger.
			 *
			 * @return void
			 * @since 1.5.0
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Handles manully run  the action after a cart is marked as abandoned.
	 *
	 * Retrieves the cart details based on the abandoned ID and status, and triggers an automation event
	 * to process the abandoned cart data.
	 *
	 * @param int $abandoned_id The ID of the abandoned cart.
	 * @param int $automation_id The ID of the automation id.
	 * @return void
	 * @since 1.5.0
	 */
	public function manually_run_abandoned_data( $abandoned_id, $automation_id ) {
		// Get the cart details based on the ID and status.
		$cart_details = Model::get_cart_details_by_key( 'id', $abandoned_id, 'abandoned' );
		if ( !empty( $cart_details ) ) {
			$data = array(
				'connector_name'          => $this->connector_name,
				'manually_run_automation' => true,
				'trigger_name'            => 'wc_abandoned_cart',
				'data'                    => array(
					'user_email'           => !empty( $cart_details['email'] ) ? $cart_details['email'] : '',
					'session_key'          => !empty( $cart_details['session_key'] ) ? $cart_details['session_key'] : '',
					'items'                => maybe_unserialize( $cart_details['items'] ),
					'abandoned_id'         => $abandoned_id,
					'manual_automation_id' => $automation_id,
				),
			);

			/**
			 * Trigger an automation based on the provided data.
			 *
			 * @param string $action_name The name of the action to trigger the automation.
			 * @param array  $data        The data associated with the automation trigger.
			 *
			 * @return void
			 * @since 1.5.0
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Handles the action after a cart is marked as Lost.
	 *
	 * Retrieves the cart details based on the abandoned ID and status, and triggers an automation event
	 * to process the abandoned cart data.
	 *
	 * @param array $payload The ID of the abandoned cart.
	 * @return void
	 * @since 1.5.0
	 */
	public function after_abandoned_cart_lost( $payload ) {
		// Get the cart details based on the ID and status.
		$id           = !empty( $payload['id'] ) ? $payload['id'] : 0;
		$cart_details = Model::get_cart_details_by_key( 'id', $id, 'lost' );
		if ( !empty( $cart_details ) ) {
			$checkout_data = isset($cart_details['checkout_data']) ? maybe_unserialize($cart_details['checkout_data']) : array();

			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_abandoned_cart_lost',
				'data'           => array(
					'user_email'   => !empty( $cart_details['email'] ) ? $cart_details['email'] : '',
					'session_key'  => !empty( $cart_details['session_key'] ) ? $cart_details['session_key'] : '',
					'items'        => maybe_unserialize( $cart_details['items'] ),
					'abandoned_id' => $id,
					'first_name'   => isset($checkout_data['fields']['billing_first_name']) ? $checkout_data['fields']['billing_first_name'] : (isset($checkout_data['fields']['billing-first_name']) ? $checkout_data['fields']['billing-first_name'] : ''),
					'last_name'    => isset($checkout_data['fields']['billing_last_name']) ? $checkout_data['fields']['billing_last_name'] : (isset($checkout_data['fields']['billing-last_name']) ? $checkout_data['fields']['billing-last_name'] : ''),
				),
			);

			/**
			 * Trigger an automation based on the provided data.
			 *
			 * @param string $action_name The name of the action to trigger the automation.
			 * @param array  $data        The data associated with the automation trigger.
			 *
			 * @return void
			 * @since 1.5.0
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Handles the action after a cart is marked as Lost.
	 *
	 * Retrieves the cart details based on the abandoned ID and status, and triggers an automation event
	 * to process the abandoned cart data.
	 *
	 * @param array $payload The ID of the abandoned cart.
	 * @return void
	 * @since 1.5.0
	 */
	public function after_abandoned_cart_recovered( $payload ) {
		// Get the cart details based on the ID and status.
		$id           = !empty( $payload['id'] ) ? $payload['id'] : 0;
		$cart_details = Model::get_cart_details_by_key( 'id', $id, 'recovered' );
		if ( !empty( $cart_details ) ) {
			$checkout_data = isset($cart_details['checkout_data']) ? maybe_unserialize($cart_details['checkout_data']) : array();

			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'wc_abandoned_cart_recovered',
				'data'           => array(
					'user_email'   => !empty( $cart_details['email'] ) ? $cart_details['email'] : '',
					'session_key'  => !empty( $cart_details['session_key'] ) ? $cart_details['session_key'] : '',
					'items'        => maybe_unserialize( $cart_details['items'] ),
					'abandoned_id' => $id,
					'first_name'   => isset($checkout_data['fields']['billing_first_name']) ? $checkout_data['fields']['billing_first_name'] : (isset($checkout_data['fields']['billing-first_name']) ? $checkout_data['fields']['billing-first_name'] : ''),
					'last_name'    => isset($checkout_data['fields']['billing_last_name']) ? $checkout_data['fields']['billing_last_name'] : (isset($checkout_data['fields']['billing-last_name']) ? $checkout_data['fields']['billing-last_name'] : ''),
				),
			);

			/**
			 * Trigger an automation based on the provided data.
			 *
			 * @param string $action_name The name of the action to trigger the automation.
			 * @param array  $data        The data associated with the automation trigger.
			 *
			 * @return void
			 * @since 1.5.0
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Checks if the order matches the category criteria.
	 *
	 * @since 1.5.5
	 *
	 * @param array  $step_data The step data containing criteria settings.
	 * @param object $order The order object to check.
	 * @return bool Whether the order matches the category criteria.
	 */
	public static function check_category_criteria( $step_data, $order ) {
		if ( !empty( $step_data['settings']['product_settings']['category'] ) ) {
			$selected_category = $step_data['settings']['product_settings']['category'];
			$items             = $order->get_items();
			foreach ( $items as $item ) {
				$product_id = $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
				$terms      = get_the_terms( $product_id, 'product_cat' );
				// Check if $terms is a valid array.
				if (is_array($terms)) {
					$term_ids            = array_column($terms, 'term_id');
					$selected_categories = array_column($selected_category, 'value');
					$intersection        = array_intersect($selected_categories, $term_ids);
					if (!empty($intersection)) {
						return true;
					}
				}
			}
			return false;
		}
		return true;
	}

	/**
	 * Checks if the order matches the product criteria.
	 *
	 * @since 1.5.5
	 *
	 * @param array  $step_data The step data containing criteria settings.
	 * @param object $order The order object to check.
	 * @return bool Whether the order matches the product criteria.
	 */
	public static function check_product_criteria( $step_data, $order ) {
		if ( !empty( $step_data['settings']['product_settings']['products'] ) ) {
			$selected_products = $step_data['settings']['product_settings']['products'];
			$items             = $order->get_items();

			// Ensure $selected_products is an array.
			if (is_array($selected_products)) {
				foreach ($items as $item) {
					$product_id = $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
					$key        = array_search($product_id, array_column($selected_products, 'value')); //phpcs:ignore
					if (false !== $key) {
						return true;
					}
				}
			}
			return false;
		}
		return true;
	}

	/**
	 * Check if a product is included based on the selected options.
	 *
	 * This function determines whether a product should be included in a specific context
	 * based on the selected option type, selected category, and selected products.
	 *
	 * @param object $product_id         The WooCommerce product ID.
	 * @param string $option_type       The type of option selected ('choose-category' or 'choose-product').
	 * @param int    $selected_category The selected category ID.
	 * @param array  $selected_products An array of selected product IDs.
	 *
	 * @return bool Whether the product is included based on the selected options.
	 *
	 * @since 1.5.14
	 */
	public function is_product_included( $product_id, $option_type, $selected_category, $selected_products ) {
		if ( 'choose-category' === $option_type ) {
			return empty( $selected_category ) || $this->is_product_in_selected_category( $product_id, $selected_category );
		} elseif ( 'choose-product' === $option_type ) {
			return empty( $selected_products ) || $this->is_product_in_selected_products( $product_id, $selected_products );
		}
		return true;
	}

	/**
	 * Check if a product is in the selected category.
	 *
	 * This function determines whether a WooCommerce product is in the selected category.
	 *
	 * @param object $product_id        The WooCommerce product ID.
	 * @param array  $selected_category An array of selected category IDs.
	 *
	 * @return bool Whether the product is in the selected category.
	 *
	 * @since 1.5.14
	 */
	public function is_product_in_selected_category( $product_id, $selected_category ) {
		$terms = get_the_terms( $product_id, 'product_cat' );
		// Ensure $terms is a valid array.
		if (is_array($terms)) {
			$term_ids            = array_column($terms, 'term_id');
			$selected_categories = is_array($selected_category) ? array_column($selected_category, 'value') : array();
			$intersection        = array_intersect($selected_categories, $term_ids);
			return !empty($intersection);
		}
	}

	/**
	 * Check if a product is in the selected products.
	 *
	 * This function determines whether a WooCommerce product is in the selected products.
	 *
	 * @param object $product_id        The WooCommerce product ID.
	 * @param array  $selected_products An array of selected product IDs.
	 *
	 * @return bool Whether the product is in the selected products.
	 *
	 * @since 1.5.14
	 */
	public function is_product_in_selected_products( $product_id, $selected_products ) {
		$product = wc_get_product($product_id);

		if ( $product->is_type( 'variable' ) ) {
			$variation_ids   = $product->get_children();
			$variation_ids[] = $product_id;
			$key             = array_intersect( $variation_ids, array_column( $selected_products, 'value' ) );
			return false !== $key;
		}

		$key = array_search( $product_id, array_column( $selected_products, 'value' ), true );
		return false !== $key;
	}

	/**
	 * Prepare data for the WooCommerce review automation trigger.
	 *
	 * This method takes a comment object and a product object, extracts relevant
	 * information, and structures it into an array suitable for triggering an automation.
	 *
	 * @param object $comment  The comment object.
	 * @param object $product  The product object associated with the review.
	 *
	 * @return array An array containing the connector name, trigger name, and data for the automation trigger.
	 * @since 1.5.14
	 */
	public function prepare_review_data( $comment, $product ) {
		return array(
			'connector_name' => $this->connector_name,
			'trigger_name'   => 'wc_review_received',
			'data'           => array(
				'user_email' => $comment->comment_author_email,
				'product_id' => $product->get_id(),
			),
		);
	}

	/**
	 * Get contacts for the price dropped event.
	 * 
	 * This function is used to get contacts for the price dropped event.
	 * 
	 * @param array $settings The settings for the price dropped event.
	 * @param int $per_batch The number of contacts to retrieve per batch.
	 * 
	 * @return array
	 * @since 1.13.0
	 */
	private function get_contacts($settings, $offset, $per_batch){
		$contacts = array();
		$lists    = isset($settings['lists']) ? $settings['lists'] : array();
		$tags     = isset($settings['tags']) ? $settings['tags'] : array();
		$segments = isset($settings['segments']) ? $settings['segments'] : array();

		if ((!MrmCommon::is_mailmint_pro_active() || !MrmCommon::is_mailmint_pro_license_active()) || (empty($lists) && empty($tags) && empty($segments))) {
			global $wpdb;
			$contact_table = $wpdb->prefix . ContactSchema::$table_name;
			$query_results = $wpdb->get_results($wpdb->prepare("SELECT id, email FROM $contact_table WHERE status = %s LIMIT %d, %d", array('subscribed', $offset, $per_batch)), ARRAY_A); // db call ok. ; no-cache ok.
			$contacts      = array_merge($contacts, $query_results);
		} else {
			$contacts = apply_filters('mint_process_post_published', $settings, $contacts, $offset, $per_batch);
		}

		return $contacts;
	}
}

