<?php
/**
 * WooCommerce Membership Triggers
 * Description: This file is used to run triggers for WooCommerce Membership events.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Class MembershipTriggers
 * Description: This class is used to run triggers for WooCommerce Membership events.
 *
 * @since 1.15.0
 */
class WishlistTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var string Holds the name of the connector.
	 * @since 1.15.0
	 */
	public $connector_name = 'Wishlist';

    /**
     * Holds the admin membership object.
     *
     * @var null|WC_Memberships_User_Membership
     * @since 1.15.0
     */
    private $admin_membership = null;

	/**
	 * Initializes the triggers for the WooCommerce Membership connector.
	 *
	 * @return void
	 * @since 1.15.0
	 */
	public function init() {
        // WooCommerce User Adds Product to Wishlist.
        add_action( 'woocommerce_wishlist_add_item', array( $this, 'handle_add_item_to_wishlist' ), PHP_INT_MAX, 7 );
	}

    public function handle_add_item_to_wishlist( $cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data, $wishlist_id ) {
		$data['wishlist_id']    = $wishlist_id;
		$data['cart_item_data'] = $cart_item_data;
		$data['product_id']     = $product_id;
		$user_email       = get_post_meta( $wishlist_id, '_wishlist_email', true );
		$first_name       = get_post_meta( $wishlist_id, '_wishlist_first_name', true );
		$last_name       = get_post_meta( $wishlist_id, '_wishlist_last_name', true );

		if ( empty( $user_email ) ) {
			return;
		}

		$data = array(
			'connector_name' => $this->connector_name,
			'trigger_name'   => 'wcw_user_adds_product',
			'data'           => array(
				'user_email'         => $user_email,
				'first_name'         => $first_name,
				'last_name'          => $last_name,
				'product_id' => $product_id,
				'cart_item_data' => $cart_item_data,
				'wishlist_id' => $wishlist_id,
			),
		);

		/**
		 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
		 *
		 * This action triggers automation processes with the given data for handling automation events.
		 *
		 * @param mixed $data The data associated with the automation trigger.
		 * @since 1.8.0
		 */
		do_action( MINT_TRIGGER_AUTOMATION, $data );
	}

	/**
	 * Validate the settings based on the trigger name.
	 *
	 * This function retrieves step data and delegates validation to specific methods 
	 * based on the trigger name.
	 *
	 * @param array $step_data An array containing the automation ID and step ID.
	 * @param array $data      An array containing the trigger name and other data.
	 *
	 * @return bool True if the settings are valid, false otherwise.
	 * @since 1.15.0
	 */
	public function validate_settings( $step_data, $data ) {
		$step_data    = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
		$trigger_name = isset( $data['trigger_name'] ) ? $data['trigger_name'] : '';

		switch ( $trigger_name ) {
			case 'wcw_user_adds_product':
				return $this->validate_user_add_product_wishlist_settings( $step_data, $data );

			default:
				return false;
		}
	}

	/**
	 * Validate the settings for the 'wcm_membership_created' trigger.
	 *
	 * This function checks the settings for the 'wcm_membership_created' trigger 
	 * to ensure they are correctly configured.
	 *
	 * @param array $step_data An array containing the step data, including settings.
	 * @param array $data      An array containing the subscription data, including the subscription ID.
	 *
	 * @return bool True if the settings are valid and the subscription matches the criteria, false otherwise.
	 * @since 1.15.0
	 */
	private function validate_user_add_product_wishlist_settings( $step_data, $data ) {
		$settings    = isset( $step_data['settings']['product_settings'] ) ? $step_data['settings']['product_settings'] : array();
		$option_type = isset( $settings['option_type'] ) ? $settings['option_type'] : 'choose-all';
	
		if ( 'choose-all' === $option_type ) {
			return true;
		}
	
		$product_id = isset( $data['data']['product_id'] ) ? $data['data']['product_id'] : '';

		if ( 'choose-product' === $option_type ) {
			return $this->check_products_in_subscription( $product_id, isset( $settings['products'] ) ? $settings['products'] : array() );
		}
	
		if ( 'choose-category' === $option_type ) {
			return $this->check_categories_in_subscription( $product_id, $settings['category'] ?? array() );
		}
	}

	/**
	 * Check if any of the specified products are in the subscription.
	 *
	 * This function checks if any of the products specified in the settings are part of the given subscription.
	 *
	 * @param WC_Subscription $subscription The subscription object to check.
	 * @param array           $products     An array of products to check against the subscription.
	 *
	 * @return bool True if any of the specified products are found in the subscription, false otherwise.
	 * @since 1.15.0
	 */
	private function check_products_in_subscription( $product_id, $products ) {
		if ( empty( $products ) ) {
			return false;
		}
	
		$product_ids = array_column( $products, 'value' );
	
		if ( in_array( $product_id, $product_ids, true ) ) {
			return true;
		}
	
		return false;
	}

	/**
	 * Check if any of the specified categories are in the subscription.
	 *
	 * This function checks if any of the categories specified in the settings are part of the given subscription.
	 *
	 * @param WC_Subscription $subscription The subscription object to check.
	 * @param array           $categories   An array of categories to check against the subscription.
	 *
	 * @return bool True if any of the specified categories are found in the subscription, false otherwise.
	 * @since 1.15.0
	 */
	private function check_categories_in_subscription( $product_id, $categories ) {
		if ( empty( $categories ) ) {
			return false;
		}
	
		$categories = array_column( $categories, 'value' );
		$terms      = get_the_terms( $product_id, 'product_cat' );
	
		if ( $terms ) {
			$term_ids     = array_column( $terms, 'term_id' );
			$intersection = array_intersect( $categories, $term_ids );

			if ( ! empty( $intersection ) ) {
				return true;
			}
		}
	
		return false;
	}
}

