<?php
/**
 * WooCommerce Subscription Triggers
 * Description: This file is used to run triggers for WooCommerce Subscription events.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;
use WC_Order;
use Mint\App\Internal\Cron\BackgroundProcessHelper;

/**
 * Class SubscriptionTriggers
 * Description: This class is used to run triggers for WooCommerce Subscription events.
 *
 * @since 1.15.0
 */
class SubscriptionTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var string Holds the name of the connector.
	 * @since 1.15.0
	 */
	public $connector_name = 'Subscription';

	/**
	 * Automation ID for the currently running automation.
	 * 
	 * @var int Holds the automation ID.
	 * @since 1.15.0
	 */
	private $automation_id;

	/**
	 * Initializes the triggers for the WooCommerce Subscription connector.
	 *
	 * @return void
	 * @since 1.15.0
	 */
	public function init() {
        // WooCommerce Subscription Created.
        add_action( 'woocommerce_checkout_subscription_created', array( $this, 'handle_subscription_created' ), PHP_INT_MAX, 2 );
		add_action( 'wcs_api_subscription_created', array( $this, 'handle_subscription_created' ), PHP_INT_MAX );
		add_action( 'woocommerce_admin_created_subscription', array( $this, 'handle_subscription_created' ), PHP_INT_MAX );

		// WooCommerce Subscription Status Changed.
		add_action( 'woocommerce_subscription_status_updated', array( $this, 'handle_subscription_status_updated' ), PHP_INT_MAX, 3 );

		// WooCommerce Subscription Trial End.
		add_action( 'woocommerce_scheduled_subscription_trial_end', array( $this, 'handle_scheduled_subscription_trial_end' ), PHP_INT_MAX, 1 );

		// WooCommerce Subscription Before Renewal.
		add_action( 'mailmint_process_wcs_renewal_daily', array( $this, 'handle_wcs_before_renewal_daily' ), 10, 4 );
		add_action( 'mailmint_process_wcs_renewal_daily_once', array( $this, 'handle_wcs_before_renewal_daily' ), 10, 4 );

		// WooCommerce Subscription Before End.
		add_action( 'mailmint_process_wcs_end_daily', array( $this, 'handle_wcs_before_end_daily' ), 10, 4 );
		add_action( 'mailmint_process_wcs_end_daily_once', array( $this, 'handle_wcs_before_end_daily' ), 10, 4 );
	}

	/**
	 * Validate the settings based on the trigger name.
	 *
	 * This function retrieves step data and delegates validation to specific methods 
	 * based on the trigger name.
	 *
	 * @param array $step_data An array containing the automation ID and step ID.
	 * @param array $data      An array containing the trigger name and other data.
	 *
	 * @return bool True if the settings are valid, false otherwise.
	 * @since 1.15.0
	 */
	public function validate_settings( $step_data, $data ) {
		$step_data    = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
		$trigger_name = isset( $data['trigger_name'] ) ? $data['trigger_name'] : '';

		switch ( $trigger_name ) {
			case 'wcs_subscription_created':
				return $this->validate_subscription_created_settings( $step_data, $data );
	
			case 'wcs_subscription_status_changed':
				return $this->validate_subscription_status_changed_settings( $step_data, $data );
			
			case 'wcs_subscription_trial_end':
				return $this->validate_subscription_created_settings( $step_data, $data );

			case 'wcs_subscription_before_renewal':
				if ( $step_data['automation_id'] === $this->automation_id ) {
					return $this->validate_subscription_created_settings( $step_data, $data );
				}
				return false;

			case 'wcs_subscription_before_end':
				if ( $step_data['automation_id'] === $this->automation_id ) {
					return $this->validate_subscription_created_settings( $step_data, $data );
				}
				return false;
			default:
				return false;
		}
	}

	/**
	 * Validate the settings for the 'wcs_subscription_created' trigger.
	 *
	 * This function checks the settings for the 'wcs_subscription_created' trigger 
	 * to ensure they are correctly configured.
	 *
	 * @param array $step_data An array containing the step data, including settings.
	 * @param array $data      An array containing the subscription data, including the subscription ID.
	 *
	 * @return bool True if the settings are valid and the subscription matches the criteria, false otherwise.
	 * @since 1.15.0
	 */
	private function validate_subscription_created_settings( $step_data, $data ) {
		$settings    = isset( $step_data['settings']['product_settings'] ) ? $step_data['settings']['product_settings'] : array();
		$option_type = isset( $settings['option_type'] ) ? $settings['option_type'] : 'choose-all';

		$subscription_id = isset( $data['data']['subscription_id'] ) ? $data['data']['subscription_id'] : '';
		$subscription    = wcs_get_subscription( $subscription_id );

		// only trigger for active subscriptions
		if ( 'wcs_subscription_before_renewal' === $data['trigger_name'] && ! $subscription->has_status( 'active' ) ) {
			return false;
		}

		// only trigger for active subscriptions
		if ( 'wcs_subscription_before_end' === $data['trigger_name'] && ! $subscription->has_status( [ 'active', 'pending-cancel' ] ) ) {
			return false;
		}
	
		if ( 'choose-all' === $option_type ) {
			return true;
		}
	
		if ( 'choose-product' === $option_type ) {
			return $this->check_products_in_subscription( $subscription, isset( $settings['products'] ) ? $settings['products'] : array() );
		}
	
		if ( 'choose-category' === $option_type ) {
			return $this->check_categories_in_subscription( $subscription, $settings['category'] ?? array() );
		}
	
		return false;
	}

	/**
	 * Validate the settings for the 'wcs_subscription_status_changed' trigger.
	 *
	 * This function checks the settings for the 'wcs_subscription_status_changed' trigger 
	 * to ensure they are correctly configured.
	 *
	 * @param array $step_data An array containing the step data, including settings.
	 * @param array $data      An array containing the subscription data, including the subscription ID.
	 *
	 * @return bool True if the settings are valid and the subscription matches the criteria, false otherwise.
	 * @since 1.15.0
	 */
	public function validate_subscription_status_changed_settings( $step_data, $data ) {
		$settings    = isset( $step_data['settings']['product_settings'] ) ? $step_data['settings']['product_settings'] : array();
		$option_type = isset( $settings['option_type'] ) ? $settings['option_type'] : 'choose-all';
		$status_from = isset( $settings['status_from'] ) ? $settings['status_from'] : 'wc-any';
		$status_to   = isset( $settings['status_to'] ) ? $settings['status_to'] : 'wc-any';
		$status_from = str_replace( 'wc-', '', $status_from );
		$status_to   = str_replace( 'wc-', '', $status_to );

		$new_status = isset( $data['data']['new_status'] ) ? $data['data']['new_status'] : '';
		$old_status = isset( $data['data']['old_status'] ) ? $data['data']['old_status'] : '';

		if ( ( 'any' === $status_from && 'any' === $status_to ) ||
			( 'any' === $status_from && $status_to === $new_status ) ||
			( $status_from === $old_status && 'any' === $status_to ) ||
			( $status_from === $old_status && $status_to === $new_status ) ) {
        
			if ( 'choose-all' === $option_type ) {
				return true;
			}

			$subscription_id = isset( $data['data']['subscription_id'] ) ? $data['data']['subscription_id'] : '';
			$subscription    = wcs_get_subscription( $subscription_id );

			if ( ! $subscription ) {
				return false;
			}

			if ( 'choose-product' === $option_type ) {
				return $this->check_products_in_subscription( $subscription, isset( $settings['products'] ) ? $settings['products'] : array() );
			}

			if ( 'choose-category' === $option_type ) {
				return $this->check_categories_in_subscription( $subscription, $settings['category'] ?? array() );
			}

			return false;
		}
	
		return false;
	}

	/**
	 * Check if any of the specified products are in the subscription.
	 *
	 * This function checks if any of the products specified in the settings are part of the given subscription.
	 *
	 * @param WC_Subscription $subscription The subscription object to check.
	 * @param array           $products     An array of products to check against the subscription.
	 *
	 * @return bool True if any of the specified products are found in the subscription, false otherwise.
	 * @since 1.15.0
	 */
	private function check_products_in_subscription( $subscription, $products ) {
		if ( empty( $products ) ) {
			return false;
		}
	
		$product_ids = array_column( $products, 'value' );
		$items       = $subscription->get_items();
	
		foreach ( $items as $item ) {
			$product_id = $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
			if ( in_array( $product_id, $product_ids, true ) ) {
				return true;
			}
		}
	
		return false;
	}

	/**
	 * Check if any of the specified categories are in the subscription.
	 *
	 * This function checks if any of the categories specified in the settings are part of the given subscription.
	 *
	 * @param WC_Subscription $subscription The subscription object to check.
	 * @param array           $categories   An array of categories to check against the subscription.
	 *
	 * @return bool True if any of the specified categories are found in the subscription, false otherwise.
	 * @since 1.15.0
	 */
	private function check_categories_in_subscription( $subscription, $categories ) {
		if ( empty( $categories ) ) {
			return false;
		}
	
		$selected_categories = array_column( $categories, 'value' );
		$items               = $subscription->get_items();
	
		foreach ( $items as $item ) {
			$product_id = $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
			$terms      = get_the_terms( $product_id, 'product_cat' );
	
			if ( $terms ) {
				$term_ids     = array_column( $terms, 'term_id' );
				$intersection = array_intersect( $selected_categories, $term_ids );
	
				if ( ! empty( $intersection ) ) {
					return true;
				}
			}
		}
	
		return false;
	}

	/**
	 * Handle the subscription created event.
	 *
	 * This function is triggered when a new subscription is created. It prepares the necessary data 
	 * and triggers an automation event with the relevant subscription details.
	 *
	 * @param int|WC_Subscription $subscription The subscription object or ID.
	 * @param WC_Order|string     $order        The order object or an empty string if not provided.
	 *
	 * @return void
	 * @since 1.15.0
	 */
	public function handle_subscription_created( $subscription, $order = '' ) {
		$subscription = wcs_get_subscription( $subscription );
		if ( ! $subscription ) {
			return;
		}

		$subscription_id = $subscription->get_id();
		$order_id        = $order instanceof WC_Order ? $order->get_id() : $subscription->get_last_order();

		// Prepare the data for the trigger and run the trigger.
		$data = array(
			'connector_name' => $this->connector_name,
			'trigger_name'   => 'wcs_subscription_created',
			'data'           => array(
				'user_email'      => $subscription->get_billing_email(),
				'first_name'      => $subscription->get_billing_first_name(),
				'last_name'       => $subscription->get_billing_last_name(),
				'order_id'        => $order_id,
				'subscription_id' => $subscription_id,
			),
		);
		do_action( MINT_TRIGGER_AUTOMATION, $data );
	}

	/**
	 * Handle the subscription status updated event.
	 *
	 * This function is triggered when the status of a subscription is updated. It prepares the necessary data 
	 * and triggers an automation event with the relevant subscription details.
	 *
	 * @param int|WC_Subscription $subscription The subscription object or ID.
	 * @param string              $new_status  The new status of the subscription.
	 * @param string              $old_status  The old status of the subscription.
	 *
	 * @return void
	 * @since 1.15.0
	 */
	public function handle_subscription_status_updated( $subscription, $new_status, $old_status ) {
		// Handle the subscription status updated event.
		$subscription = wcs_get_subscription( $subscription );
		if ( ! $subscription ) {
			return;
		}

		$subscription_id = $subscription->get_id();

		// Prepare the data for the trigger and run the trigger.
		$data = array(
			'connector_name' => $this->connector_name,
			'trigger_name'   => 'wcs_subscription_status_changed',
			'data'           => array(
				'user_email'      => $subscription->get_billing_email(),
				'first_name'      => $subscription->get_billing_first_name(),
				'last_name'       => $subscription->get_billing_last_name(),
				'subscription_id' => $subscription_id,
				'new_status'      => $new_status,
				'old_status'      => $old_status,
			),
		);
		do_action( MINT_TRIGGER_AUTOMATION, $data );
	}

	/**
	 * Handle the subscription trial end event.
	 *
	 * This function is triggered when the trial of a subscription is end. It prepares the necessary data 
	 * and triggers an automation event with the relevant subscription details.
	 *
	 * @param int|WC_Subscription $subscription_id The subscription ID.
	 *
	 * @return void
	 * @since 1.15.0
	 */
	public function handle_scheduled_subscription_trial_end( $subscription_id ) {
		$subscription = wcs_get_subscription( $subscription_id );
		if ( ! $subscription ) {
			return;
		}

		// Prepare the data for the trigger and run the trigger.
		$data = array(
			'connector_name' => $this->connector_name,
			'trigger_name'   => 'wcs_subscription_trial_end',
			'data'           => array(
				'user_email'      => $subscription->get_billing_email(),
				'first_name'      => $subscription->get_billing_first_name(),
				'last_name'       => $subscription->get_billing_last_name(),
				'subscription_id' => $subscription_id,
			),
		);
		do_action( MINT_TRIGGER_AUTOMATION, $data );
	}

	/**
	 * Handles the daily renewal process for WooCommerce subscriptions.
	 *
	 * This function is triggered daily to process subscriptions before renewal.
	 *
	 * @param int $automation_id The ID of the automation.
	 * @param int $step_id       The ID of the step in the automation.
	 * @param int $offset        The offset for the batch of subscriptions.
	 * @param int $per_batch     The number of subscriptions to process per batch.
	 *
	 * @return bool True if there are more subscriptions to process, false otherwise.
	 * @since 1.15.0
	 */
	public function handle_wcs_before_renewal_daily( $automation_id, $step_id, $offset, $per_batch ) {
		$this->automation_id = $automation_id;
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );
		$settings  = isset( $step_data['settings']['product_settings'] ) ? $step_data['settings']['product_settings'] : array();
		$subscription_ids = $this->get_subscriptions( $settings, $offset, $per_batch );

		if ( !$subscription_ids ) {
            return false;
        }

		$start_time = time();
		$has_more   = true;
        $run        = true;

        if ( $subscription_ids && $run ) {
            foreach ( $subscription_ids as $subscription_id ) {
				$subscription = wcs_get_subscription( $subscription_id );
				$data = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'wcs_subscription_before_renewal',
					'data'           => array(
						'user_email'      => $subscription->get_billing_email(),
						'first_name'      => $subscription->get_billing_first_name(),
						'last_name'       => $subscription->get_billing_last_name(),
						'subscription_id' => $subscription_id,
					),
				);
				do_action( MINT_TRIGGER_AUTOMATION, $data );				
            }

            if ( BackgroundProcessHelper::memory_exceeded() || time() - $start_time > 40 ) {
                $run      = false;
                $has_more = true;
            } else {
                $subscription_ids = $this->get_subscriptions( $settings, $offset, $per_batch );
                if ( !$subscription_ids ) {
                    $run      = false;
                    $has_more = false;
                }
            }
        }

		// Update the offset for the next batch.
		$offset += $per_batch;

		if ( $has_more ) {
            // run again after 120 seconds.
			$group = 'mailmint-process-wcs-renewal-' . $automation_id;
			$args  = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'offset'        => $offset,
				'per_page'      => $per_batch,
			);
			as_schedule_single_action( time() + 120, 'mailmint_process_wcs_renewal_daily_once', $args, $group );
        }
        return $has_more;
	}

	/**
	 * Handles the daily end process for WooCommerce subscriptions.
	 *
	 * This function is triggered daily to process subscriptions before end.
	 *
	 * @param int $automation_id The ID of the automation.
	 * @param int $step_id       The ID of the step in the automation.
	 * @param int $offset        The offset for the batch of subscriptions.
	 * @param int $per_batch     The number of subscriptions to process per batch.
	 *
	 * @return bool True if there are more subscriptions to process, false otherwise.
	 * @since 1.15.0
	 */
	public function handle_wcs_before_end_daily( $automation_id, $step_id, $offset, $per_batch ) {
		$this->automation_id = $automation_id;
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );
		$settings  = isset( $step_data['settings']['product_settings'] ) ? $step_data['settings']['product_settings'] : array();
		$subscription_ids = $this->get_subscriptions_before_end( $settings, $offset, $per_batch );

		if ( !$subscription_ids ) {
            return false;
        }

		$start_time = time();
		$has_more   = true;
        $run        = true;

        if ( $subscription_ids && $run ) {
            foreach ( $subscription_ids as $subscription_id ) {
				$subscription = wcs_get_subscription( $subscription_id );
				$data = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'wcs_subscription_before_end',
					'data'           => array(
						'user_email'      => $subscription->get_billing_email(),
						'first_name'      => $subscription->get_billing_first_name(),
						'last_name'       => $subscription->get_billing_last_name(),
						'subscription_id' => $subscription_id,
					),
				);
				do_action( MINT_TRIGGER_AUTOMATION, $data );				
            }

            if ( BackgroundProcessHelper::memory_exceeded() || time() - $start_time > 40 ) {
                $run      = false;
                $has_more = true;
            } else {
                $subscription_ids = $this->get_subscriptions_before_end( $settings, $offset, $per_batch );
                if ( !$subscription_ids ) {
                    $run      = false;
                    $has_more = false;
                }
            }
        }

		// Update the offset for the next batch.
		$offset += $per_batch;

		if ( $has_more ) {
            // run again after 120 seconds.
			$group = 'mailmint-process-wcs-end-' . $automation_id;
			$args  = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'offset'        => $offset,
				'per_page'      => $per_batch,
			);
			as_schedule_single_action( time() + 120, 'mailmint_process_wcs_end_daily_once', $args, $group );
        }
        return $has_more;
	}

	/**
	 * Retrieves subscriptions that are due for end within a specific timeframe.
	 *
	 * @param array $settings The settings array containing 'days_before' and other settings.
	 * @param int   $offset   The offset for the batch of subscriptions.
	 * @param int   $limit    The number of subscriptions to retrieve.
	 *
	 * @return array The IDs of the subscriptions that match the criteria.
	 * @since 1.15.0
	 */
	protected function get_subscriptions_before_end( $settings, $offset, $limit ) {
		$days_before_end = isset( $settings['days_before'] ) ? (int)$settings['days_before'] : 7;
		$date = ( new \DateTime() )->add( new \DateInterval( "P{$days_before_end}D" ) );

		return $this->query_subscriptions_for_day( $date, '_schedule_end', array( 'wc-active', 'wc-pending-cancel' ), $offset, $limit );
	}

	/**
	 * Retrieves subscriptions that are due for renewal within a specific timeframe.
	 *
	 * @param array $settings The settings array containing 'days_before' and other settings.
	 * @param int   $offset   The offset for the batch of subscriptions.
	 * @param int   $limit    The number of subscriptions to retrieve.
	 *
	 * @return array The IDs of the subscriptions that match the criteria.
	 * @since 1.15.0
	 */
	protected function get_subscriptions( $settings, $offset, $limit ) {
		$days_before_renewal = isset( $settings['days_before'] ) ? (int)$settings['days_before'] : 7;
		$date = ( new \DateTime() )->add( new \DateInterval( "P{$days_before_renewal}D" ) );

		return $this->query_subscriptions_for_day( $date, '_schedule_next_payment', array( 'wc-active' ), $offset, $limit );
	}

	/**
	 * Queries subscriptions for a specific day based on meta key and statuses.
	 *
	 * @param DateTime $date          The date object representing the target day.
	 * @param string   $date_meta_key The meta key to query by date.
	 * @param array    $statuses      The statuses of the subscriptions to retrieve.
	 * @param int      $offset        The offset for the batch of subscriptions.
	 * @param int      $limit         The number of subscriptions to retrieve.
	 *
	 * @return array The IDs of the subscriptions that match the criteria.
	 * @since 1.15.0
	 */
	protected function query_subscriptions_for_day( $date, $date_meta_key, $statuses, $offset, $limit ) {
		// Get site timezone or default to UTC if not set
		$site_timezone_string = get_option('timezone_string');
		if (empty($site_timezone_string)) {
			$site_timezone_string = 'UTC';
		}
		$site_timezone = new \DateTimeZone($site_timezone_string);
		$date->setTimezone($site_timezone);
	
		// Clone the date for day start and day end
		$day_start = clone $date;
		$day_end = clone $date;
	
		// Set time to day start and end
		$day_start->setTime(0, 0, 0);
		$day_end->setTime(23, 59, 59);
	
		// Convert to UTC time
		$utc_timezone = new \DateTimeZone('UTC');
		$day_start->setTimezone($utc_timezone);
		$day_end->setTimezone($utc_timezone);
	
		// Convert to MySQL date format
		$day_start_mysql = $day_start->format('Y-m-d H:i:s');
		$day_end_mysql = $day_end->format('Y-m-d H:i:s');

		// Check if the function exists before calling it.
		if ( function_exists( 'wcs_get_orders_with_meta_query' ) ) {
			return wcs_get_orders_with_meta_query(
				[
					'type'          => 'shop_subscription',
					'status'        => $statuses,
					'return'        => 'ids',
					'limit'         => $limit,
					'offset'        => $offset,
					'no_found_rows' => true,
					'meta_query'    => [
						[
							'key'     => $date_meta_key,
							'compare' => 'BETWEEN',
							'value'   => [ $day_start_mysql, $day_end_mysql ],
						],
					],
				]
			);
		}

		// Fallback for querying subscriptions before HPOS compatibility was added.
		$query = new \WP_Query(
			[
				'post_type'      => 'shop_subscription',
				'post_status'    => $statuses,
				'fields'         => 'ids',
				'posts_per_page' => $limit,
				'offset'         => $offset,
				'no_found_rows'  => true,
				'meta_query'     => [
					[
						'key'     => $date_meta_key,
						'compare' => '>=',
						'value'   => $day_start_mysql,
					],
					[
						'key'     => $date_meta_key,
						'compare' => '<=',
						'value'   => $day_end_mysql,
					],
				],
			]
		);

		return $query->posts;
	}
}

