<?php
/**
 * WooCommerce Membership Triggers
 * Description: This file is used to run triggers for WooCommerce Membership events.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;
use WC_Order;
use Mint\App\Internal\Cron\BackgroundProcessHelper;

/**
 * Class MembershipTriggers
 * Description: This class is used to run triggers for WooCommerce Membership events.
 *
 * @since 1.15.0
 */
class MembershipTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var string Holds the name of the connector.
	 * @since 1.15.0
	 */
	public $connector_name = 'Membership';

    /**
     * Holds the admin membership object.
     *
     * @var null|WC_Memberships_User_Membership
     * @since 1.15.0
     */
    private $admin_membership = null;

	/**
	 * Initializes the triggers for the WooCommerce Membership connector.
	 *
	 * @return void
	 * @since 1.15.0
	 */
	public function init() {
        // WooCommerce Membership Created.
        add_action( 'wc_memberships_user_membership_created', array( $this, 'handle_memberships_user_membership_created' ), PHP_INT_MAX, 2 );
        // Checking if the membership created via admin.
		if ( is_admin() ) {
			add_action( 'transition_post_status', array( $this, 'transition_post_status' ), PHP_INT_MAX, 3 );
		}

        // WooCommerce Membership Status Change
        add_action( 'wc_memberships_user_membership_status_changed', array( $this, 'handle_memberships_user_membership_status_changed' ), PHP_INT_MAX, 3 );
	}

    /**
     * Handles the transition of a post status for user memberships.
     *
     * @param string $new_status The new status of the post.
     * @param string $old_status The old status of the post.
     * @param WP_Post $post      The post object representing the user membership.
     *
     * @return void
     * @since 1.15.0
     */
	public function transition_post_status( $new_status, $old_status, $post ) {
		if ( $old_status === 'auto-draft' && $post->post_type === 'wc_user_membership' ) {
			// Don't trigger now as post transition happens before data is saved.
			$this->admin_membership = $post->ID;
			add_action( 'wc_memberships_user_membership_saved', array( $this, 'membership_created_via_admin' ), PHP_INT_MAX, 2 );
		}
	}

    /**
     * Handles the event when a membership is created via the admin interface.
     *
     * @param WC_Memberships_Membership_Plan $membership_plan      The membership plan object.
     * @param array                          $user_membership_data An array containing the user membership data, including 'user_membership_id'.
     *
     * @return void
     * @since 1.15.0
     */
    public function membership_created_via_admin( $membership_plan, $user_membership_data ) {
		// Check the created membership is a match.
		if ( $this->admin_membership == $user_membership_data['user_membership_id'] ) {
			$this->handle_memberships_user_membership_created( $membership_plan, $user_membership_data );
		}
	}

    /**
     * Handles the event when a user membership is created.
     *
     * @param WC_Memberships_Membership_Plan $membership_plan      The membership plan object.
     * @param array                          $user_membership_data An array containing the user membership data, including 'user_membership_id', 'user_id', and 'is_update'.
     *
     * @return void
     * @since 1.15.0
     */
    public function handle_memberships_user_membership_created( $membership_plan, $user_membership_data ) {
		// Checked if user id available.
		if ( empty( $user_membership_data['user_id'] ) ) {
			return;
		}

		// Check if membership plan available.
		if ( ! $membership_plan instanceof \WC_Memberships_Membership_Plan ) {
			return;
		}

		// If the membership update then return.
		if ( false === $user_membership_data['is_update'] || ! empty( $this->admin_membership ) ) {
			$user_membership_id = $user_membership_data['user_membership_id'];
			$user_id            = $user_membership_data['user_id'];
			$user_data          = get_userdata( $user_id );

            if ( $user_data ) {
                $data = array(
                    'connector_name' => $this->connector_name,
                    'trigger_name'   => 'wcm_membership_created',
                    'data'           => array(
                        'user_email'         => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
                        'first_name'         => isset( $user_data->first_name ) ? $user_data->first_name : '',
                        'last_name'          => isset( $user_data->last_name ) ? $user_data->last_name : '',
                        'user_membership_id' => $user_membership_id,
                        'membership_plan_id' => $membership_plan->get_id(),
                    ),
                );

                /**
                 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
                 *
                 * This action triggers automation processes with the given data for handling automation events.
                 *
                 * @param mixed $data The data associated with the automation trigger.
                 * @since 1.8.0
                 */
                do_action( MINT_TRIGGER_AUTOMATION, $data );
            }
			$this->admin_membership = null;
		}

	}

	/**
	 * Validate the settings based on the trigger name.
	 *
	 * This function retrieves step data and delegates validation to specific methods 
	 * based on the trigger name.
	 *
	 * @param array $step_data An array containing the automation ID and step ID.
	 * @param array $data      An array containing the trigger name and other data.
	 *
	 * @return bool True if the settings are valid, false otherwise.
	 * @since 1.15.0
	 */
	public function validate_settings( $step_data, $data ) {
		$step_data    = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
		$trigger_name = isset( $data['trigger_name'] ) ? $data['trigger_name'] : '';

		switch ( $trigger_name ) {
			case 'wcm_membership_created':
				return $this->validate_membership_created_settings( $step_data, $data );

            case 'wcm_membership_status_changed':
				return $this->validate_membership_status_changed_settings( $step_data, $data );

			default:
				return false;
		}
	}

	/**
	 * Validate the settings for the 'wcm_membership_created' trigger.
	 *
	 * This function checks the settings for the 'wcm_membership_created' trigger 
	 * to ensure they are correctly configured.
	 *
	 * @param array $step_data An array containing the step data, including settings.
	 * @param array $data      An array containing the subscription data, including the subscription ID.
	 *
	 * @return bool True if the settings are valid and the subscription matches the criteria, false otherwise.
	 * @since 1.15.0
	 */
	private function validate_membership_created_settings( $step_data, $data ) {
		$settings = isset( $step_data['settings']['membership_settings'] ) ? $step_data['settings']['membership_settings'] : array();
		$plans    = isset( $settings['plans'] ) ? $settings['plans'] : array();

		if( empty( $plans ) ) {
            return true;
        }
	
		$membership_plan_id = $data['data']['membership_plan_id'];
		return in_array( $membership_plan_id, array_column( $plans, 'value' ), true );
	}

	/**
	 * Validate the settings for the 'wcm_membership_status_changed' trigger.
	 *
	 * This function checks the settings for the 'wcm_membership_status_changed' trigger 
	 * to ensure they are correctly configured.
	 *
	 * @param array $step_data An array containing the step data, including settings.
	 * @param array $data      An array containing the subscription data, including the subscription ID.
	 *
	 * @return bool True if the settings are valid and the subscription matches the criteria, false otherwise.
	 * @since 1.15.0
	 */
	public function validate_membership_status_changed_settings( $step_data, $data ) {
		$settings    = isset( $step_data['settings']['membership_settings'] ) ? $step_data['settings']['membership_settings'] : array();
		$plans       = isset( $settings['plans'] ) ? $settings['plans'] : array();
		$status_from = isset( $settings['status_from'] ) ? $settings['status_from'] : 'any';
		$status_to   = isset( $settings['status_to'] ) ? $settings['status_to'] : 'any';
		$new_status  = isset( $data['data']['new_status'] ) ? $data['data']['new_status'] : '';
		$old_status  = isset( $data['data']['old_status'] ) ? $data['data']['old_status'] : '';

		if ( ( 'any' === $status_from && 'any' === $status_to ) ||
			( 'any' === $status_from && $status_to === $new_status ) ||
			( $status_from === $old_status && 'any' === $status_to ) ||
			( $status_from === $old_status && $status_to === $new_status ) ) {
        
			if ( empty( $plans ) ) {
				return true;
			}

			$membership_plan_id = $data['data']['membership_plan_id'];
		    return in_array( $membership_plan_id, array_column( $plans, 'value' ), true );
		}
		return false;
	}

	/**
	 * Handle the membership status updated event.
	 *
	 * This function is triggered when the status of a membership is updated. It prepares the necessary data 
	 * and triggers an automation event with the relevant membership details.
	 *
	 * @param \WC_Memberships_User_Membership $membership The membership.
	 * @param string              $old_status  The old status of the membership.
	 * @param string              $new_status  The new status of the membership.
	 *
	 * @return void
	 * @since 1.15.0
	 */
	public function handle_memberships_user_membership_status_changed( $membership, $old_status, $new_status ) {
        $user_id   = $membership->get_user_id();
		$user_data = get_userdata( $user_id );

		// Prepare the data for the trigger and run the trigger.
        if ( $user_data ) {
            $data = array(
                'connector_name' => $this->connector_name,
                'trigger_name'   => 'wcm_membership_status_changed',
                'data'           => array(
                    'user_email'         => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
                    'first_name'         => isset( $user_data->first_name ) ? $user_data->first_name : '',
                    'last_name'          => isset( $user_data->last_name ) ? $user_data->last_name : '',
                    'user_membership_id' => $membership->get_id(),
                    'membership_plan_id' => $membership->get_plan_id(),
                    'old_status'         => $old_status,
                    'new_status'         => $new_status,
                ),
            );

            /**
             * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
             *
             * This action triggers automation processes with the given data for handling automation events.
             *
             * @param mixed $data The data associated with the automation trigger.
             * @since 1.8.0
             */
            do_action( MINT_TRIGGER_AUTOMATION, $data );
        }
	}
}

