<?php
/**
 * Customer Win Back triggers
 * Description: This file is used to run triggers for re-engaging customers.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use MailMintPro\App\Utilities\Helper\MintWC;
use MailMintPro\Mint\Internal\Admin\Segmentation\FilterSegmentContacts;
use Mint\App\Internal\Cron\BackgroundProcessHelper;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;
use MRM\Common\MrmCommon;

/**
 * Class CustomerWinBackTriggers
 * Description: This class is used to run triggers for re-engaging customers.
 *
 * @since 1.12.0
 */
class CustomerWinBackTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var string Holds the name of the connector.
	 * @since 1.12.0
	 */
	public $connector_name = 'CustomerWinBack';

	/**
	 * Automation ID for the currently running automation.
	 * 
	 * @var int Holds the automation ID.
	 * @since 1.12.0
	 */
	private $automation_id;

	/**
	 * Option key for the customer win back email dates.
	 * 
	 * @var string Holds the option key.
	 * @since 1.12.0
	 */
	private $option_key = 'mint_customer_win_back_email_dates';

	/**
	 * Email dates for the customer win back emails.
	 * 
	 * @var array Holds the email dates.
	 * @since 1.12.0
	 */
	private $email_dates = array();

	/**
	 * Initializes the triggers for the customer win back events.
	 *
	 * @return void
	 * @since 1.12.0
	 */
	public function init() {
		add_action( 'mailmint_process_customer_win_back_daily', array( $this, 'mint_process_customer_win_back_daily' ), 10, 4 );
		add_action( 'mailmint_process_customer_win_back_daily_once', array( $this, 'mint_process_customer_win_back_daily' ), 10, 4 );

		// Retrieve the tracking data from the options table.
		$this->email_dates = get_option( $this->option_key, array() );
	}

	/**
	 * Validate the settings for the customer win back trigger.
	 * This function is used to validate the settings for the customer win back trigger.
	 *
	 * @param array $step_data The step data.
	 * @param array $data The data.
	 * @return bool
	 * @since 1.12.0
	 */
	public function validate_settings( $step_data, $data ) {
		$step_data    = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
		$trigger_name = isset( $data['trigger_name'] ) ? $data['trigger_name'] : '';
		if ( 'wc_customer_winback' === $trigger_name && $step_data['automation_id'] === $this->automation_id ) {
			$customer = isset( $data['data']['customer'] ) ? $data['data']['customer'] : array();
			$settings = isset( $step_data['settings']['wc_customer_winback_settings'] ) ? $step_data['settings']['wc_customer_winback_settings'] : array();
			return $this->validate_woocommerce_customer( $customer, $settings );
		}
	}

	/**
	 * Process the customer win back trigger.
	 * This function is used to process the customer win back trigger.
	 *
	 * @param int $automation_id The automation ID.
	 * @param int $step_id The step ID.
	 * @param int $offset The offset.
	 * @param int $per_batch The per batch.
	 * @return bool
	 * @since 1.12.0
	 */
	public function mint_process_customer_win_back_daily( $automation_id, $step_id, $offset, $per_batch ) {
		$this->automation_id = $automation_id;
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );
		$settings  = isset( $step_data['settings']['wc_customer_winback_settings'] ) ? $step_data['settings']['wc_customer_winback_settings'] : array();
		$customers = $this->get_woocommerce_customers( $settings, $offset, $per_batch );

		if ( !$customers ) {
            return false;
        }

		$start_time = time();
		$has_more   = true;
        $run        = true;

        if ( $customers && $run ) {
            foreach ( $customers as $customer ) {
				$customer_email = isset( $customer['customer_email'] ) ? $customer['customer_email'] : '';
				$data = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'wc_customer_winback',
					'data'           => array(
						'user_email' => $customer_email,
						'customer'   => $customer,
					),
				);
				do_action( MINT_TRIGGER_AUTOMATION, $data );				
            }

            if ( BackgroundProcessHelper::memory_exceeded() || time() - $start_time > 40 ) {
                $run      = false;
                $has_more = true;
            } else {
                $customers = $this->get_woocommerce_customers( $settings, $offset, $per_batch );
                if ( !$customers ) {
                    $run      = false;
                    $has_more = false;
                }
            }
        }

		// Save the updated tracking data back to the options table.
        update_option( $this->option_key, $this->email_dates );
		// Update the offset for the next batch.
		$offset += $per_batch;

		if ( $has_more ) {
            // run again after 120 seconds.
			$group = 'mailmint-process-customer-win-back-' . $automation_id;
			$args  = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'offset'        => $offset,
				'per_page'      => $per_batch,
			);
			as_schedule_single_action( time() + 120, 'mailmint_process_customer_win_back_daily_once', $args, $group );
        }
        return $has_more;
	}

	/**
	 * Get the customers for the customer win back trigger.
	 * This function is used to get the customers for the customer win back trigger.
	 *
	 * @param array $settings The settings.
	 * @param int $offset The offset.
	 * @param int $per_batch The per batch.
	 * @return array
	 * @since 1.12.0
	 */
	private function get_woocommerce_customers( $settings, $offset, $per_batch ) {
		global $wpdb;
		$customers  = array();
		// Get the current date.
        $current_date = current_time('Y-m-d');

		// Get the conditions from the automation.
		$min_days_since_purchase = isset( $settings['order_period_minimum_days'] ) ? $settings['order_period_minimum_days'] : 30;
		$max_days_since_purchase = isset( $settings['order_period_maximum_days'] ) ? $settings['order_period_maximum_days'] : 45;

		// Check if HPOS is enabled.
        if ( MrmCommon::is_hpos_enable() ) {
			// Query for HPOS enabled.
			$customers = $wpdb->get_results($wpdb->prepare("SELECT o.billing_email AS customer_email, MAX(o.date_created_gmt) AS last_purchase_date
				FROM {$wpdb->prefix}wc_orders o
				WHERE o.status IN ('wc-completed', 'wc-processing', 'wc-on-hold', 'wc-pending', 'wc-wpfnl-main-order')
				GROUP BY o.billing_email
				HAVING DATEDIFF(%s, MAX(o.date_created_gmt)) BETWEEN %d AND %d
				LIMIT %d OFFSET %d
				", $current_date, $min_days_since_purchase, $max_days_since_purchase, $per_batch, $offset), ARRAY_A);
		} else {
			// Query to get customers who have made purchases within the specified period.
			$customers = $wpdb->get_results( $wpdb->prepare( "SELECT pm.meta_value AS customer_email, MAX(p.post_date) AS last_purchase_date
				FROM {$wpdb->prefix}posts p
				JOIN {$wpdb->prefix}postmeta pm ON p.ID = pm.post_id
				WHERE p.post_type = 'shop_order'
				AND p.post_status IN ('wc-completed', 'wc-processing', 'wc-on-hold', 'wc-pending', 'wc-wpfnl-main-order')
				AND pm.meta_key = '_billing_email'
				GROUP BY pm.meta_value
				HAVING DATEDIFF(%s, MAX(p.post_date)) BETWEEN %d AND %d
				LIMIT %d OFFSET %d
				", $current_date, $min_days_since_purchase, $max_days_since_purchase, $per_batch, $offset ), ARRAY_A );
		}

		return $customers;
    }

	/**
	 * Validate the WooCommerce customer.
	 * This function is used to validate the customer.
	 * 
	 * @param array $customer The customer.
	 * @param array $attribute The attribute.
	 * @return bool
	 * @since 1.12.0
	 */
	private function validate_woocommerce_customer( $customer, $attribute ) {
		$customer_email     = $customer['customer_email'];
        $last_purchase_date = $customer['last_purchase_date'];

		// Check if the email was already sent for this purchase period.
		if ( !isset( $this->email_dates[$customer_email] ) || $this->email_dates[$customer_email] != $last_purchase_date) {
			// Update the last email date in the options table.
			$this->email_dates[$customer_email] = $last_purchase_date;
			return true;
		}
		return false;
	}
}

