<?php
/**
 * Edd triggers
 *
 * @package MintMail\App\Internal\Automation\Connector\trigger;
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use WPFunnels\Wpfnl_functions;
use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;
/**
 * Edd triggers
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class TutorTriggers {

	use Singleton;



	/**
	 * Connector name
	 *
	 * @var $connector_name
	 */
	public $connector_name = 'Tutor';

	/**
	 * Initialization of Edd hooks
	 */
	public function init() {
		add_action( 'tutor_after_student_signup', array( $this, 'mint_tutor_after_student_signup' ), 10, 1 );

		add_action( 'tutor_enrollment/after/cancel', array( $this, 'mint_tutor_cancel_enrollment' ), 10, 1 );
		add_action( 'tutor_enrollment/after/cancelled', array( $this, 'mint_tutor_cancel_enrollment' ), 10, 1 );

		add_action( 'tutor_enrollment/after/complete', array( $this, 'mint_tutor_complete_enrollment' ), 10, 1 );
		add_action( 'tutor_enrollment/after/completed', array( $this, 'mint_tutor_complete_enrollment' ), 10, 1 );

		add_action( 'tutor_enrollment/after/delete', array( $this, 'mint_tutor_delete_enrollment' ), 10, 1 );

		add_action( 'tutor_after_enrolled', array( $this, 'mint_tutor_after_enrolled' ), 10, 3 );

		add_action( 'delete_tutor_course_progress', array( $this, 'mint_delete_tutor_course_progress' ), 10, 2 );

		add_action( 'tutor/course/enrol_status_change/after', array( $this, 'mint_enrol_status_change' ), 10, 2 );

		add_action( 'tutor_after_approved_instructor', array( $this, 'mint_tutor_after_approved_instructor' ), 10, 1 );
		add_action( 'tutor_course_complete_after', array( $this, 'mint_tutor_course_complete_after' ), 10, 2 );
		add_action( 'tutor_lesson_completed_after', array( $this, 'mint_tutor_lesson_completed_after' ), 10, 2 );
	}


	/**
	 * Validate trigger settings
	 *
	 * @param array $step_data Get Step data.
	 * @param array $data Get All data.
	 * @return bool
	 */
	public function validate_settings( $step_data, $data ) {
		if ( isset( $data['trigger_name'] ) && ( 'tutor_enrol_status_change' === $data['trigger_name'] ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}
			if ( isset( $step_data['settings']['status_settings']['status'] ) && 'any' !== $step_data['settings']['status_settings']['status'] ) {
				if ( isset( $data['data']['new_status'] ) ) {
					return $data['data']['new_status'] === $step_data['settings']['status_settings']['status'];
				}
			} else {
				return true;
			}
		}

		/**
		 * Check if the trigger data corresponds to the 'tutor_after_enrolled' trigger
		 * and if the associated course ID is included in the selected courses for the automation step.
		 *
		 * This function is typically used in the context of processing triggers for Tutor LMS enrollments.
		 */
		if ( isset( $data['trigger_name'] ) && ( 'tutor_after_enrolled' === $data['trigger_name'] || 'tutor_complete_course' === $data['trigger_name'] ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}
			if ( isset( $step_data['settings']['tutor_lms_settings']['courses'] ) && !empty( $step_data['settings']['tutor_lms_settings']['courses'] ) ) {
				$selected_courses = $step_data['settings']['tutor_lms_settings']['courses'];
				$course_id        = $data['data']['course_id'];
				return in_array( $course_id, array_column( $selected_courses, 'value' ), true );
			}
			return true;
		}

		/**
		 * Check if the trigger data corresponds to the 'tutor_complete_lesson' trigger
		 * and if the associated lesson ID is included in the selected lessons for the automation step.
		 *
		 * This function is typically used in the context of processing triggers for Tutor LMS lesson completeness.
		 */
		if ( isset( $data['trigger_name'] ) && ( 'tutor_complete_lesson' === $data['trigger_name'] ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}
			if ( isset( $step_data['settings']['tutor_lms_settings']['lessons'] ) && !empty( $step_data['settings']['tutor_lms_settings']['lessons'] ) ) {
				$selected_lessons = $step_data['settings']['tutor_lms_settings']['lessons'];
				$lesson_id        = $data['data']['lesson_id'];
				return in_array( $lesson_id, array_column( $selected_lessons, 'value' ), true );
			}
			return true;
		}

		return true;
	}


	/**
	 * Tutorafter student signup
	 *
	 * @param string $user_id User ID.
	 *
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_tutor_after_student_signup( $user_id ) {
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_after_student_signup',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'new_status' => $new_status,
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Tutorafter student signup
	 *
	 * @param string $id Enrol.
	 *
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_tutor_cancel_enrollment( $id ) {
		$user_id   = get_post_field( 'post_author', $id );
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_cancel_enrollment',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Tutor complete enrollment.
	 *
	 * @param string $id Enrol.
	 *
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_tutor_complete_enrollment( $id ) {
		$user_id   = get_post_field( 'post_author', $id );
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_complete_enrollment',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Tutor delete enrollment.
	 *
	 * @param string $id Enrol id.
	 *
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_tutor_delete_enrollment( $id ) {
		$user_id   = get_post_field( 'post_author', $id );
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_delete_enrollment',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Tutor after student signup.
	 *
	 * @param string $course_id Course ID.
	 * @param string $user_id User ID.
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_delete_tutor_course_progress( $course_id, $user_id ) {
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'delete_tutor_course_progress',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Tutor after enrolled.
	 *
	 * @param string $course_id course id.
	 * @param string $user_id course id.
	 * @param bool   $is_enrolled enrollment id.
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_tutor_after_enrolled( $course_id, $user_id, $is_enrolled ) {
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_after_enrolled',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'course_id'  => $course_id,
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Tutor after enrolled.
	 *
	 * @param string $enrol_id Enroll id.
	 * @param string $new_status Status.
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_enrol_status_change( $enrol_id, $new_status ) {
		$user_id   = get_post_field( 'post_author', $enrol_id );
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_enrol_status_change',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'new_status' => $new_status,
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Tutor after approved instructor.
	 *
	 * @param string $instructor_id instructor id.
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_tutor_after_approved_instructor( $instructor_id ) {
		$user_data = get_userdata( $instructor_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_after_approved_instructor',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Trigger automation after a Tutor LMS course is completed.
	 *
	 * This function is hooked into the 'tutor_course_complete_after' action,
	 * which is fired after a Tutor LMS course is marked as completed for a user.
	 *
	 * @param int $course_id The ID of the completed Tutor LMS course.
	 * @param int $user_id   The ID of the user who completed the Tutor LMS course.
	 *
	 * @return void
	 *
	 * @since 1.8.1
	 */
	public function mint_tutor_course_complete_after( $course_id, $user_id ) {
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_complete_course',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'course_id'  => $course_id,
				),
			);

			/** This action is documented in app/Internal/Automation/Connectors/MemberPress/Triggers/MemberPressTriggers.php */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Trigger automation after a Tutor LMS lesson is completed.
	 *
	 * This function is hooked into the 'tutor_lesson_completed_after' action,
	 * which is fired after a Tutor LMS lesson is marked as completed for a user.
	 *
	 * @param int $lesson_id The ID of the completed Tutor LMS lesson.
	 * @param int $user_id   The ID of the user who completed the Tutor LMS lesson.
	 *
	 * @return void
	 *
	 * @since 1.8.1
	 */
	public function mint_tutor_lesson_completed_after( $lesson_id, $user_id ) {
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'tutor_complete_lesson',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'lesson_id'  => $lesson_id,
				),
			);

			/** This action is documented in app/Internal/Automation/Connectors/MemberPress/Triggers/MemberPressTriggers.php */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}
}

