<?php
/**
 * MemberPress triggers
 *
 * @package MintMail\App\Internal\Automation\Connector\Trigger;
 */

namespace MintMail\App\Internal\Automation\Connector\Trigger;

use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;

/**
 * MemberPress triggers
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class MemberPressTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var $connector_name
	 * @since 1.8.0
	 */
	public $connector_name = 'MemberPress';

	/**
	 * Initialization of MemberPress hooks
	 */
	public function init() {
		add_action( 'mepr-account-is-active', array( $this, 'mint_after_added_to_a_membership' ), 12, 1 );
		add_action( 'mepr-event-transaction-expired', array( $this, 'mint_after_transaction_expired' ), 20, 1 );
	}

	/**
	 * Validate settings based on LearnDash course conditions.
	 *
	 * @param array $step_data The data related to the automation step.
	 * @param array $data      The data associated with the trigger.
	 *
	 * @return bool True if the settings are valid, false otherwise.
	 * @since 1.8.0
	 */
	public function validate_settings( $step_data, $data ) {
		$allowed_trigger_names = array( 'memberpress_member_added', 'memberpress_subscription_expired' );
		if ( isset( $data['trigger_name'] ) && in_array( $data['trigger_name'], $allowed_trigger_names, true ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}

			if ( isset( $step_data['settings']['member_press_settings']['levels'] ) && !empty( $step_data['settings']['member_press_settings']['levels'] ) ) {
				$selected_levels = $step_data['settings']['member_press_settings']['levels'];
				$product_id      = $data['data']['product_id'];
				return in_array( $product_id, array_column( $selected_levels, 'value' ) );
			}
			return true;
		}

		return true;
	}

	/**
	 * Handle automation trigger after a user is added to a MemberPress membership.
	 *
	 * This function is triggered when a user is added to a MemberPress membership,
	 * and it prepares the necessary data to trigger an automation process for the
	 * 'memberpress_member_added' event.
	 *
	 * @param MeprTransaction $txn The MemberPress transaction object associated with the membership.
	 * @return void
	 * @since 1.8.0
	 */
	public function mint_after_added_to_a_membership( $txn ) {
		$product_id = $txn->product_id;
		$user_id    = $txn->user_id;

		if ( !$product_id || !$user_id ) {
			return;
		}

		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'memberpress_member_added',
				'data'           => array(
					'user_email' => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'product_id' => $product_id,
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.8.0
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Handle automation trigger after a MemberPress subscription transaction expires.
	 *
	 * This function is triggered when a MemberPress subscription transaction expires,
	 * and it prepares the necessary data to trigger an automation process for the
	 * 'memberpress_subscription_expired' event.
	 *
	 * @param MeprSubscriptionTransaction $event The MemberPress subscription transaction event.
	 * @return void
	 * @since 1.8.0
	 */
	public function mint_after_transaction_expired( $event ) {
		$txn          = $event->get_data();
		$subscription = $txn->subscription();

		if ( strtotime( $txn->expires_at ) <= time() && ( empty( $subscription ) || $subscription->is_expired() ) ) {
			$product_id = $txn->product_id;
			$user_id    = $txn->user_id;

			if ( !$product_id || !$user_id ) {
				return;
			}

			$user_data = get_userdata( $user_id );
			if ( $user_data ) {
				$data = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'memberpress_subscription_expired',
					'data'           => array(
						'user_email' => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
						'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
						'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
						'product_id' => $product_id,
					),
				);

				/**
				 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
				 *
				 * This action triggers automation processes with the given data for handling automation events.
				 *
				 * @param mixed $data The data associated with the automation trigger.
				 * @since 1.8.0
				 */
				do_action( MINT_TRIGGER_AUTOMATION, $data );
			}
		}
	}
}

