<?php
/**
 * Webhook triggers
 * Description: This file is used to run triggers for Webhook.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use Mint\MRM\DataBase\Models\ContactModel;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Class WebhookTriggers
 * Description: This class is used to run triggers for Webhook.
 *
 * @since 1.14.0
 */
class WebhookTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var string Holds the name of the connector.
	 * @since 1.14.0
	 */
	public $connector_name = 'Webhook';

	/**
	 * Automation ID for the currently running automation.
	 *
	 * @var int Holds the automation ID.
	 * @since 1.14.0
	 */
	private $automation_id;

	/**
	 * Initializes the triggers for the anniversary events.
	 *
	 * @return void
	 * @since 1.14.0
	 */
	public function init() {
		add_action( 'mint_after_incoming_webhook', array( $this, 'mint_process_after_incoming_webhook' ), 10, 2 );
	}

	/**
	 * Validates the settings for the 'mint_webhook_received' trigger.
	 *
	 * This function checks if the provided step data and webhook data are valid for the 'mint_webhook_received' trigger.
	 *
	 * @param array $step_data The step data containing automation and step IDs.
	 * @param array $data The data containing the trigger name and webhook ID.
	 * @return bool True if the settings are valid, false otherwise.
	 * 
	 * @since 1.14.0
	 */
	public function validate_settings( $step_data, $data ) {
		if ( !isset( $data['trigger_name'] ) || 'mint_webhook_received' !== $data['trigger_name'] ) {
			return false;
		}
	
		if ( !isset( $step_data['automation_id'], $step_data['step_id'] ) ) {
			return false;
		}
	
		$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
	
		$webhook_settings = isset( $step_data['settings']['webhook_settings']['webhook'] ) ? $step_data['settings']['webhook_settings']['webhook'] : array();
		if ( empty( $webhook_settings ) ) {
			return false;
		}
	
		$webhook_id = isset( $webhook_settings['value'] ) ? $webhook_settings['value'] : '';
		if ( empty( $webhook_id ) ) {
			return false;
		}
	
		return isset( $data['data']['webhook_id'] ) && $data['data']['webhook_id'] === $webhook_id;
	}

	/**
	 * Processes actions after an incoming webhook is handled in Mail Mint.
	 *
	 * This function is triggered after the incoming webhook is processed. It retrieves the necessary details from the webhook
	 * and contact, then triggers the automation action hook with the relevant data.
	 *
	 * @param array $webhook The webhook data, including settings and metadata.
	 * @param array $contact The contact data created or updated from the webhook.
	 * 
	 * @since 1.14.0
	 */
	public function mint_process_after_incoming_webhook( $webhook, $contact ) {
		$webhook_id = isset( $webhook['id'] ) ? $webhook['id'] : '';
		$contact_id = isset( $contact['contact_id'] ) ? $contact['contact_id'] : '';
		$email      = ContactModel::get_contact_email_by_id( $contact_id );

		$data = array(
			'connector_name' => $this->connector_name,
			'trigger_name'   => 'mint_webhook_received',
			'data'           => array(
				'user_email' => $email,
				'webhook_id' => $webhook_id,
			),
		);
		do_action( MINT_TRIGGER_AUTOMATION, $data );
	}
}

