<?php

/**
 * Segment triggers
 * Description: This file is used to run triggers for Segment.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use MailMintPro\Mint\Internal\Admin\Segmentation\FilterSegmentContacts;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Class SegmentTriggers
 * Description: This class is used to run triggers for Segment.
 *
 * @since 1.15.4
 */
class SegmentTriggers{

    use Singleton;

    /**
     * Connector name
     *
     * @var string Holds the name of the connector.
     * @since 1.15.4
     */
    public $connector_name = 'Segment';

    /**
     * Automation ID for the currently running automation.
     *
     * @var int Holds the automation ID.
     * @since 1.15.4
     */
    private $automation_id;

    /**
     * Initializes the triggers for the segment events.
     *
     * @return void
     * @since 1.15.4
     */
    public function init(){
        add_action('mint_after_contact_creation', array($this, 'mint_process_after_contact_creation'));
    }

    /**
     * Validates the settings for the segment trigger.
     *
     * This method validates the settings for the segment trigger by checking if the trigger name is 'mint_add_to_segment'
     * and if the segment settings are set.
     *
     * @param array $step_data The data of the step.
     * @param array $data The data of the trigger.
     * @return bool True if the settings are valid, false otherwise.
     * @since 1.15.4
     */
    public function validate_settings($step_data, $data){
        if (!isset($data['trigger_name']) || 'mint_add_to_segment' !== $data['trigger_name']) {
            return false;
        }

        if (!isset($step_data['automation_id'], $step_data['step_id'])) {
            return false;
        }

        $step_data = HelperFunctions::get_step_data($step_data['automation_id'], $step_data['step_id']);

        $segment_settings = isset($step_data['settings']['segment_settings']['segment']) ? $step_data['settings']['segment_settings']['segment'] : array();
        if (empty($segment_settings)) {
            return false;
        }

        $segment_id = isset($segment_settings['value']) ? $segment_settings['value'] : '';
        if (empty($segment_id)) {
            return false;
        }

        $user_email       = isset($data['data']['user_email']) ? $data['data']['user_email'] : '';
        $segment_data     = FilterSegmentContacts::get_segment($segment_id);
        $segment_contacts = isset($segment_data['contacts']['data']) ? $segment_data['contacts']['data'] : array();
        
        if (empty($segment_contacts)) {
            return false;
        }

        return $this->contact_exist_on_segment($user_email, $segment_contacts);
    }

    /**
     * Processes actions after a contact is created.
     *
     * This method retrieves the email of the contact by their ID and triggers the MINT_TRIGGER_AUTOMATION action
     * with the relevant data.
     *
     * @param int $contact_id The ID of the newly created contact.
     * @since 1.15.4
     */
    public function mint_process_after_contact_creation($contact_id){
        if(!$contact_id){
            return;
        }

        $email = ContactModel::get_contact_email_by_id($contact_id);
        $data  = array(
            'connector_name' => $this->connector_name,
            'trigger_name'   => 'mint_add_to_segment',
            'data'           => array(
                'user_email' => $email,
                'contact_id' => $contact_id,
            ),
        );
        do_action(MINT_TRIGGER_AUTOMATION, $data);
    }

    /**
     * Checks if the contact exists in the segment.
     *
     * This method checks if the contact exists in the segment by comparing the email of the contact with the emails
     * of the contacts in the segment.
     *
     * @param string $email The email of the contact.
     * @param array $segment_contacts The contacts in the segment.
     * @return bool True if the contact exists in the segment, false otherwise.
     * @since 1.15.4
     */
    private function contact_exist_on_segment($email, $segment_contacts){
        // Extract the email column from the segment_contacts.
        $emails = array_column($segment_contacts, 'email');

        // Check if the email exists in the extracted column.
        return in_array($email, $emails);
    }
}
