<?php

/**
 * Segment triggers
 * Description: This file is used to run triggers for Segment.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use Mint\MRM\DataBase\Models\EmailModel;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataBase\Models\CampaignModel;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Class SegmentTriggers
 * Description: This class is used to run triggers for Segment.
 *
 * @since 1.18.4
 */
class LinkClickTriggers{

    use Singleton;

    /**
     * Connector name
     *
     * @var string Holds the name of the connector.
     * @since 1.18.4
     */
    public $connector_name = 'LinkClick';

    /**
     * Automation ID for the currently running automation.
     *
     * @var int Holds the automation ID.
     * @since 1.18.4
     */
    private $automation_id;

    /**
     * Initializes the triggers for the segment events.
     *
     * @return void
     * @since 1.18.0
     */
    public function init(){
        add_action('mailmint_after_email_click', array($this, 'mint_process_after_contact_click'), 10, 2);
    }

    /**
     * Validates the settings for the segment trigger.
     *
     * This method validates the settings for the segment trigger by checking if the trigger name is 'mint_add_to_segment'
     * and if the segment settings are set.
     *
     * @param array $step_data The data of the step.
     * @param array $data The data of the trigger.
     * @return bool True if the settings are valid, false otherwise.
     * @since 1.18.0
     */
    public function validate_settings($step_data, $data){
        if (!isset($data['trigger_name']) || 'mint_clicks_a_link' !== $data['trigger_name']) {
            return false;
        }

        if (!isset($step_data['automation_id'], $step_data['step_id'])) {
            return false;
        }

        $automation_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
        if (empty($automation_data)) {
            return false;
        }

        $reactivate = isset($automation_data['settings']['link_click_settings']['reactivate'])? $automation_data['settings']['link_click_settings']['reactivate']: false;
        if (!$reactivate) {
            return false;
        }

        $urls        = isset($automation_data['settings']['link_click_settings']['urls'])? $automation_data['settings']['link_click_settings']['urls']: [];
        $target_url  = isset($data['data']['target_url'])? $data['data']['target_url']: '';
        $linked_urls = array_map( function( $url ) { return $url['value']; }, $urls );

        if ( in_array( $target_url, $linked_urls ) ) {
            return true;
        }else{
            return false;
        }

    }

    /**
     * Processes actions after a contact is created.
     *
     * This method retrieves the email of the contact by their ID and triggers the MINT_TRIGGER_AUTOMATION action
     * with the relevant data.
     *
     * @param int $contact_id The ID of the newly created contact.
     * @since 1.18.0
     */
    public function mint_process_after_contact_click( $email_id, $target_url ){
        global $wpdb;
    
        if (!$email_id) {
            return false;
        }    

        $query  = $wpdb->prepare( "SELECT email_address FROM {$wpdb->prefix}mint_broadcast_emails WHERE id = %d", $email_id );
        $result = $wpdb->get_row($query);    
        if (!$result) {
            return false;
        }

        $email = $result->email_address;
        $data  = array(
            'connector_name' => $this->connector_name,
            'trigger_name'   => 'mint_clicks_a_link',
            'data'           => array(
                'user_email'  => $email,
                'target_url'  => $target_url
            ),
        );

        do_action(MINT_TRIGGER_AUTOMATION, $data);
    }
}
