<?php
/**
 * Anniversary triggers
 * Description: This file is used to run triggers for anniversary.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use MailMintPro\App\Utilities\Helper\MintWC;
use MailMintPro\Mint\Internal\Admin\Segmentation\FilterSegmentContacts;
use Mint\App\Internal\Cron\BackgroundProcessHelper;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Class AnniversaryTriggers
 * Description: This class is used to run triggers for anniversary.
 *
 * @since 1.12.0
 */
class AnniversaryTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var string Holds the name of the connector.
	 * @since 1.12.0
	 */
	public $connector_name = 'MailMint';

	/**
	 * Automation ID for the currently running automation.
	 *
	 * @var int Holds the automation ID.
	 * @since 1.12.0
	 */
	private $automation_id;

	/**
	 * Initializes the triggers for the anniversary events.
	 *
	 * @return void
	 * @since 1.12.0
	 */
	public function init() {
		add_action( 'mailmint_process_contact_anniversary_daily', array( $this, 'mint_process_contact_anniversary_daily' ), 10, 4 );
		add_action( 'mailmint_process_contact_anniversary_daily_once', array( $this, 'mint_process_contact_anniversary_daily' ), 10, 4 );
	}

	/**
	 * Validate the settings for the anniversary trigger.
	 * This function is used to validate the settings for the anniversary trigger.
	 *
	 * @param array $step_data The step data.
	 * @param array $data The data.
	 * @return bool
	 * @since 1.12.0
	 */
	public function validate_settings( $step_data, $data ) {
		$step_data    = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
		$trigger_name = isset( $data['trigger_name'] ) ? $data['trigger_name'] : '';

		if ('mint_anniversary_reminder' === $trigger_name && $step_data['automation_id'] === $this->automation_id) {
			$contact   = isset($data['data']['contact']) ? $data['data']['contact'] : array();
			$attribute = isset($step_data['settings']['anniversary']['attribute']) ? $step_data['settings']['anniversary']['attribute'] : array();

			if (!empty($attribute['param'])) {
				switch ($attribute['param']) {
					case 'date_of_birth':
						return $this->validate_contact_anniversary($contact, $attribute);
					case 'contact_signup_date':
						return $this->validate_contact_signup_anniversary($contact, $attribute);
					default:
						return false;
				}
			} else {
				return false;
			}
		}
	}

	/**
	 * Process the anniversary trigger.
	 * This function is used to process the anniversary trigger.
	 *
	 * @param int $automation_id The automation ID.
	 * @param int $step_id The step ID.
	 * @param int $offset The offset.
	 * @param int $per_batch The per batch.
	 * @return bool
	 * @since 1.12.0
	 */
	public function mint_process_contact_anniversary_daily( $automation_id, $step_id, $offset, $per_batch ) {
		$this->automation_id = $automation_id;

		// Get the step data.
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );
		$settings  = isset( $step_data['settings']['anniversary'] ) ? $step_data['settings']['anniversary'] : array();
		$contacts  = $this->get_anniversary_contacts( $settings, $offset, $per_batch );

		if ( !$contacts ) {
			return false;
		}

		$start_time = time();
		$has_more   = true;
		$run        = true;
		if ( $contacts && $run ) {
			foreach ( $contacts as $contact ) {
				$email = isset( $contact['email'] ) ? $contact['email'] : '';
				$data  = array(
					'connector_name' => 'Anniversary',
					'trigger_name'   => 'mint_anniversary_reminder',
					'data'           => array(
						'user_email' => $email,
						'contact'    => $contact,
					),
				);
				do_action( MINT_TRIGGER_AUTOMATION, $data );
			}

			if ( BackgroundProcessHelper::memory_exceeded() || time() - $start_time > 40 ) {
				$run      = false;
				$has_more = true;
			} else {
				$contacts = $this->get_anniversary_contacts( $settings, $offset, $per_batch );
				if ( !$contacts ) {
					$run      = false;
					$has_more = false;
				}
			}
		}

		// Update the offset for the next batch.
		$offset += $per_batch;

		if ( $has_more ) {
			// run again after 120 seconds.
			$group = 'mailmint-process-contact-anniversary-' . $automation_id;
			$args  = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'offset'        => $offset,
				'per_page'      => $per_batch,
			);
			as_schedule_single_action( time() + 120, 'mailmint_process_contact_anniversary_daily_once', $args, $group );
		}
		return $has_more;
	}

	/**
	 * Get the contacts for the anniversary trigger.
	 * This function is used to get the contacts for the anniversary trigger.
	 *
	 * @param array $settings The settings.
	 * @param int   $offset The offset.
	 * @param int   $per_batch The per batch.
	 * @return array
	 * @since 1.12.0
	 */
	private function get_anniversary_contacts( $settings, $offset, $per_batch ) {
		$contacts = array();

		if ( !empty( $settings[ 'segments' ] ) ) {
			$segment_ids = array_column( $settings[ 'segments' ], 'id' );
			if ( class_exists( 'MailMintPro\Mint\Internal\Admin\Segmentation\FilterSegmentContacts' ) ) {
				foreach ( $segment_ids as $segment_id ) {
					$segment_data = FilterSegmentContacts::get_segment( $segment_id, 'id, email, status', $offset, $per_batch );

					// Process only if there are contacts in the batch.
					if ( !empty( $segment_data['contacts']['data'] ) ) {
						$filtered_contacts = array_filter(
							$segment_data['contacts']['data'],
							function ( $contact ) {
								return 'subscribed' === $contact['status'];
							}
						);

						$contacts = array_merge( $contacts, $filtered_contacts );
					}
				}
			}
		}

		$lists    = isset($settings['lists']) ? $settings['lists'] : array();
		$tags     = isset($settings['tags']) ? $settings['tags'] : array();
		$groups   = array_merge($lists, $tags);
		$contacts = MintWC::retrieve_contacts_by_group($groups, $offset, $per_batch, $contacts);

		return $contacts;
	}

	/**
	 * Validate the contact anniversary.
	 * This function is used to validate the contact anniversary.
	 *
	 * @param array $contact The contact.
	 * @param array $attribute The attribute.
	 * @return bool
	 * @since 1.12.0
	 */
	private function validate_contact_anniversary( $contact, $attribute ) {
		global $wpdb;
		$param        = isset( $attribute['param'] ) ? $attribute['param'] : 'date_of_birth';
		$condition    = isset( $attribute['condition_value'] ) ? $attribute['condition_value'] : 'is_today';
		$value        = isset( $attribute['value'] ) ? $attribute['value'] : 0;
		$contact_id   = isset( $contact['id'] ) ? $contact['id'] : 0;
		$compare_date = gmdate( 'Y-m-d' );

		switch ( $condition ) {
			case 'is_today':
				$compare_date = gmdate( 'Y-m-d' );
				break;
			case 'is_tomorrow':
				$compare_date = gmdate( 'Y-m-d', strtotime( '+1 day' ) );
				break;
			case 'is_in_x_days':
				$compare_date = gmdate( 'Y-m-d', strtotime( "+{$value} days" ) );
				break;
			default:
				return false;
		}

		$row = $wpdb->get_row( // phpcs:ignore
			$wpdb->prepare(
				"SELECT * FROM {$wpdb->prefix}mint_contact_meta WHERE contact_id = %d AND meta_key = %s AND MONTH(meta_value) = MONTH(%s) AND DAY(meta_value) = DAY(%s)",
				$contact_id,
				$param,
				$compare_date,
				$compare_date
			),
			ARRAY_A
		);

		if ( !$row ) {
			return false;
		}

		return true;
	}

	/**
	 * Validate the contact signup anniversary.
	 * This function is used to validate the contact signup anniversary.
	 *
	 * @param array $contact The contact.
	 * @param array $attribute The attribute.
	 * @return bool
	 * @since 1.12.0
	 */
	private function validate_contact_signup_anniversary($contact, $attribute){
		global $wpdb;
		$condition    = isset($attribute['condition_value']) ? $attribute['condition_value'] : 'is_today';
		$value        = isset($attribute['value']) ? $attribute['value'] : 0;
		$contact_id   = isset($contact['id']) ? $contact['id'] : 0;
		$compare_date = gmdate('Y-m-d');
		switch ($condition) {
			case 'is_today':
				$compare_date = gmdate('Y-m-d');
				break;
			case 'is_tomorrow':
				$compare_date = gmdate('Y-m-d', strtotime('+1 day'));
				break;
			case 'is_in_x_days':
				$compare_date = gmdate('Y-m-d', strtotime("+{$value} days"));
				break;
			default:
				return false;
		}

		$query = $wpdb->prepare(
			"SELECT * FROM {$wpdb->prefix}mint_contacts WHERE id = %d  AND MONTH(created_at) = MONTH(%s) AND DAY(created_at) = DAY(%s)",
			$contact_id,
			$compare_date,
			$compare_date
		);

		$row = $wpdb->get_row($query, ARRAY_A);
		if (!$row) {
			return false;
		}

		return true;
	}
}

