<?php
/**
 * LifterLMS triggers
 * Description: This file is used to run triggers for LifterLMS.
 *
 * @package MintMail\App\Internal\Automation\Connector
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use MailMintPro\App\Utilities\Helper\MintWC;
use MailMintPro\Mint\Internal\Admin\Segmentation\FilterSegmentContacts;
use Mint\App\Internal\Cron\BackgroundProcessHelper;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Class LifterLMSTriggers
 * Description: This class is used to run triggers for LifterLMS.
 *
 * @since 1.12.0
 */
class LifterLMSTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var string Holds the name of the connector.
	 * @since 1.12.0
	 */
	public $connector_name = 'LifterLMS';

	/**
	 * Initializes the triggers for the LifterLMS events.
	 *
	 * @return void
	 * @since 1.12.0
	 */
	public function init() {
		add_action( 'llms_user_enrolled_in_course', array( $this, 'mint_llms_user_enrolled_in_course' ), PHP_INT_MAX, 2 );
		add_action( 'llms_user_added_to_membership_level', array( $this, 'mint_llms_user_added_to_membership_level' ), PHP_INT_MAX, 2 );
	}

	/**
	 * Validate the settings for the LifterLMS trigger.
	 * This function is used to validate the settings for the LifterLMS trigger.
	 *
	 * @param array $step_data The step data.
	 * @param array $data The data.
	 * @return bool
	 * @since 1.12.0
	 */
	public function validate_settings( $step_data, $data ) {
		$allowed_trigger_names = array( 'lifterlms_enrolled_course', 'lifterlms_enrolled_membership' );
		if ( isset( $data['trigger_name'] ) && in_array( $data['trigger_name'], $allowed_trigger_names, true ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}

			if ( isset( $step_data['settings']['lifter_settings']['courses'] ) && !empty( $step_data['settings']['lifter_settings']['courses'] ) ) {
				$selected_courses = $step_data['settings']['lifter_settings']['courses'];
				$course_id        = $data['data']['course_id'];
				return in_array( $course_id, array_column( $selected_courses, 'value' ) );
			}
			return true;
		}
		return true;
	}

	/**
	 * Process the llms_user_enrolled_in_course trigger.
	 * This function is used to process when a student is enrolled in a course.
	 *
	 * @param int $student_id The WP User ID.
	 * @param int $course_id The WP Post ID of the Course.
	 * @return bool
	 * @since 1.12.0
	 */
	public function mint_llms_user_enrolled_in_course( $student_id, $course_id ) {
		$user_data = get_userdata( $student_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'lifterlms_enrolled_course',
				'data'           => array(
					'user_email' => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'course_id'  => $course_id,
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.6.1
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Process the llms_user_added_to_membership_level trigger.
	 * This function is used to process when a student is enrolled in a membership lavel.
	 *
	 * @param int $student_id The WP User ID.
	 * @param int $course_id The WP Post ID of the membership.
	 * @return bool
	 * @since 1.12.0
	 */
	public function mint_llms_user_added_to_membership_level( $student_id, $course_id ) {
		$user_data = get_userdata( $student_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'lifterlms_enrolled_membership',
				'data'           => array(
					'user_email' => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'course_id'  => $course_id,
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.6.1
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}
}

