<?php
/**
 * LearnDash triggers
 *
 * @package MintMail\App\Internal\Automation\Connector\Trigger;
 */

namespace MintMail\App\Internal\Automation\Connector\Trigger;

use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;
use WP_Post;

/**
 * LearnDash triggers
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class LearnDashTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var $connector_name
	 * @since 1.6.1
	 */
	public $connector_name = 'LearnDash';

	/**
	 * Initialization of LearnDash hooks
	 */
	public function init() {
		add_action( 'learndash_update_course_access', array( $this, 'mint_after_enrolls_in_course' ), 10, 4 );
		add_action( 'learndash_course_completed', array( $this, 'mint_after_completes_a_course' ) );
		add_action( 'learndash_lesson_completed', array( $this, 'mint_after_completes_a_lesson' ), 9, 1 );
		add_action( 'learndash_quiz_completed', array( $this, 'mint_after_completes_a_quiz' ), 9, 2 );
		add_action( 'ld_added_group_access', array( $this, 'mint_after_user_added_to_a_group' ), 9, 2 );
		add_action( 'learndash_topic_completed', array( $this, 'mint_after_completes_a_topic' ), 9, 1 );
	}

	/**
	 * Validate settings based on LearnDash course conditions.
	 *
	 * @param array $step_data The data related to the automation step.
	 * @param array $data      The data associated with the trigger.
	 *
	 * @return bool True if the settings are valid, false otherwise.
	 * @since 1.6.1
	 */
	public function validate_settings( $step_data, $data ) {
		$allowed_trigger_names = array( 'learndash_enrolled_course', 'learndash_complete_course' );
		if ( isset( $data['trigger_name'] ) && in_array( $data['trigger_name'], $allowed_trigger_names, true ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}

			if ( isset( $step_data['settings']['learn_dash_settings']['courses'] ) && !empty( $step_data['settings']['learn_dash_settings']['courses'] ) ) {
				$selected_courses = $step_data['settings']['learn_dash_settings']['courses'];
				$course_id        = $data['data']['ld_course_id'];
				return in_array( $course_id, array_column( $selected_courses, 'value' ) );
			}
			return true;
		}

		if ( isset( $data['trigger_name'] ) && ( 'learndash_complete_lesson' === $data['trigger_name'] ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}

			if ( isset( $step_data['settings']['learn_dash_settings']['lessons'] ) && !empty( $step_data['settings']['learn_dash_settings']['lessons'] ) ) {
				$selected_lessons = $step_data['settings']['learn_dash_settings']['lessons'];
				$lesson_id        = $data['data']['ld_lesson_id'];
				return in_array( $lesson_id, array_column( $selected_lessons, 'value' ) );
			}
			return true;
		}

		if ( isset( $data['trigger_name'] ) && ( 'learndash_completes_quiz' === $data['trigger_name'] ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}

			if ( isset( $step_data['settings']['learn_dash_settings']['quizzes'] ) && !empty( $step_data['settings']['learn_dash_settings']['quizzes'] ) ) {
				$selected_quizzes = $step_data['settings']['learn_dash_settings']['quizzes'];
				$quiz_id          = $data['data']['ld_quiz_id'];
				return in_array( $quiz_id, array_column( $selected_quizzes, 'value' ) );
			}
			return true;
		}

		if ( isset( $data['trigger_name'] ) && ( 'learndash_enrolls_groups' === $data['trigger_name'] ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}

			if ( isset( $step_data['settings']['learn_dash_settings']['groups'] ) && !empty( $step_data['settings']['learn_dash_settings']['groups'] ) ) {
				$selected_groups = $step_data['settings']['learn_dash_settings']['groups'];
				$group_id        = $data['data']['ld_group_id'];
				return in_array( $group_id, array_column( $selected_groups, 'value' ) );
			}
			return true;
		}

		if ( isset( $data['trigger_name'] ) && ( 'learndash_complete_topic' === $data['trigger_name'] ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}

			if ( isset( $step_data['settings']['learn_dash_settings']['topics'] ) && !empty( $step_data['settings']['learn_dash_settings']['topics'] ) ) {
				$selected_topics = $step_data['settings']['learn_dash_settings']['topics'];
				$topic_id        = $data['data']['ld_topic_id'];
				return in_array( $topic_id, array_column( $selected_topics, 'value' ) );
			}
			return true;
		}

		return true;
	}

	/**
	 * Trigger an action after a user enrolls in a course.
	 *
	 * @param int   $user_id           The ID of the user enrolling in the course.
	 * @param int   $course_id         The ID of the course being enrolled in.
	 * @param array $course_access_list List of course access details (not used in this function).
	 * @param bool  $remove            Whether to remove the user (not used in this function).
	 *
	 * @since 1.6.1
	 */
	public function mint_after_enrolls_in_course( $user_id, $course_id, $course_access_list, $remove ) {
		if ( false !== $remove ) {
			return;
		}

		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'learndash_enrolled_course',
				'data'           => array(
					'user_email'   => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name'   => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'    => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'ld_course_id' => $course_id,
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.6.1
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Trigger an action after a user completes a course in LearnDash.
	 *
	 * @param array $course An array containing information about the completed course, including user and course data.
	 * @since 1.6.1
	 */
	public function mint_after_completes_a_course( $course ) {
		$user_data = get_userdata( $course['user']->ID );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'learndash_complete_course',
				'data'           => array(
					'user_email'   => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name'   => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'    => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'ld_course_id' => $course['course']->ID,
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.6.1
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Trigger an action after a user completes a lesson in LearnDash.
	 *
	 * @param array $lesson An array containing information about the completed lesson, including user, course, and lesson data.
	 * @since 1.6.1
	 */
	public function mint_after_completes_a_lesson( $lesson ) {
		$user_data = get_userdata( $lesson['user']->ID );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'learndash_complete_lesson',
				'data'           => array(
					'user_email'   => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name'   => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'    => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'ld_course_id' => $lesson['course']->ID,
					'ld_lesson_id' => $lesson['lesson']->ID,
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.6.1
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Trigger an action after a user completes a quiz in LearnDash.
	 *
	 * @param array $quiz_data An array containing information about the completed quiz, including user and quiz data.
	 * @param WP_Post $current_user The user who completed the quiz.
	 * @since 1.6.1
	 */
	public function mint_after_completes_a_quiz( $quiz_data, $current_user ) {
		$user_data = get_userdata( $current_user->ID );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'learndash_completes_quiz',
				'data'           => array(
					'user_email'   => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name'   => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'    => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'ld_quiz_id'   => isset( $quiz_data['quiz'] ) ? $quiz_data['quiz'] : 0,
					'ld_quiz_data' => $quiz_data,
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.6.1
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Trigger an action after a user is added to a group in LearnDash.
	 *
	 * @param int $user_id The ID of the user being added to the group.
	 * @param int $group_id The ID of the group the user is being added to.
	 * @since 1.6.1
	 */
	public function mint_after_user_added_to_a_group( $user_id, $group_id ) {
		$user_data = get_userdata( $user_id );
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'learndash_enrolls_groups',
				'data'           => array(
					'user_email'  => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name'  => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'   => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'ld_group_id' => $group_id,
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.6.1
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Trigger an action after a user completes a topic in LearnDash.
	 *
	 * @param array $topic_data An array of topic complete data.
	 * @since 1.11.0
	 */
	public function mint_after_completes_a_topic( $topic_data ) {
		$user_data = get_userdata( $topic_data['user']->ID );

		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'learndash_complete_topic',
				'data'           => array(
					'user_email'   => isset( $user_data->data->user_email ) ? $user_data->data->user_email : '',
					'first_name'   => isset( $user_data->first_name ) ? $user_data->first_name : '',
					'last_name'    => isset( $user_data->last_name ) ? $user_data->last_name : '',
					'ld_course_id' => $topic_data['course']->ID,
					'ld_lesson_id' => $topic_data['lesson']->ID,
					'ld_topic_id'  => $topic_data['topic']->ID
				),
			);

			/**
			 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
			 *
			 * This action triggers automation processes with the given data for handling automation events.
			 *
			 * @param mixed $data The data associated with the automation trigger.
			 * @since 1.6.1
			 */
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}
}

