<?php

/**
 * Edd triggers
 *
 * @package MintMail\App\Internal\Automation\Connector\trigger;
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use Mint\MRM\Admin\API\Controllers\MessageController;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataStores\ContactData;
use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;

/**
 * Edd triggers
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class GformTriggers
{

	use Singleton;



	/**
	 * Connector name
	 *
	 * @var $connector_name
	 */
	public $connector_name = 'Gform';

	/**
	 * Initialization of Gform hooks
	 */
	public function init()
	{
		add_action('gform_after_submission', array($this, 'mint_gform_after_submission'), 10, 2);
		add_action('gform_after_email', array($this, 'mint_gform_after_email'), 10, 13);
		add_action('gform_send_email_failed', array($this, 'mint_gform_send_email_failed'), 10, 3);
	}


	/**
	 * Validate trigger settings
	 *
	 * @param array $step_data Get Step data.
	 * @param array $data Get All data.
	 * @return bool
	 */
	public function validate_settings($step_data, $data)
	{
		if (isset($data['trigger_name']) && (('gform_after_submission' === $data['trigger_name']) || 'gform_after_email' === $data['trigger_name'])) {
			if (isset($step_data['automation_id'], $step_data['automation_id'])) {
				$step_data = HelperFunctions::get_step_data($step_data['automation_id'], $step_data['step_id']);
			}
			if (isset($step_data['settings']['gform_settings']['form_id']) && 'any' !== $step_data['settings']['gform_settings']['form_id']) {
				if (isset($data['data']['form_id'])) {
					// Check if the form_id is an array or not.
					$form_id          = $step_data['settings']['gform_settings']['form_id'];
					$settings_form_id = is_array($form_id) ? $form_id['value'] : $form_id;

					$is_validate = ($data['data']['form_id'] == $settings_form_id); //phpcs:ignore
					if ($is_validate) {
						$form_data = isset($data['data']) ? $data['data'] : array();
						$status    = isset($step_data['settings']['gform_settings']['status']) ? $step_data['settings']['gform_settings']['status'] : 'pending';
						$email     = isset($data['data']['user_email']) ? $data['data']['user_email'] : '';

						if (empty($email)) {
							return false;
						}

						$user_data = array(
							'meta_fields' => array(),
							'status'      => $status,
							'source'      => 'Automation'
						);
						// Populate the user_data dynamically.
						foreach ($form_data as $key => $value) {
							switch ($key) {
								case 'first_name':
								case 'last_name':
								case 'last_activity':
									$user_data[$key] = $value;
									break;
								case 'status':
								case 'source':
								case 'form_id':
								case 'user_email':
									// Skip these keys since they are already set or will be set separately.
									break;
								default:
									$user_data['meta_fields'][$key] = $value;
									break;
							}
						}

						$contact       = new ContactData($email, $user_data);
						$exist_email   = ContactModel::is_contact_exist($email);
						if (!$exist_email) {
							$contact_id = ContactModel::insert($contact);
							if (isset($user_data['status']) && 'pending' === $user_data['status']) {
								MessageController::get_instance()->send_double_opt_in($contact_id);
							}
						}
						return true;
					}
				}
			} else {
				return true;
			}
		}
		return true;
	}


	/**
	 * After gform submission
	 *
	 * @param array $entry Entry.
	 * @param array $form Form.
	 *
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_gform_after_submission($entry, $form)
	{
		$user_data = array();
		foreach ($form['fields'] as $field) {
			$inputs = $field->get_entry_inputs();
			if (is_array($inputs)) {
				foreach ($inputs as $input) {
					if (isset($input['id']) && function_exists('rgar')) {
						$value = rgar($entry, (string) $input['id']);
						if (isset($input['label']) && 'First' === $input['label']) {
							$user_data['first_name'] = $value;
						} elseif (isset($input['label']) && 'Last' === $input['label']) {
							$user_data['last_name'] = $value;
						}
					}
				}
			} else {
				if (isset($field->id) && function_exists('rgar')) {
					$value = rgar($entry, (string) $field->id);
					if (isset($field->type) && 'email' === $field->type) {
						$user_data['email'] = $value;
					}
				}
			}
		}

		$data = array(
			'connector_name' => $this->connector_name,
			'trigger_name'   => 'gform_after_submission',
			'data'           => array(
				'user_email' => isset($user_data['email']) ? $user_data['email'] : '',
				'first_name' => isset($user_data['first_name']) ? $user_data['first_name'] : '',
				'last_name'  => isset($user_data['last_name']) ? $user_data['last_name'] : '',
				'form_id'    => isset($form['id']) ? $form['id'] : '',
				'entry'	     => $entry,
			),
		);
		do_action(MINT_TRIGGER_AUTOMATION, $data);
	}

	/**
	 * After gform send email
	 *
	 * @param boolean $is_success Form submission true or False.
	 * @param string  $to Receiver email.
	 * @param string  $subject Email subject.
	 * @param string  $message Email message.
	 * @param string  $headers Email Header.
	 * @param string  $attachments Email attachemnt.
	 * @param string  $message_format Message Formate.
	 * @param string  $from Which form is submitted.
	 * @param string  $from_name Form name.
	 * @param string  $bcc Email BCC.
	 * @param string  $reply_to Email replay.
	 * @param string  $entry Form entries.
	 * @param string  $cc Email cc.
	 * @return void
	 */
	public function mint_gform_after_email($is_success, $to, $subject, $message, $headers, $attachments, $message_format, $from, $from_name, $bcc, $reply_to, $entry, $cc)
	{
		$user_data = array();
		if (!empty($entry['form_id'])) {
			if (class_exists('GFFormsModel')) {
				$form = \GFFormsModel::get_form_meta(absint($entry['form_id']));
				foreach ($form['fields'] as $field) {
					$inputs = $field->get_entry_inputs();
					if (is_array($inputs)) {
						foreach ($inputs as $input) {
							if (isset($input['id']) && function_exists('rgar')) {
								$value = rgar($entry, (string) $input['id']);
								if (isset($input['label']) && 'First' === $input['label']) {
									$user_data['first_name'] = $value;
								} elseif (isset($input['label']) && 'Last' === $input['label']) {
									$user_data['last_name'] = $value;
								}
							}
						}
					} else {
						if (isset($field->id) && function_exists('rgar')) {
							$value = rgar($entry, (string) $field->id);
							if (isset($field->type) && 'email' === $field->type) {
								$user_data['email'] = $value;
							}
						}
					}
				}
			}
		}
		if ($user_data && isset($user_data['email'])) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'gform_after_email',
				'data'           => array(
					'user_email' => $user_data['email'],
					'first_name' => isset($user_data['first_name']) ? $user_data['first_name'] : '',
					'last_name'  => isset($user_data['last_name']) ? $user_data['last_name'] : '',
					'form_id'    => isset($entry['form_id']) ? $entry['form_id'] : '',
				),
			);
			do_action(MINT_TRIGGER_AUTOMATION, $data);
		}
	}

	/**
	 * After fail gform send email
	 *
	 * @param string $error Error message.
	 * @param string $email Email address.
	 * @param string $entry Form Entries.
	 * @return void
	 */
	public function mint_gform_send_email_failed($error, $email, $entry)
	{
		$user_data = array();
		if (!empty($entry['form_id'])) {
			if (class_exists('GFFormsModel')) {
				$form = \GFFormsModel::get_form_meta(absint($entry['form_id']));
				foreach ($form['fields'] as $field) {
					$inputs = $field->get_entry_inputs();
					if (is_array($inputs)) {
						foreach ($inputs as $input) {
							if (isset($input['id']) && function_exists('rgar')) {
								$value = rgar($entry, (string) $input['id']);
								if (isset($input['label']) && 'First' === $input['label']) {
									$user_data['first_name'] = $value;
								} elseif (isset($input['label']) && 'Last' === $input['label']) {
									$user_data['last_name'] = $value;
								}
							}
						}
					} else {
						if (isset($field->id) && function_exists('rgar')) {
							$value = rgar($entry, (string) $field->id);
							if (isset($field->type) && 'email' === $field->type) {
								$user_data['email'] = $value;
							}
						}
					}
				}
			}
		}
		if ($user_data && isset($user_data['email'])) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'gform_send_email_failed',
				'data'           => array(
					'user_email' => $user_data['email'],
					'first_name' => isset($user_data['first_name']) ? $user_data['first_name'] : '',
					'last_name'  => isset($user_data['last_name']) ? $user_data['last_name'] : '',
					'form_id'    => isset($entry['form_id']) ? $entry['form_id'] : '',
				),
			);
			do_action(MINT_TRIGGER_AUTOMATION, $data);
		}
	}
}
