<?php
/**
 * fluent_booking
 *
 * @package MintMail\App\Internal\Automation\Connector\trigger;
 */

namespace MintMail\App\Internal\Automation\Connector\Trigger;

use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;
/**
 * fluent_booking
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class FluentBookingTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var $connector_name
	 */
	public $connector_name = 'FluentBooking';

	/**
	 * Initialization of fluent_booking hooks
	 */
	public function init() {
		add_action( 'fluent_booking/after_booking_scheduled', array( $this, 'mint_fluentbooking_new_booking' ), 20, 3 );
		add_action( 'fluent_booking/booking_schedule_cancelled', array( $this, 'mint_fluentbooking_cancelled' ), 20, 2 );
		add_action( 'fluent_booking/booking_schedule_completed', array( $this, 'mint_fluentbooking_completed' ), 20, 2 );
		add_action( 'fluent_booking/after_booking_rescheduled', array( $this, 'mint_fluentbooking_rescheduled' ), 20, 3 );
	}

	/**
	 * Validate trigger settings
	 *
	 * @param array $step_data Get Step data.
	 * @param array $data Get All data.
	 * @return bool
	 * @since 1.18.2
	 */
	public function validate_settings( $step_data, $data ) {
		$allowed_trigger_names = array( 'fluentbooking_new_booking','fluentbooking_cancelled','fluentbooking_completed','fluentbooking_rescheduled');
		if ( isset( $data['trigger_name'] ) && in_array( $data['trigger_name'], $allowed_trigger_names, true ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}

			if ( isset( $step_data['settings']['fluentbooking_settings']['calenders'] ) && !empty( $step_data['settings']['fluentbooking_settings']['calenders'] ) ) {
				$selected_calenders= $step_data['settings']['fluentbooking_settings']['calenders'];
				$calender_id       = $data['data']['calender_id'];
				return in_array( $calender_id, array_column( $selected_calenders, 'value' ) );
			}
			return false;
		}
		return false;
	}

	/**
	 * Mint trigger for Fluent Booking submission inserted event.
	 *
	 * @param object  $booking The data submitted during the booking.
	 * @param object calendarEvent object.
	 *
	 * @return void
	 * @since 1.18.2
	 */
	public function mint_fluentbooking_new_booking( $booking, $calendar_slot ) {
		if ($booking->email) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'fluentbooking_new_booking',
				'data'           => array(
					'user_email'   => $booking->email,
					'first_name'   => $booking->first_name,
					'last_name'    => $booking->last_name,
					'booking_id'   => $booking->id,
					'calender_id'  => $calendar_slot->id,
					'booking_data' => $booking,
				),
			);
			do_action(MINT_TRIGGER_AUTOMATION, $data);
		}
	}

	/**
	 * Mint trigger for Fluent Booking cancelled event.
	 *
	 * @param object $booking Booking object.
	 * @param object $calendar_slot Calendar slot object.
	 *
	 * @return void
	 * @since 1.18.2
	 */
	public function mint_fluentbooking_cancelled( $booking, $calendar_slot ) {
		if ( $booking->email ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'fluentbooking_cancelled',
				'data'           => array(
					'user_email'   => $booking->email,
					'first_name'   => $booking->first_name,
					'last_name'    => $booking->last_name,
					'booking_id'   => $booking->id,
					'calender_id'  => $calendar_slot->id,
					'booking_data' => $booking,
				),
			);
			do_action(MINT_TRIGGER_AUTOMATION, $data);
		}
	}

	/**
	 * Mint trigger for Fluent Booking completed event.
	 *
	 * @param object $booking Booking object.
	 * @param object $calendar_slot Calendar slot object.
	 *
	 * @return void
	 * @since 1.18.2
	 */
	public function mint_fluentbooking_completed( $booking, $calendar_slot ) {
		if ($booking->email) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'fluentbooking_completed',
				'data'           => array(
					'user_email'   => $booking->email,
					'first_name'   => $booking->first_name,
					'last_name'    => $booking->last_name,
					'booking_id'   => $booking->id,
					'calender_id'  => $calendar_slot->id,
					'booking_data' => $booking,
				),
			);
			do_action(MINT_TRIGGER_AUTOMATION, $data);
		}
	}

	/**
	 * Handles the rescheduling of a FluentBooking.
	 *
	 * This function is triggered when a booking is rescheduled in FluentBooking.
	 * It collects user data from the previous booking and triggers an automation action.
	 *
	 * @param object $booking 	   The new booking data.
	 * @param object $old_booking  The previous booking data.
	 * @param object $calendar_slot The calendar slot object.
	 *
	 * @return array
	 * @since 1.18.2
	 */
	public function mint_fluentbooking_rescheduled( $booking, $old_booking, $calendar_slot ) {
		if ( $booking->email ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'fluentbooking_rescheduled',
				'data'           => array(
					'user_email'       => $booking->email,
					'first_name'       => $booking->first_name,
					'last_name'        => $booking->last_name,
					'booking_id'       => $old_booking->id,
					'calender_id'      => $calendar_slot->id,
					'booking_data'     => $old_booking,
					'existing_booking' => $booking,
				),
			);
			do_action(MINT_TRIGGER_AUTOMATION, $data);
		}
	}

}



