<?php
/**
 * Edd triggers
 *
 * @package MintMail\App\Internal\Automation\Connector\trigger;
 */

namespace MintMail\App\Internal\Automation\Connector\trigger;

use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;
/**
 * Edd triggers
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class EddTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var $connector_name
	 */
	public $connector_name = 'Edd';

	/**
	 * Initialization of Edd hooks
	 */
	public function init() {
		add_action( 'edd_complete_purchase', array( $this, 'mint_edd_complete_purchase' ), 10, 1 );
		add_action( 'edd_insert_user', array( $this, 'mint_edd_insert_user' ), 10, 2 );
		add_action( 'edd_update_payment_status', array( $this, 'mint_edd_update_payment_status' ), 10, 3 );
		add_action( 'edd_recurring_update_subscription', array( $this, 'mint_edd_recurring_update_subscription' ), 10, 1 );
	}


	/**
	 * Validate trigger settings
	 *
	 * @param array $step_data Get Step data.
	 * @param array $data Get All data.
	 * @return bool
	 */
	public function validate_settings( $step_data, $data ) {
		if ( isset( $data['trigger_name'] ) && ( 'edd_update_payment_status' === $data['trigger_name'] || 'edd_recurring_update_subscription' === $data['trigger_name'] ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}
			if ( isset( $step_data['settings']['status_settings']['status'] ) && 'any' !== $step_data['settings']['status_settings']['status'] ) {
				if ( isset( $data['data']['new_status'] ) ) {
					return $data['data']['new_status'] === $step_data['settings']['status_settings']['status'];
				}
			} else {
				return true;
			}
		}

		if ( isset( $data['trigger_name'], $data['data']['payment_id'] ) && ( 'edd_complete_purchase' === $data['trigger_name'] ) ) {
			$payment_meta = edd_get_payment_meta( $data['data']['payment_id'] );
			if ( $payment_meta && isset( $payment_meta['cart_details'] ) ) {
				if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
					$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
					if ( !empty( $step_data['settings']['product_settings']['products'] ) ) {
						$selected_products = $step_data['settings']['product_settings']['products'];
						$cart_details      = $payment_meta['cart_details'];
						foreach ( $cart_details as $key =>$cart ) {
							if ( !empty( $cart['id'] ) ) {
								$key = array_search( $cart['id'], array_column( $selected_products, 'value' ) ); //phpcs:ignore
								if ( false !== $key ) { //phpcs:ignore
									return true;
								}
							}
						}
						return false;
					}
					return true;
				}
			}
		}

		return true;
	}


	/**
	 * Edd complete purchase
	 *
	 * @param string $payment_id Payment ID.
	 *
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_edd_complete_purchase( $payment_id ) {
		$payment_meta = edd_get_payment_meta( $payment_id );

		if ( $payment_meta ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'edd_complete_purchase',
				'data'           => array(
					'user_email' => isset( $payment_meta['user_info']['email'] ) ? $payment_meta['user_info']['email'] : '',
					'first_name' => isset( $payment_meta['user_info']['first_name'] ) ? $payment_meta['user_info']['first_name'] : '',
					'last_name'  => isset( $payment_meta['user_info']['last_name'] ) ? $payment_meta['user_info']['last_name'] : '',
					'payment_id' => $payment_id,
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Edd insert user
	 *
	 * @param string $user_id user ID.
	 * @param string $user_data user data.
	 *
	 * @return void
	 * @since  1.0.0
	 */
	public function mint_edd_insert_user( $user_id, $user_data ) {
		if ( $user_data ) {
			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'edd_insert_user',
				'data'           => array(
					'user_email' => isset( $user_data['user_email'] ) ? $user_data['user_email'] : '',
					'first_name' => isset( $user_data['user_first'] ) ? $user_data['user_first'] : '',
					'last_name'  => isset( $user_data['user_last'] ) ? $user_data['user_last'] : '',
				),
			);
			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}


	/**
	 * Edd insert user
	 *
	 * @param string $payment_id Payment ID.
	 * @param string $new_status new status.
	 * @param string $old_status old status.
	 *
	 * @return void
	 * @since  1.0.0
	 * @since  1.10.0 Bugfix: Added user email, first name and last name to the data array.
	 */
	public function mint_edd_update_payment_status( $payment_id, $new_status, $old_status ) {
		static $has_run = false;

		if ( $has_run ) {
			return;
		}

		$has_run = true;
		if ( class_exists( 'EDD_Payment' ) ) {
			$payment   = edd_get_payment( $payment_id );
			$user_id   = $payment->user_id;
			$user_data = get_userdata( $user_id );

			$data = array(
				'connector_name' => $this->connector_name,
				'trigger_name'   => 'edd_update_payment_status',
				'data'           => array(
					'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : $payment->email,
					'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : $payment->first_name,
					'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : $payment->last_name,
					'new_status' => $new_status,
					'payment_id' => $payment_id,
				),
			);

			do_action( MINT_TRIGGER_AUTOMATION, $data );
		}
	}

	/**
	 * Edd subscription status update
	 *
	 * @param string $subscription_id Subscription ID.
	 *
	 * @return void
	 * @since  1.0.0
	 * @since  1.10.0 Bugfix: Added user email, first name and last name to the data array.
	 */
	public function mint_edd_recurring_update_subscription( $subscription_id = null ) {
		static $has_run = false;

		if ( $has_run ) {
			return;
		}

		$has_run = true;
		if ( class_exists( 'EDD_Subscription' ) && $subscription_id ) {
			$subscription = new \EDD_Subscription( $subscription_id );

			if ( $subscription ) {
				$user_id   = isset( $subscription->customer->user_id ) ? $subscription->customer->user_id : '';
				$user_data = get_userdata( $user_id );
				$data = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'edd_recurring_update_subscription',
					'data'           => array(
						'user_email' => isset( $user_data->data->ID ) ? $user_data->data->user_email : $subscription->customer->email,
						'first_name' => isset( $user_data->first_name ) ? $user_data->first_name : $subscription->customer->first_name,
						'last_name'  => isset( $user_data->last_name ) ? $user_data->last_name : $subscription->customer->last_name,
						'new_status' => $subscription->get_status(),
						'payment_id' => $subscription->get_original_payment_id(),
					),
				);

				do_action( MINT_TRIGGER_AUTOMATION, $data );
			}
		}
	}
}

