<?php
/**
 * Contact Form 7 submission triggers
 *
 * @package MintMail\App\Internal\Automation\Connector\trigger;
 */

namespace MintMail\App\Internal\Automation\Connector\Trigger;

use Mint\MRM\Admin\API\Controllers\MessageController;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataStores\ContactData;
use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;
/**
 * Contact Form 7 submission triggers
 *
 * @package MintMail\App\Internal\Automation\Connector
 */
class ContactFormTriggers {

	use Singleton;

	/**
	 * Connector name
	 *
	 * @var $connector_name
	 */
	public $connector_name = 'ContactForm';

	/**
	 * Initialization of Contact Form 7 hooks
	 */
	public function init() {
		add_action( 'wpcf7_submit', array( $this, 'mint_contact_form_submission_inserted' ), 20, 2 );
	}


	/**
	 * Validate trigger settings
	 *
	 * @param array $step_data Get Step data.
	 * @param array $data Get All data.
	 * @return bool
	 * @since 1.5.12
	 */
	public function validate_settings( $step_data, $data ) {
		if ( isset( $data['trigger_name'] ) && ( ( 'wpcf7_submit' === $data['trigger_name'] ) ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}
			if ( isset( $step_data['settings']['contact_form_settings']['form_id'] ) ) {
				if ( isset( $data['data']['form_id'] ) ) {
					
					// Check if the form_id is an array or not.
					$form_id          = $step_data['settings']['contact_form_settings']['form_id'];
					$settings_form_id = is_array($form_id) ? $form_id['value'] : $form_id;

					$is_validate = ($data['data']['form_id'] == $settings_form_id); //phpcs:ignore
					if ($is_validate) {
						$form_data = isset($data['data']) ? $data['data'] : array();
						$status    = isset($step_data['settings']['contact_form_settings']['status']) ? $step_data['settings']['contact_form_settings']['status'] : 'pending';
						$email     = isset($data['data']['user_email']) ? $data['data']['user_email'] : '';
						if (empty($email)) {
							return false;
						}
						$user_data = array(
							'meta_fields' => array(),
							'status'      => $status,
							'source'      => 'Automation'
						);
						// Populate the user_data dynamically.
						foreach ($form_data as $key => $value) {
							switch ($key) {
								case 'first_name':
								case 'last_name':
								case 'last_activity':
									$user_data[$key] = $value;
									break;
								case 'status':
								case 'source':
								case 'form_id':
								case 'user_email':
									// Skip these keys since they are already set or will be set separately.
									break;
								default:
									$user_data['meta_fields'][$key] = $value;
									break;
							}
						}

						$contact       = new ContactData($email, $user_data);
						$exist_email   = ContactModel::is_contact_exist($email);
						if (!$exist_email) {
							$contact_id = ContactModel::insert($contact);
							if (isset($user_data['status']) && 'pending' === $user_data['status']) {
								MessageController::get_instance()->send_double_opt_in($contact_id);
							}
						}
						return true;
					}
				}
			} else {
				return false;
			}
		}
		return false;
	}

	/**
	 * Mint trigger for Contact Form 7 submission inserted event.
	 *
	 * @param array  $form_data The data submitted with the form.
	 * @param object $form The Contact Form 7 form object.
	 *
	 * @return void
	 * @since 1.5.12
	 */
	public function mint_contact_form_submission_inserted( $form_data, $form ) {
		if ( 'validation_failed' === $form['status'] || ( isset( $form['demo_mode'] ) && false !== $form['demo_mode'] ) ) {
			return;
		}

		$form_data = $this->get_submitted_form_values( $form_data );

		if ( !empty( $form['contact_form_id'] ) ) {
			if ( is_array( $form_data ) ) {
				$data = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'wpcf7_submit',
					'data'           => array(
						'form_data' => $form_data,
						'form_id'   => $form['contact_form_id'],
					),
				);

				/**
				 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
				 *
				 * This action triggers automation processes with the given data for handling automation events.
				 *
				 * @param mixed $data The data associated with the automation trigger.
				 */
				do_action( MINT_TRIGGER_AUTOMATION, $data );
			}
		}
	}

	/**
	 * Retrieve submitted form values from a Contact Form 7 form.
	 *
	 * This method scans the provided Contact Form 7 form for tags and retrieves the submitted values. It processes pipes and correctly handles select fields with multiple options.
	 *
	 * @param \WPCF7_ContactForm $form The Contact Form 7 form object to retrieve values from.
	 * @return array An array of submitted form values with field names as keys and their respective values as values.
	 *
	 * @since 1.5.12
	 */
	private function get_submitted_form_values( $form ) {
		$tags = $form->scan_form_tags();

		$data = array();
		foreach ( $tags as $tag ) {
			if ( empty( $tag->name ) || false !== strpos( $tag->type, 'file' ) || 'submit' === $tag->type ) {
				continue;
			}

			$pipes = $tag->pipes;

			$value = ( ! empty( $_POST[ $tag->name ] ) ) ? $_POST[ $tag->name ] : ''; //phpcs:ignore
			if ( ! WPCF7_USE_PIPE || ! $pipes instanceof \WPCF7_Pipes || $pipes->zero() ) {
				$data[ $tag->name ] = $value;
			}

			// Select field pipes.
			if ( is_array( $value ) ) {
				$new_value = array();

				foreach ( $value as $v ) {
					$new_value[] = $pipes->do_pipe( wp_unslash( $v ) );
				}

				$value = $new_value;
			} else {
				$value = $pipes->do_pipe( wp_unslash( $value ) );
			}

			$data[ $tag->name ] = $value;
		}

		return $data;
	}
}

