<?php
/**
 * This class handles the triggers for Bricks Form within the Mail Mint automation system.
 *
 * This class handles the triggers for Bricks Form within the Mail Mint automation system.
 * It includes methods for initializing the triggers, validating settings, handling form submissions,
 * and extracting form values.
 *
 * @package MintMail\App\Internal\Automation\Connector\Trigger
 */

namespace MintMail\App\Internal\Automation\Connector\Trigger;

use Mint\MRM\Admin\API\Controllers\MessageController;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataStores\ContactData;
use MintMail\App\Internal\Automation\HelperFunctions;
use Mint\Mrm\Internal\Traits\Singleton;

/**
 * Class BricksFormTriggers
 *
 * This class handles the triggers for Bricks Form within the Mail Mint automation system.
 *
 * @since 1.14.0
 */
class BricksFormTriggers {

	use Singleton;

	/**
     * @var string $connector_name The name of the connector.
	 * 
	 * @since 1.14.0
     */
	public $connector_name = 'BricksForm';

	/**
	 * Initialization of Bricks Form hooks
	 *
	 * This method initializes the Bricks Form hooks for handling form submissions.
	 * 
	 * @since 1.14.0
	 */
	public function init() {
		add_action( 'bricks/form/custom_action', array( $this, 'mint_bricks_form_after_submission' ) );
	}

	/**
	 * Validate trigger settings
	 *
	 * This method validates the settings for the Bricks Form trigger.
	 * 
	 * @param array $step_data Get Step data.
	 * @param array $data Get All data.
	 * @return bool
	 * @since 1.14.0
	 */
	public function validate_settings( $step_data, $data ) {
		if ( isset( $data['trigger_name'] ) && ( ( 'bricks_form_submit' === $data['trigger_name'] ) ) ) {
			if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
				$step_data = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			}
			if ( isset( $step_data['settings']['bricks_form_settings']['form_id'] ) && 'any' !== $step_data['settings']['bricks_form_settings']['form_id'] ) {
				if ( isset( $data['data']['form_id'] ) ) {
					$form_id          = $step_data['settings']['bricks_form_settings']['form_id'];
					$settings_form_id = is_array($form_id) ? $form_id['value'] : $form_id;
					$is_validate = ($data['data']['form_id'] == $settings_form_id); //phpcs:ignore
					if ( $is_validate ) {
						$form_data = isset( $data['data'] ) ? $data['data'] : array();
						$status    = isset( $step_data['settings']['bricks_form_settings']['status'] ) ? $step_data['settings']['bricks_form_settings']['status'] : 'subscribed';
						$email     = isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
						$user_data = array(
							'meta_fields' => array(),
							'status' => $status,
							'source' => 'Automation'
						);

						// Populate the user_data dynamically.
						foreach ($form_data as $key => $value) {
							switch ($key) {
								case 'first_name':
								case 'last_name':
								case 'last_activity':
									$user_data[$key] = $value;
									break;
								case 'status':
								case 'source':
								case 'form_id':
								case 'user_email':
									// Skip these keys since they are already set or will be set separately.
									break;
								default:
									$user_data['meta_fields'][$key] = $value;
									break;
							}
						}

						$contact       = new ContactData( $email, $user_data );
						$exist_email   = ContactModel::is_contact_exist( $email );
						if ( !$exist_email ) {
							$contact_id = ContactModel::insert( $contact );
							if ( isset( $user_data['status'] ) && 'pending' === $user_data['status'] ) {
								MessageController::get_instance()->send_double_opt_in( $contact_id );
							}
						}
						return true;
					}
				}
			} else {
				return false;
			}
		}
		return false;
	}

	/**
	 * Mint trigger for Bricks Form submission inserted event.
	 *
	 * This method triggers the automation process for Bricks Form submissions.
	 *
	 * @param array  $form The form data submitted with the form.
	 * @since 1.14.0
	 */
	public function mint_bricks_form_after_submission( $form ) {
		$form_data = $this->get_submitted_form_values( $form );
		if ( !empty( $form_data['formId'] ) ) {
			if ( is_array( $form_data ) ) {
				$data = array(
					'connector_name' => $this->connector_name,
					'trigger_name'   => 'bricks_form_submit',
					'data'           => array(
						'form_data' => $form_data,
						'form_id'   => $form_data['formId'],
					),
				);

				/**
				 * Perform an action named 'MINT_TRIGGER_AUTOMATION' with the provided data.
				 *
				 * This action triggers automation processes with the given data for handling automation events.
				 *
				 * @param mixed $data The data associated with the automation trigger.
				 */
				do_action( MINT_TRIGGER_AUTOMATION, $data );
			}
		}
	}

	/**
	 * Extract form values
	 *
	 * This method extracts the form values from the submitted form data.
	 *
	 * @param object $form The submitted form object.
	 * @return array The extracted form values.
	 * @since 1.14.0
	 */
	private function get_submitted_form_values( $form ) {
		$fields = $form->get_fields();
        $files  = $form->get_uploaded_files();

        $data = array();
        foreach ( $fields as $key => $value ) {
            $field_id          = str_replace( 'form-field-', '', $key );
            $data[ $field_id ] = ( \is_array( $value ) && \count( $value ) == 1 ) ? $value[0] : $value;
        }

        foreach ( $files as $key => $item ) {
            $field_id = str_replace( 'form-field-', '', $key );

            if ( \is_array( $item ) ) {
                foreach ( $item as $file ) {
                    if ( !isset( $file['file'] ) ) {
                        continue;
                    }
                    $data[ $field_id ][] = $file['file'];
                }
            } else {
                if ( !isset( $item['file'] ) ) {
                    continue;
                }
                $data[ $field_id ] = $item['file'];
            }
        }
		return $data;
	}
}

