<?php
/**
 * Automation manager
 *
 * @package MintMail\App\Internal\Automation
 */

namespace MailMintPro\App\Internal\Automation;

use MailMint\App\Helper;
use Mint\MRM\DataBase\Models\CampaignEmailBuilderModel;
use Mint\MRM\DataBase\Models\ContactModel;
use MintMail\App\Internal\Automation\HelperFunctions;
use MRM\Common\MrmCommon;
use Mint\Mrm\Internal\Traits\Singleton;
use Mint\MRM\DataBase\Models\EmailModel;
use Mint\MRM\Internal\Parser\Parser;
use Mint\MRM\Utilites\Helper\Email;
use MintMail\App\Internal\Automation\Action\AutomationAction;
use MintMailPro\Mint_Pro_Helper;

/**
 * Automation manager
 *
 * @package MintMail\App\Internal\Automation
 */
class AutomationManager {

	/**
	 * Initialization
	 */
	public function __construct() {
		add_action( 'mint_process_sequence', array( $this, 'process_sequence_data' ), 10 );
	}


	/**
	 * Process sequence data;
	 *
	 * @param array $data Get All data.
	 */
	public function process_sequence_data( $data ) {
		/**
		 * Applies filters before processing an automation sequence.
		 *
		 * This function applies filters to the provided data before processing it
		 * in an automation sequence. The filters are hooked to the 'mailmint_before_automation_sequence_process' action.
		 *
		 * @param mixed $data The data to be processed in the automation sequence.
		 * @return mixed The filtered data after applying the filters.
		 */
		$data = apply_filters( 'mailmint_before_automation_sequence_process', $data );
		if ( !empty( $data['email_id'] ) && HelperFunctions::maybe_user( $data['user_email'] ) ) {
			$default_business_settings = MrmCommon::business_settings_default_configuration();
			$business_settings         = get_option( '_mrm_business_basic_info_setting', $default_business_settings );
			$business_settings         = is_array( $business_settings ) && ! empty( $business_settings ) ? $business_settings : $default_business_settings;

			$contact_id   = HelperFunctions::get_contact_id_by_broadcast_table($data['user_email']);
			$post_id      = isset($data['data']['data']['post_id']) ? $data['data']['data']['post_id'] : '';
			$order_id	  = isset($data['data']['data']['order_id']) ? $data['data']['data']['order_id'] : '';
			$abandoned_id = isset($data['data']['data']['abandoned_id']) ? $data['data']['data']['abandoned_id'] : '';
			$payment_id   = isset($data['data']['data']['payment_id']) ? $data['data']['data']['payment_id'] : '';
			$subscription_id = isset($data['data']['data']['subscription_id']) ? $data['data']['data']['subscription_id'] : '';
			$wishlist_id 	 = isset($data['data']['wishlist_id']) ? $data['data']['wishlist_id'] : '';
			$user_membership_id = isset($data['data']['user_membership_id']) ? $data['data']['user_membership_id'] : '';
			$product_id	  	 = isset($data['data']['product_id']) ? $data['data']['product_id'] : '';
			$ld_course_id	 = isset($data['data']['ld_course_id']) ? $data['data']['ld_course_id'] : '';
			$ld_quiz_id		 = isset($data['data']['ld_quiz_id']) ? $data['data']['ld_quiz_id'] : '';
			$ld_lesson_id	 = isset($data['data']['ld_lesson_id']) ? $data['data']['ld_lesson_id'] : '';
			$ld_group_id	 = isset($data['data']['ld_group_id']) ? $data['data']['ld_group_id'] : '';
			$ld_topic_id	 = isset($data['data']['ld_topic_id']) ? $data['data']['ld_topic_id'] : '';
			$booking_id 	 = isset($data['data']['booking_id']) ? $data['data']['booking_id'] : '';

			// Get contact and merge meta fields with contact fields.
			$contact = ContactModel::get($contact_id);
			if (isset($contact['meta_fields']) && is_array($contact['meta_fields'])) {
				$contact = array_merge($contact, $contact['meta_fields']);
				unset($contact['meta_fields']);
			}

			$email_data  = HelperFunctions::get_email_body_by_email_id( $data['email_id'] );
			$email       = HelperFunctions::get_campaign_email_by_email_id( $data['email_id'] );
			$email_body  = isset( $email_data['email_body'] ) ? html_entity_decode( $email_data['email_body'] ) : ''; // phpcs:ignore
			$domain_link = get_site_url();
			$rand_hash   = MrmCommon::get_rand_email_hash( $data['user_email'], $data['automation_id'] );
			$track_url   = add_query_arg(
				array(
					'mint'  => 1,
					'route' => 'open',
					'hash'  => $rand_hash,
				),
				$domain_link
			);

			$image_url  = "<img src= $track_url />";
			$email_body = $email_body . $image_url;
			$email_body = Helper::replace_url( $email_body, $rand_hash );
			$email_body = Parser::parse(
				$email_body,
				$contact,
				$post_id,
				$order_id,
				array(
					'abandoned_id'       => $abandoned_id,
					'edd_payment_id'     => $payment_id,
					'subscription_id'    => $subscription_id,
					'product_id'         => $product_id,
					'user_membership_id' => $user_membership_id,
					'wishlist_id'        => $wishlist_id,
					'ld_course_id'       => $ld_course_id,
					'ld_quiz_id'         => $ld_quiz_id,
					'ld_lesson_id'       => $ld_lesson_id,
					'ld_group_id'        => $ld_group_id,
					'ld_topic_id'        => $ld_topic_id,
					'booking_id'         => $booking_id
				)
			);
			$email_body = Helper::replace_dynamic_coupon($email_body, $data['user_email']);

			// Prepare email header information.
			$header_data = array(
				'reply_name'  => ! empty( $email['reply_name'] ) ? $email['reply_name'] : '',
				'reply_email' => ! empty( $email['reply_email'] ) ? $email['reply_email'] : '',
				'from_email'  => ! empty( $email['sender_email'] ) ? $email['sender_email'] : '',
				'from_name'   => ! empty( $email['sender_name'] ) ? $email['sender_name'] : '',
			);

			$unsub_url = Helper::get_unsubscribed_url( $rand_hash );
			$headers   = Email::get_mail_header( $header_data, $unsub_url );
			$preview   = isset( $email['email_preview_text'] ) ? $email['email_preview_text'] : '';
			$preview   = Parser::parse(
				$preview,
				$contact,
				$post_id,
				$order_id,
				array(
					'abandoned_id'       => $abandoned_id,
					'edd_payment_id'     => $payment_id,
					'subscription_id'    => $subscription_id,
					'product_id'         => $product_id,
					'user_membership_id' => $user_membership_id,
					'wishlist_id'        => $wishlist_id,
					'ld_course_id'       => $ld_course_id,
					'ld_quiz_id'         => $ld_quiz_id,
					'ld_lesson_id'       => $ld_lesson_id,
					'ld_group_id'        => $ld_group_id,
					'ld_topic_id'        => $ld_topic_id,
					'booking_id'         => $booking_id
				)
			);
			$headers[] = 'X-PreHeader: ' . $preview;

			$email_subject  = isset( $email['email_subject'] ) ? $email['email_subject'] : '';
			$email_subject  = Parser::parse(
				$email_subject,
				$contact,
				$post_id,
				$order_id,
				array(
					'abandoned_id'       => $abandoned_id,
					'edd_payment_id'     => $payment_id,
					'subscription_id'    => $subscription_id,
					'product_id'         => $product_id,
					'user_membership_id' => $user_membership_id,
					'wishlist_id'        => $wishlist_id,
					'ld_course_id'       => $ld_course_id,
					'ld_quiz_id'         => $ld_quiz_id,
					'ld_lesson_id'       => $ld_lesson_id,
					'ld_group_id'        => $ld_group_id,
					'ld_topic_id'        => $ld_topic_id
				)
			);
			$scheduler_data = array(
				'body'           => $email_body,
				'subject'        => $email_subject,
				'receiver_email' => !empty( $data['user_email'] ) ? $data['user_email'] : '',
				'header'         => $headers,
			);
			/**
			 * Summary: Applies the 'mint_replace_abandoned_carts_placeholder' filter to replace abandoned carts placeholders in the email body.
			 *
			 * Description: This code section applies the 'mint_replace_abandoned_carts_placeholder' filter hook to the email body.
			 * The filter allows for customization of the replacement of abandoned carts placeholders with actual values in the email body.
			 *
			 * @param string $scheduler_data['body'] The email body to be filtered.
			 * @param array  $data            An array containing data to be passed to the filter hook.
			 *
			 * @return string The filtered email body with replaced abandoned carts placeholders.
			 * @since 1.5.0
			 */
			$scheduler_data['body'] = apply_filters( 'mint_automation_email_body', $scheduler_data['body'], $data['data'] );

			// Process URL for lead-magnet tracking if MailMint Pro is active.
			if (MrmCommon::is_mailmint_pro_active()) {
				$scheduler_data['body'] = Mint_Pro_Helper::replace_automatic_latest_content($scheduler_data['body'], get_post_type($post_id));
				if (MrmCommon::is_mailmint_pro_version_compatible('1.15.1')) {
					$scheduler_data['body'] = Mint_Pro_Helper::process_lead_magnet_tracking($scheduler_data['body'], $data['user_email']);
				}
			}

			$is_sent                = $this->send_message( $scheduler_data );
			$payload                = array(
				'automation_id' => $data['automation_id'],
				'step_id'       => $data['step_id'],
				'email_id'      => $data['email_id'],
				'email_type'    => 'automation',
				'email_address' => !empty( $data['user_email'] ) ? $data['user_email'] : '',
				'contact_id'    => $contact_id,
				'email_hash'    => $rand_hash,
				'created_at'    => current_time( 'mysql' ),
				'updated_at'    => current_time( 'mysql' ),
			);
			if ( $is_sent ) {
				$payload['status'] = 'sent';
			} else {
				$payload['status'] = 'failed';
			}
			EmailModel::insert( $payload );

			$log_payload = array(
				'automation_id' => $data['automation_id'],
				'step_id'       => $data['step_id'],
				'status'        => 'completed',
				'identifier'    => $data['identifier'],
				'email'         => !empty( $data['user_email'] ) ? $data['user_email'] : '',
			);
			HelperFunctions::update_log( $log_payload );
		}
	}


	/**
	 * Send a message to contact
	 *
	 * @param mixed $data Single message object.
	 * @return bool
	 * @since 1.0.0
	 */
	public function send_message( $data ) {
		$to      = isset( $data['receiver_email'] ) ? $data['receiver_email'] : '';
		$subject = isset( $data['subject'] ) ? $data['subject'] : 'Welcome to WPF';
		$body    = isset( $data['body'] ) ? html_entity_decode( $data['body'] ) : ''; //phpcs:ignore
		$body    = str_replace( '</html>', CampaignEmailBuilderModel::get_email_footer_watermark() . '</html>', $body );
		$headers = isset( $data['header'] ) ? $data['header'] : '';
		if ( $to ) {
			return MM()->mailer->send( $to, $subject, $body, $headers );
		}
		return false;
	}

}
