<?php
/**
 * Automation action class for MRM Autoamtion
 *
 * Class Webhook
 *
 * @package MintMail\App\Internal\Automation\Action
 */

namespace MintMail\App\Internal\Automation\Action;

use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\Internal\Parser\Parser;


/**
 * Webhook action
 *
 * Class Webhook
 *
 * @package MintMail\App\Internal\Automation\Action
 */
class WebHookOutgoing extends AbstractAutomationAction {

	use Singleton;
	/**
	 * Action scheduler.
	 *
	 * @var object $action_scheduler
	 */
	private $action_scheduler;
	/**
	 * Initialization
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}


	/**
	 * Get action name
	 *
	 * @return String
	 * @since  1.0.0
	 */
	public function get_name() {
		return __( 'Webhook', 'mailmint-pro' );
	}


	/**
	 * Process.
	 *
	 * @param array $data Get All data.
	 */
	public function process( $data ) {
		if ( $data ) {
			$email          = isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
			$automation_id  = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
			$step_id        = isset( $data['step_id'] ) ? $data['step_id'] : '';
			$post_id         = isset($data['data']['post_id']) ? $data['data']['post_id'] : '';
			$order_id	     = isset($data['data']['order_id']) ? $data['data']['order_id'] : '';
			$abandoned_id    = isset($data['data']['abandoned_id']) ? $data['data']['abandoned_id'] : '';
			$payment_id      = isset($data['data']['payment_id']) ? $data['data']['payment_id'] : '';
			$subscription_id = isset($data['data']['subscription_id']) ? $data['data']['subscription_id'] : '';
			$product_id	  	 = isset($data['data']['product_id']) ? $data['data']['product_id'] : '';
			$user_membership_id = isset($data['data']['user_membership_id']) ? $data['data']['user_membership_id'] : '';
			$wishlist_id 	 = isset($data['data']['wishlist_id']) ? $data['data']['wishlist_id'] : '';
			$ld_course_id	 = isset($data['data']['ld_course_id']) ? $data['data']['ld_course_id'] : '';
			$ld_quiz_id		 = isset($data['data']['ld_quiz_id']) ? $data['data']['ld_quiz_id'] : '';
			$ld_lesson_id	 = isset($data['data']['ld_lesson_id']) ? $data['data']['ld_lesson_id'] : '';
			$ld_group_id	 = isset($data['data']['ld_group_id']) ? $data['data']['ld_group_id'] : '';
			$ld_topic_id	 = isset($data['data']['ld_topic_id']) ? $data['data']['ld_topic_id'] : '';
			$booking_id 	 = isset($data['data']['booking_id']) ? $data['data']['booking_id'] : '';

			$log_payload   = array(
				'automation_id' => $data['automation_id'],
				'step_id'       => $data['step_id'],
				'status'        => 'hold',
				'identifier'    => $data['identifier'],
				'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
			);
			HelperFunctions::update_log( $log_payload );
			if ( $email && $automation_id && $step_id ) {
				$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );
				if ( isset( $step_data['settings']['message_data'] ) ) {
					$settings  = $step_data['settings']['message_data'];
					$body_data = array(
						'body_request' => isset( $settings['body_request'] ) ? $settings['body_request'] : '',
						'body_data'    => isset( $settings['body_request_data'] ) ? $settings['body_request_data'] : array(),
					);

					$header_data  = array(
						'header_request' => isset( $settings['header_request'] ) ? $settings['header_request'] : '',
						'header_data'    => isset( $settings['header_request_data'] ) ? $settings['header_request_data'] : array(),
					);
					$data_format  = isset( $settings['data_format'] ) ? $settings['data_format'] : 'form';
					$method       = isset( $settings['method'] ) ? strtoupper( $settings['method'] ) : 'GET';
					$url          = $this->get_url( $settings );
					$header       = $this->prepare_header( $header_data, $data_format, $method, $url, $email, $post_id, $order_id, $abandoned_id, $payment_id, $subscription_id, $product_id, $user_membership_id, $wishlist_id, $ld_course_id, $ld_quiz_id, $ld_lesson_id, $ld_group_id, $ld_topic_id, $booking_id );
					$body         = $this->prepare_body( $body_data, $email, $post_id, $order_id, $abandoned_id, $payment_id, $subscription_id, $product_id, $user_membership_id, $wishlist_id, $ld_course_id, $ld_quiz_id, $ld_lesson_id, $ld_group_id, $ld_topic_id, $booking_id );
					if ( 'GET' === $method ) {
						$url = add_query_arg( $body, $url );
					}
					$payload      = array(
						'url'          => $url,
						'method'       => $method,
						'data_format'  => $data_format,
						'request_body' => ( 'POST' === $method ) ? $body : null,
						'header'       => $header,
					);
					$request_args = $this->prepare_request_args( $payload );
					$this->send_webhook_data( $url, $request_args );
				}
			}
			if ( !isset( $data['double_optin'] ) ) {
				$next_step = HelperFunctions::get_next_step( $data['automation_id'], $data['step_id'] );
				HelperFunctions::update_job( $data['automation_id'], isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
				if ( $next_step ) {
					$next_step['data']       = $data['data'];
					$next_step['identifier'] = $data['identifier'];
					do_action(MINT_PROCESS_AUTOMATION, $next_step);
				}
			}
		}
	}


	/**
	 * Prepare herder data.
	 *
	 * @param array  $header_data Data.
	 * @param string $data_format Get Date Formate.
	 * @param string $method Get received method.
	 * @param string $url Get URL.
	 * @param string $email Get email address.
	 * @param int    $post_id Get post id.
	 * @param int    $order_id Get order id.
	 * @param int    $abandoned_id Get abandoned id.
	 * @param int    $product_id Get product id.
	 * @param int    $user_membership_id Get user membership id.
	 * @param int    $wishlist_id Get wishlist id.
	 * @param int    $course_id Get course id.
	 * 
	 * @return array
	 * @since 1.0.0
	 */
	private function prepare_header( $header_data, $data_format, $method, $url, $email, $post_id, $order_id, $abandoned_id, $payment_id, $subscription_id, $product_id, $user_membership_id, $wishlist_id, $ld_course_id, $ld_quiz_id, $ld_lesson_id, $ld_group_id, $ld_topic_id, $booking_id ) {
		$contact_id = ContactModel::get_id_by_email($email);
		$contact    = ContactModel::get($contact_id);

		if (isset($contact['meta_fields']) && is_array($contact['meta_fields'])) {
			$contact = array_merge($contact, $contact['meta_fields']);
			unset($contact['meta_fields']);
		}

		$maybe_custom_header = !empty( $header_data['header_request'] ) && 'no_header' === $header_data['header_request'] ? false : true;
		if ( $maybe_custom_header && is_array( $header_data['header_data'] ) ) {
			foreach ( $header_data['header_data'] as $herder ) {
				if ( isset( $herder['header_key'], $herder['header_value'] ) ) {
					$key        = str_replace( ' ', '-', $herder['header_key'] );
					$data[$key] = Parser::parse( 
						$herder['header_value'],
						$contact,
						$post_id,
						$order_id,
						array(
							'abandoned_id'       => $abandoned_id,
							'edd_payment_id'     => $payment_id,
							'subscription_id'    => $subscription_id,
							'product_id'         => $product_id,
							'user_membership_id' => $user_membership_id,
							'wishlist_id'        => $wishlist_id,
							'ld_course_id'       => $ld_course_id,
							'ld_quiz_id'         => $ld_quiz_id,
							'ld_lesson_id'       => $ld_lesson_id,
							'ld_group_id'        => $ld_group_id,
							'ld_topic_id'        => $ld_topic_id,
							'booking_id'         => $booking_id
						)
					);
				}
			}
		}
		return $data;
	}


	/**
	 * Prepare herder data
	 *
	 * @param array  $body_data Data.
	 * @param string $email Get email address.
	 * @param int    $post_id Get post id.
	 * @param int    $order_id Get order id.
	 * @param int    $abandoned_id Get abandoned id.
	 * @param int    $product_id Get product id.
	 * @param int    $user_membership_id Get user membership id.
	 * @param int    $wishlist_id Get wishlist id.
	 * @param int    $course_id Get course id.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	private function prepare_body( $body_data, $email,$post_id, $order_id, $abandoned_id, $payment_id, $subscription_id, $product_id, $user_membership_id, $wishlist_id, $ld_course_id, $ld_quiz_id, $ld_lesson_id, $ld_group_id, $ld_topic_id, $booking_id ) {
		$data = array();
		if ( isset( $body_data['body_request'], $body_data['body_data'] ) ) {

			$contact_id = ContactModel::get_id_by_email( $email );
			$contact    = ContactModel::get( $contact_id );

			if ( isset($contact['meta_fields'] ) && is_array($contact['meta_fields'] ) ) {
				$contact = array_merge( $contact, $contact['meta_fields'] );
				unset($contact['meta_fields'] );
			}

			if ( 'raw' === $body_data['body_request'] ) {
				$data = $contact;
			} else {
				foreach ( $body_data['body_data'] as $item ) {
					if ( empty( $item['body_key'] ) || empty( $item['body_value'] ) ) {
						continue;
					}
					$item['body_key']          = str_replace( ' ', '_', $item['body_key'] );
					$data[ $item['body_key'] ] = Parser::parse(
						$item['body_value'],
						$contact,
						$post_id,
						$order_id,
						array(
							'abandoned_id'       => $abandoned_id,
							'edd_payment_id'     => $payment_id,
							'subscription_id'    => $subscription_id,
							'product_id'         => $product_id,
							'user_membership_id' => $user_membership_id,
							'wishlist_id'        => $wishlist_id,
							'ld_course_id'       => $ld_course_id,
							'ld_quiz_id'         => $ld_quiz_id,
							'ld_lesson_id'       => $ld_lesson_id,
							'ld_group_id'        => $ld_group_id,
							'ld_topic_id'        => $ld_topic_id,
							'booking_id '        => $booking_id
						)
					);
				}
			}
		}
		return $data;
	}


	/**
	 * Get request url
	 *
	 * @param array $settings Settings.
	 * @return string
	 * @since 1.0.0
	 */
	private function get_url( $settings ) {
		$url = '';
		if ( isset( $settings['remote_url'] ) ) {
			$url = $settings['remote_url'];
		}
		return $url;
	}


	/**
	 * Send webhook data.
	 *
	 * @param string $request_url Webhook url.
	 * @param array  $request_args Request data.
	 *
	 * @return void
	 * @since 1.0.0
	 */
	private function send_webhook_data( $request_url, $request_args ) {
		wp_remote_request( $request_url, $request_args );
	}


	/**
	 * Prepare common request body for webhook
	 *
	 * @param array $payload Payload data.
	 * @return array
	 */
	private function prepare_request_args( $payload ) {
		if ( isset( $payload['method'], $payload['data_format'] ) ) {
			if ( 'GET' !== $payload['method'] ) {
				if ( isset( $payload['data_format'] ) ) {
					if ( 'form' === $payload['data_format'] ) {
						$request_body = $payload['request_body'];
						$request_body = http_build_query( $request_body );
					} elseif ( 'json' === $payload['data_format'] ) {
						$request_body = wp_json_encode( $payload['request_body'] );
					}
				}
			}

			$request_args = array(
				'body'      => $request_body,
				'method'    => $payload['method'],
				'headers'   => $payload['header'],
				'sslverify' => 1,
			);
			return $request_args;
		}
		return array();
	}
}
