<?php
/**
 * Automation action class for MRM Automation
 *
 * Class UpdateWpUserMeta
 *
 * @package MintMail\App\Internal\Automation\Action
 */

namespace MintMail\App\Internal\Automation\Action;

use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\Internal\Parser\Parser;

/**
 * UpdateUserMeta action
 *
 * Class UpdateUserMeta
 *
 * @package MintMail\App\Internal\Automation\Action
 */
class UpdateWPUserMeta extends AbstractAutomationAction {

    use Singleton;

    /**
     * Action scheduler.
     *
     * @var ActionScheduler
     */
    private $action_scheduler;

    /**
     * Initialization
     */
    public function __construct() {
        $this->action_scheduler = new ActionScheduler();
    }

    /**
     * Get action name
     *
     * @return string
     * @since 1.18.0
     */
    public function get_name() {
        return __( 'Update WP User Meta', 'mailmint-pro' );
    }

    /**
     * Process the action.
     *
     * @param array $data Input data for processing.
     * @return bool True on success, false otherwise.
     */
    public function process( $data ) {
        // Check if essential data is provided.
		if ( ! isset( $data['automation_id'] ) || ! isset( $data['step_id'] ) ) {
			return;
		}

		// Extract necessary data from the input.
		$user_email    = isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$step_id       = isset( $data['step_id'] ) ? $data['step_id'] : '';
		$identifier    = isset( $data['identifier'] ) ? $data['identifier'] : '';

		// Log that the step is in a 'hold' status.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'hold',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

        $user = get_user_by( 'email', $user_email );
        if ( ! $user ) {
            // If there's a next step, enqueue it for processing.
			$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
			HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
			if ( $next_step ) {
				$next_step['data']       = $data['data'];
				$next_step['identifier'] = $data['identifier'];
				do_action(MINT_PROCESS_AUTOMATION, $next_step);
			}
        }

        $step_data = HelperFunctions::get_step_data( $automation_id, $step_id );
        $settings  = $step_data['settings']['update_settings'] ?? [];

        $contact    = ContactModel::get_contact_by_email( $user_email );
		$contact_id = isset( $contact['id'] ) ? $contact['id'] : '';
		$contact    = ContactModel::get( $contact_id );
		if (isset($contact['meta_fields']) && is_array($contact['meta_fields'])) {
			$contact = array_merge($contact, $contact['meta_fields']);
			unset($contact['meta_fields']);
		}

        $user_metas = [];

        foreach ($settings['meta_properties'] as $pair) {
            if (empty($pair['metaKey']) || empty($pair['metaValue'])) {
                continue;
            }
            $user_metas[sanitize_text_field($pair['metaKey'])] = Parser::parse($pair['metaValue'], $contact);
        }

        foreach ($user_metas as $meta_key => $meta_value) {
            update_user_meta($user->ID, $meta_key, $meta_value);
        }

        // Log that the step is completed.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'completed',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// If there's a next step, enqueue it for processing.
		$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
		HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
		if ( $next_step ) {
			$next_step['data']       = $data['data'];
			$next_step['identifier'] = $data['identifier'];
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
		}
    }
}
