<?php
/**
 * Send Message action class for the Automation
 *
 * Class TwilioSendMessage
 *
 * @package MintMail\App\Internal\Automation\Action
 * @namespace MintMail\App\Internal\Automation\Action
 * @author [getwpfunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-01-03 11:03:17
 * @modify date 2024-01-03 11:03:17
 */

namespace MintMail\App\Internal\Automation\Action;

use Exception;
use Mint\MRM\Internal\Parser\Parser;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;

/**
 * TwilioSendMessage action
 *
 * Class TwilioSendMessage
 *
 * @package MintMail\App\Internal\Automation\Action
 * @since 1.8.0
 */
class TwilioSendMessage extends AbstractAutomationAction {

	use Singleton;

	/**
	 * Holds an instance of the ActionScheduler class.
	 *
	 * @var ActionScheduler
	 *
	 * @access private
	 * @since 1.8.0
	 */
	private $action_scheduler;

	/**
	 * Class constructor.
	 *
	 * @access public
	 * @since 1.8.0
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}

	/**
	 * Get the name of the action.
	 *
	 * Description: Retrieves the localized name for the "Send Message" action.
	 *
	 * @access public
	 * @return string The localized name of the action.
	 * @since 1.8.0
	 */
	public function get_name() {
		return __( 'Send Message', 'mailmint-pro' );
	}

	/**
	 * Process automation step for creating a coupon.
	 *
	 * Description: Processes the automation step for creating a coupon. Updates logs, creates the coupon,
	 *              and handles the next step in the automation.
	 *
	 * @access public
	 * @param array $data The data for the automation step.
	 * @since 1.8.0
	 */
	public function process( $data ) {
		// Check if essential data is provided.
		if ( ! isset( $data['automation_id'] ) || ! isset( $data['step_id'] ) ) {
			return;
		}

		/**
		 * Action hook to run before sending the SMS.
		 * 
		 * @param array $data The data for the automation step.
		 * @since 1.14.1
		 */
		do_action('mint_before_sending_sms', $data);

		// Extract necessary data from the input.
		$user_email    = isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$step_id       = isset( $data['step_id'] ) ? $data['step_id'] : '';
		$identifier    = isset( $data['identifier'] ) ? $data['identifier'] : '';

		// Extract necessary data from the input.
		$post_id         = isset( $data['data']['post_id'] ) ? $data['data']['post_id'] : '';
		$order_id	     = isset( $data['data']['order_id'] ) ? $data['data']['order_id'] : '';
		$abandoned_id    = isset( $data['data']['abandoned_id'] ) ? $data['data']['abandoned_id'] : '';
		$payment_id      = isset( $data['data']['payment_id'] ) ? $data['data']['payment_id'] : '';
		$subscription_id = isset( $data['data']['subscription_id'] ) ? $data['data']['subscription_id'] : '';
		$wishlist_id 	 = isset($data['data']['wishlist_id']) ? $data['data']['wishlist_id'] : '';
		$user_membership_id = isset($data['data']['user_membership_id']) ? $data['data']['user_membership_id'] : '';
		$product_id      = isset( $data['data']['product_id'] ) ? $data['data']['product_id'] : '';
		$ld_course_id	 = isset($data['data']['ld_course_id']) ? $data['data']['ld_course_id'] : '';
		$ld_quiz_id		 = isset($data['data']['ld_quiz_id']) ? $data['data']['ld_quiz_id'] : '';
		$ld_lesson_id	 = isset($data['data']['ld_lesson_id']) ? $data['data']['ld_lesson_id'] : '';
		$ld_group_id	 = isset($data['data']['ld_group_id']) ? $data['data']['ld_group_id'] : '';
		$ld_topic_id	 = isset($data['data']['ld_topic_id']) ? $data['data']['ld_topic_id'] : '';
		$booking_id 	 = isset($data['data']['booking_id']) ? $data['data']['booking_id'] : '';

		// Log that the step is in a 'hold' status.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'hold',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// Retrieve settings data for the current step.
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );

		// Check if coupon settings are available.
		if ( ! isset( $step_data['settings']['twilio_data'] ) || empty( $step_data['settings']['twilio_data'] ) ) {
			return;
		}

		// Prepare data for creating the coupon.
		$settings     = $step_data['settings']['twilio_data'];
		$number_key   = isset( $settings['phone_number'] ) ? $settings['phone_number'] : '{{contact.phone_number}}';
		$parts        = explode( ".", trim( $number_key, "{}" ) );
		$number_key   = $parts[1];

		$settings_from_number = isset( $settings['from_number'] ) ? $settings['from_number'] : '';
		$method               = isset( $settings['method'] ) ? $settings['method'] : 'sms';
		$message_body         = isset( $settings['message_body'] ) ? $settings['message_body'] : '';
		
		$contact    = ContactModel::get_contact_by_email( $user_email );
		$contact_id = isset( $contact['id'] ) ? $contact['id'] : '';
		$contact    = ContactModel::get( $contact_id );

		// Get the phone number from the contact.
		$phone_number = isset( $contact['meta_fields'] ) && isset( $contact['meta_fields'][ $number_key ] ) ? $contact['meta_fields'][ $number_key ] : '';

		if( empty( $phone_number ) ) {
			return;
		}

		if (isset($contact['meta_fields']) && is_array($contact['meta_fields'])) {
			$contact = array_merge($contact, $contact['meta_fields']);
			unset($contact['meta_fields']);
		}

		$message_body = Parser::parse(
			$message_body,
			$contact,
			$post_id,
			$order_id,
			array(
				'abandoned_id'       => $abandoned_id,
				'edd_payment_id'     => $payment_id,
				'subscription_id'    => $subscription_id,
				'product_id'         => $product_id,
				'user_membership_id' => $user_membership_id,
				'wishlist_id'        => $wishlist_id,
				'ld_course_id'       => $ld_course_id,
				'ld_quiz_id'         => $ld_quiz_id,
				'ld_lesson_id'       => $ld_lesson_id,
				'ld_group_id'        => $ld_group_id,
				'ld_topic_id'        => $ld_topic_id,
				'booking_id'         => $booking_id
			)
		);

		$settings = get_option( '_mint_integration_settings', array() );
		$twilio   = isset( $settings['twilio'] ) ? $settings['twilio'] : array();

		if( $twilio && $twilio['is_integrated'] ){
            $account_sid = $twilio['account_sid'];
            $auth_token  = $twilio['auth_token'];
            $from_number = $twilio['from_number'];

            try {
                $to   = $phone_number;
                $from = empty( $settings_from_number ) ? $from_number : $settings_from_number;
                if ('whatsapp' === $method) {
                    $to = "whatsapp:".$phone_number;
                    $from = "whatsapp:".$from_number;
                }

                $url = "https://api.twilio.com/2010-04-01/Accounts/$account_sid/Messages.json";

                $args = array(
                    'method' => 'POST',
                    'headers' => array(
                        'Content-Type'  => 'application/x-www-form-urlencoded',
                        'Authorization' => 'Basic ' . base64_encode($account_sid . ':' . $auth_token),
                    ),
                    'body' => array(
                        'From' => $from,
                        'To'   => $to,
                        'Body' => $message_body,
                    ),
                );
        
                $response = wp_remote_request( $url, $args );

                if ( !is_wp_error( $response ) ) {
                    $body = wp_remote_retrieve_body( $response );

                    if ( $this->is_json( $body ) ) {
                        $body = json_decode( $body, true );
                    }

                    $body = maybe_unserialize( $body );

                    if( 'queued' === $body['status'] ) {
                        // Log that the step is completed.
						$log_payload = array(
							'automation_id' => $automation_id,
							'step_id'       => $step_id,
							'status'        => 'completed',
							'identifier'    => $identifier,
							'email'         => $user_email,
						);
						HelperFunctions::update_log( $log_payload );

						// If there's a next step, enqueue it for processing.
						$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
						HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
						if ( $next_step ) {
							$next_step['data']       = $data['data'];
							$next_step['identifier'] = $data['identifier'];
							do_action(MINT_PROCESS_AUTOMATION, $next_step);
						}
                    } else {
                        return;
                    }
                }
            } catch ( Exception $e ) {
                return;
            }
        }
	}

	public function is_json( $string ) {
		json_decode( $string );

		return ( json_last_error() === JSON_ERROR_NONE );
	}

}
