<?php
/**
 * Automation action class for MRM Autoamtion
 *
 * Class SpecificTimeDelay
 *
 * @package MintMail\App\Internal\Automation\Action
 */

namespace MintMail\App\Internal\Automation\Action;

use DateTime;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;

/**
 * Specific Time Delay action
 *
 * Class SpecificTimeDelay
 *
 * @package MintMail\App\Internal\Automation\Action
 */
class SpecificTimeDelay extends AbstractAutomationAction {

	use Singleton;

	/**
	 * The Action Scheduler instance used by this class.
	 *
	 * @object $action_scheduler An instance of the ActionScheduler class.
	 */
	private $action_scheduler;

	/**
	 * Initializes the class and creates a new instance of the ActionScheduler.
	 *
	 * @return void
	 * @since 1.2.7
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}

	/**
	 * Returns the name of the action.
	 *
	 * @return string The name of the action.
	 * @since  1.2.7
	 */
	public function get_name() {
		return __( 'Specific Time Delay', 'mailmint-pro' );
	}

	/**
	 * Process the Specific Time Delay step for the automation.
	 *
	 * @param array $data The data associated with the automation step.
	 *
	 * @return void
	 * @since 1.2.7
	 */
	public function process( $data ) {
		if ( ! is_array( $data ) ) {
			return;
		}

		// Extract step and next step data.
		list( $step_data, $next_step, $maybe_run ) = $this->get_step_data_and_next_step( $data );

		// Update log with initial payload.
		$this->update_log( $data, 'hold' );

		if ( !$maybe_run ) {
			return;
		}

		// Calculate delay time based on step settings.
		$time = $this->calculate_delay_time( $step_data );

		// Validate delay time.
		$time = $this->validate_delay_time( $time );

		// Update log with completed payload.
		$this->update_log( $data, 'completed' );

		// Update job status and enqueue/schedule next step.
		$this->update_job_and_schedule_next_step( $data, $next_step, $time, $maybe_run );
	}

	/**
	 * Get the step data and the next step for the given automation and step IDs,
	 * and check if it should run.
	 *
	 * @param array $data The automation and step IDs and any associated data.
	 *
	 * @return array An array containing the step data, the next step, and whether or not it should run.
	 * @since 1.2.7
	 */
	private function get_step_data_and_next_step( $data ) {
		$step_data = HelperFunctions::get_step_data( $data['automation_id'], $data['step_id'] );
		$next_step = HelperFunctions::get_next_step( $data['automation_id'], $data['step_id'] );
		$maybe_run = true;

		if ( $next_step && isset( $next_step['key'] ) && ( 'sendMail' === $next_step['key'] || 'sequence' === $next_step['key'] ) ) {
			if ( !HelperFunctions::maybe_user( $data['data']['user_email'] ) ) {
				$next_step = HelperFunctions::get_next_step( $data['automation_id'], $next_step['step_id'] );
				$maybe_run = false;
			}
		}

		return array( $step_data, $next_step, $maybe_run );
	}

	/**
	 * Update the log for the current step with the given status.
	 *
	 * @param array  $data The data related to the current step.
	 * @param string $status The status to set for the current step.
	 *
	 * @return void
	 * @since 1.2.7
	 */
	private function update_log( $data, $status ) {
		$payload = array(
			'automation_id' => $data['automation_id'],
			'step_id'       => $data['step_id'],
			'identifier'    => $data['identifier'],
			'status'        => $status,
			'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
		);

		HelperFunctions::update_log( $payload );
	}

	/**
	 * Calculates the delay time for a given step based on its settings.
	 *
	 * @param array $step_data The step data containing the delay settings.
	 *
	 * @return int The delay time in seconds.
	 * @since 1.2.7
	 */
	private function calculate_delay_time( $step_data ) {
		$delay_settings = isset( $step_data['settings']['specific_delay_settings'] ) ? $step_data['settings']['specific_delay_settings'] : array();
		$time           = $this->calculate_seconds( $delay_settings );

		return $time;
	}

	/**
	 * Validates the delay time and returns the delay time in seconds.
	 *
	 * @param int $time The delay time in seconds.
	 *
	 * @return int The validated delay time in seconds.
	 * @since 1.2.7
	 */
	private function validate_delay_time( $time ) {
		if ( !$this->validate_delay( $time ) ) {
			$time = 0;
		}

		return $time;
	}

	/**
	 * Update the job status and schedule the next step for the automation process.
	 *
	 * @param array   $data The data for the current step.
	 * @param array   $next_step The data for the next step to be executed.
	 * @param integer $time The delay time for the next step in seconds.
	 * @param bool    $maybe_run Indicates whether the next step should be executed or not.
	 *
	 * @return void
	 * @since 1.2.7
	 */
	private function update_job_and_schedule_next_step( $data, $next_step, $time, $maybe_run ) {
		HelperFunctions::update_job( $data['automation_id'], isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );

		if ( ! $next_step ) {
			return;
		}

		$next_step['data']       = isset( $data['data'] ) ? $data['data'] : array();
		$next_step['identifier'] = isset( $data['identifier'] ) ? $data['identifier'] : '';

		if ( isset( $data['double_optin'] ) ) {
			$next_step['double_optin'] = $data['double_optin'];
		}

		$scheduler_data = array( $next_step );

		if ( $this->action_scheduler->hasScheduledAction( MINT_PROCESS_AUTOMATION ) ) {
			return;
		}

		if ( ! $maybe_run ) {
			if ( isset( $data['double_optin'] ) ) {
				return;
			}
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
			return;
		}

		$this->action_scheduler->schedule( time() + $time, MINT_PROCESS_AUTOMATION, $scheduler_data );
	}

	/**
	 * Calculates the number of seconds between
	 * the current date and time and the specified date and time.
	 *
	 * @param array $data An array of delay settings.
	 *
	 * @return int The number of seconds between the two date and time values.
	 * @since 1.2.7
	 * @since 1.14.1 Added a check to ensure the delay time is in the future.
	 * @since 1.18.1 Update the createFromFormat function to use the correct date format.
	 */
	private function calculate_seconds( array $data ): int {
		$time = isset( $data['time'] ) ? $data['time'] : '';
		$time = DateTime::createFromFormat( 'm/d/Y, H:i:s A', $time );

		$date_time_string = $time->format( 'Y-m-d H:i:s' );
        $current_datetime = current_datetime()->format('Y-m-d H:i:s'); //phpcs:ignore

		if ( $current_datetime > $date_time_string ) {
			return false;
		}
		return HelperFunctions::get_time_diff_in_seconds( $current_datetime, $date_time_string );
	}

	/**
	 * Validates a delay in seconds to ensure it's within a reasonable range.
	 *
	 * @param int $seconds The delay in seconds to be validated.
	 *
	 * @return bool True if the delay is valid, false otherwise.
	 * @since 1.2.7
	 */
	private function validate_delay( int $seconds ): bool {
		if ( $seconds <= 0 ) {
			return false;
		}
		if ( $seconds > 2 * YEAR_IN_SECONDS ) {
			return false;
		}
		return true;
	}
}
