<?php
/**
 * Automation action class for MRM Autoamtion
 *
 * Class Sequence
 *
 * @package MintMail\App\Internal\Automation\Action
 */

namespace MintMail\App\Internal\Automation\Action;

use MailMint\App\Helper;
use Error;
use Mint\MRM\DataBase\Models\CampaignEmailBuilderModel;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataBase\Models\EmailModel;
use Mint\MRM\Internal\Parser\Parser;
use MintMail\App\Internal\Automation\Automation_Connector;
use Mint\Mrm\Internal\Traits\Singleton;
use Mint\MRM\Utilites\Helper\Email;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;
use MintMailPro\Mint_Pro_Helper;
use MRM\Common\MrmCommon;

/**
 * Sequence
 *
 * Class Sequence
 *
 * @package MintMail\App\Internal\Automation\Action
 */
class Sequence extends AbstractAutomationAction {

	use Singleton;

	/**
	 * Action scheduler.
	 *
	 * @var object $action_scheduler
	 */
	private $action_scheduler;
	/**
	 * Initialization
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}


	/**
	 * Get action name
	 *
	 * @return String
	 * @since  1.0.0
	 */
	public function get_name() {
		return __( 'Sequence', 'mailmint-pro' );
	}

	/**
	 * Process.
	 *
	 * @param array $data Get All data.
	 */
	public function process( $data ) {
		if ( $data ) {
			$step_id = $data['step_id'];

			$user_email   = !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
			$contact_id   = HelperFunctions::get_contact_id_by_broadcast_table( $user_email );
			$post_id      = isset( $data['data']['post_id'] ) ? $data['data']['post_id'] : '';
			$order_id	  = isset( $data['data']['order_id'] ) ? $data['data']['order_id'] : '';
			$abandoned_id = isset( $data['data']['abandoned_id']) ? $data['data']['abandoned_id'] : '';
			$payment_id   = isset( $data['data']['payment_id']) ? $data['data']['payment_id'] : '';
			$subscription_id = isset($data['data']['subscription_id']) ? $data['data']['subscription_id'] : '';
			$wishlist_id 	 = isset($data['data']['wishlist_id']) ? $data['data']['wishlist_id'] : '';
			$user_membership_id = isset($data['data']['user_membership_id']) ? $data['data']['user_membership_id'] : '';
			$product_id   = isset( $data['data']['product_id'] ) ? $data['data']['product_id'] : '';
			$ld_course_id	 = isset($data['data']['ld_course_id']) ? $data['data']['ld_course_id'] : '';
			$ld_quiz_id		 = isset($data['data']['ld_quiz_id']) ? $data['data']['ld_quiz_id'] : '';
			$ld_lesson_id	 = isset($data['data']['ld_lesson_id']) ? $data['data']['ld_lesson_id'] : '';
			$ld_group_id	 = isset($data['data']['ld_group_id']) ? $data['data']['ld_group_id'] : '';
			$ld_topic_id	 = isset($data['data']['ld_topic_id']) ? $data['data']['ld_topic_id'] : '';
			$booking_id 	 = isset($data['data']['booking_id']) ? $data['data']['booking_id'] : '';

			// Get contact and merge meta fields with contact fields.
			$contact = ContactModel::get( $contact_id );
			if (isset($contact['meta_fields']) && is_array($contact['meta_fields'])) {
				$contact = array_merge($contact, $contact['meta_fields']);
				unset($contact['meta_fields']);
			}

			$log_payload = array(
				'automation_id' => $data['automation_id'],
				'step_id'       => $step_id,
				'status'        => 'hold',
				'identifier'    => $data['identifier'],
				'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
			);
			HelperFunctions::update_log( $log_payload );
			$default_business_settings = MrmCommon::business_settings_default_configuration();
			$business_settings         = get_option( '_mrm_business_basic_info_setting', $default_business_settings );
			$business_settings         = is_array( $business_settings ) && ! empty( $business_settings ) ? $business_settings : $default_business_settings;

			// Get email settings from the options table.
			$global_email_settings = get_option( '_mrm_email_settings', Email::default_email_settings() );

			// Prepare replay to information.
			$reply_name  = isset( $global_email_settings['reply_name'] ) ? $global_email_settings['reply_name'] : '';
			$reply_email = isset( $global_email_settings['reply_email'] ) ? $global_email_settings['reply_email'] : '';

			$step_data         = HelperFunctions::get_step_data( $data['automation_id'], $data['step_id'] );
			$next_step         = HelperFunctions::get_next_step( $data['automation_id'], $data['step_id'] );
			$prev_step         = HelperFunctions::get_prev_step( $data['automation_id'], $data['step_id'] );
			$sequence_settings = isset( $step_data['settings']['sequence_settings'] ) ? $step_data['settings']['sequence_settings'] : array();
			if ( !empty( $sequence_settings['id'] ) ) {
				$id = $sequence_settings['id'];
				/**
				 * Fires before sending an email as part of an automation process.
				 *
				 * @since 1.0.0
				 *
				 * @param string $automation_id The ID of the automation.
				 * @param string $user_email The email address of the user receiving the email.
				 */
				do_action( 'mailmint_before_automation_send_mail', $data['automation_id'], $data['data']['user_email'] );
				if ( HelperFunctions::maybe_user( $data['data']['user_email'] ) ) {
					$emails = HelperFunctions::get_emails_by_campaign_id( $id );
					if ( is_array( $emails ) ) {
						$is_first_email = false;
						$i              = 1;
						$delay_time     = 0;
						foreach ( $emails as $key => $email ) {
							$from       = '';
							$email_data = HelperFunctions::get_email_body_by_email_id( $email['id'] );
							$email_body = isset( $email_data['email_body'] ) ? html_entity_decode( $email_data['email_body'] ) : '';

							$domain_link = get_site_url();
							$rand_hash   = MrmCommon::get_rand_email_hash( $data['data']['user_email'], $data['automation_id'] );
							$track_url   = add_query_arg(
								array(
									'mint'  => 1,
									'route' => 'open',
									'hash'  => $rand_hash,
								),
								$domain_link
							);
							$image_url  = "<img src= $track_url />";
							$email_body = $email_body . $image_url;
							$email_body = Helper::replace_url( $email_body, $rand_hash );
							$email_body = Parser::parse(
								$email_body,
								$contact,
								$post_id,
								$order_id,
								array(
									'abandoned_id'       => $abandoned_id,
									'edd_payment_id'     => $payment_id,
									'subscription_id'    => $subscription_id,
									'product_id'         => $product_id,
									'user_membership_id' => $user_membership_id,
									'wishlist_id'        => $wishlist_id,
									'ld_course_id'       => $ld_course_id,
									'ld_quiz_id'         => $ld_quiz_id,
									'ld_lesson_id'       => $ld_lesson_id,
									'ld_group_id'        => $ld_group_id,
									'ld_topic_id'        => $ld_topic_id,
									'booking_id'         => $booking_id
								)
							);

							// Prepare email header information.
							$header_data = array(
								'reply_name'  => ! empty( $email['reply_name'] ) ? $email['reply_name'] : '',
								'reply_email' => ! empty( $email['reply_email'] ) ? $email['reply_email'] : '',
								'from_email'  => ! empty( $email['sender_email'] ) ? $email['sender_email'] : '',
								'from_name'   => ! empty( $email['sender_name'] ) ? $email['sender_name'] : '',
							);

							$unsub_url = Helper::get_unsubscribed_url( $rand_hash );
							$headers   = Email::get_mail_header( $header_data, $unsub_url );

							$preview   = isset( $email['email_preview_text'] ) ? $email['email_preview_text'] : '';
							$preview   = Parser::parse(
								$preview,
								$contact,
								$post_id,
								$order_id,
								array(
									'abandoned_id'       => $abandoned_id,
									'edd_payment_id'     => $payment_id,
									'subscription_id'    => $subscription_id,
									'product_id'         => $product_id,
									'user_membership_id' => $user_membership_id,
									'wishlist_id'        => $wishlist_id,
									'ld_course_id'       => $ld_course_id,
									'ld_quiz_id'         => $ld_quiz_id,
									'ld_lesson_id'       => $ld_lesson_id,
									'ld_group_id'        => $ld_group_id,
									'ld_topic_id'        => $ld_topic_id,
									'booking_id'         => $booking_id
								)
							);
							$headers[] = 'X-PreHeader: ' . $preview;

							// Add preview text on the email body.
							$email_body = Email::inject_preview_text_on_email_body( $preview, $email_body );
							$email_body = Helper::replace_dynamic_coupon($email_body, $data['data']['user_email']);

							$email_subject  = isset( $email['email_subject'] ) ? $email['email_subject'] : '';
							$email_subject  = Parser::parse(
								$email_subject,
								$contact,
								$post_id,
								$order_id,
								array(
									'abandoned_id'       => $abandoned_id,
									'edd_payment_id'     => $payment_id,
									'subscription_id'    => $subscription_id,
									'product_id'         => $product_id,
									'user_membership_id' => $user_membership_id,
									'wishlist_id'        => $wishlist_id,
									'ld_course_id'       => $ld_course_id,
									'ld_quiz_id'         => $ld_quiz_id,
									'ld_lesson_id'       => $ld_lesson_id,
									'ld_group_id'        => $ld_group_id,
									'ld_topic_id'        => $ld_topic_id,
									'booking_id'         => $booking_id
								)
							);
							$scheduler_data = array(
								'body'           => $email_body,
								'subject'        => $email_subject,
								'receiver_email' => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
								'header'         => $headers,
							);

							// Process URL for lead-magnet tracking if MailMint Pro is active.
							if (MrmCommon::is_mailmint_pro_active()) {
								$scheduler_data['body'] = Mint_Pro_Helper::replace_automatic_latest_content($scheduler_data['body'], get_post_type($post_id));
								if (MrmCommon::is_mailmint_pro_version_compatible('1.15.1')) {
									$scheduler_data['body'] = Mint_Pro_Helper::process_lead_magnet_tracking($scheduler_data['body'], $user_email);
								}
							}

							if ( 1 === $i ) {
								$is_sent = $this->send_message( $scheduler_data );
								$payload = array(
									'automation_id' => $data['automation_id'],
									'step_id'       => $data['step_id'],
									'email_id'      => $email['id'],
									'email_type'    => 'automation',
									'email_address' => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
									'contact_id'    => $contact_id,
									'email_hash'    => $rand_hash,
									'created_at'    => current_time( 'mysql' ),
									'updated_at'    => current_time( 'mysql' ),
								);
								if ( $is_sent ) {
									$payload['status'] = 'sent';
								} else {
									$payload['status'] = 'failed';
								}
								EmailModel::insert( $payload );
								if ( count( $emails ) === 1 ) {
									$log_payload = array(
										'automation_id' => $data['automation_id'],
										'step_id'       => $step_id,
										'status'        => 'completed',
										'identifier'    => $data['identifier'],
										'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
									);
									HelperFunctions::update_log( $log_payload );
								}
							} else {
								if ( isset( $email['delay_count'] ) && isset( $email['delay_value'] ) ) {
									$delay_settings = array(
										'delay' => $email['delay_count'],
										'unit'  => $email['delay_value'],
									);
									$time           = $this->calculate_seconds( $delay_settings );
									$delay_time     = $delay_time + $time;

									$log_payload = array(
										'automation_id' => $data['automation_id'],
										'step_id'       => $data['step_id'],
										'status'        => 'hold',
										'identifier'    => $data['identifier'],
										'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
									);
									HelperFunctions::update_log( $log_payload );

									$background_data = array(
										array(
											'email_id'   => $email['id'],
											'user_email' => $data['data']['user_email'],
											'automation_id' => $data['automation_id'],
											'identifier' => $data['identifier'],
											'step_id'    => $data['step_id'],
											'data'       => $data,
										),
									);
									if ( isset( $data['data']['abandoned_id'] ) ) {
										$background_data[0]['abandoned_id'] = $data['data']['abandoned_id'];
									}
									if ( !$this->action_scheduler->hasScheduledAction( MINT_PROCESS_SEQUENCE ) ) {
										$this->action_scheduler->schedule( time() + $delay_time, MINT_PROCESS_SEQUENCE, $background_data );
									}
								}
							}
							$i++;
						}
					} else {
						$log_payload = array(
							'automation_id' => $data['automation_id'],
							'step_id'       => $step_id,
							'status'        => 'fail',
							'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
						);
						HelperFunctions::update_log( $log_payload );
					}
				}
				HelperFunctions::update_job( $data['automation_id'], isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
			}
			if ( !isset( $data['double_optin'] ) ) {
				$next_step = HelperFunctions::get_next_step( $data['automation_id'], $data['step_id'] );
				HelperFunctions::update_job( $data['automation_id'], isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
				if ( $next_step ) {
					$next_step['data']       = $data['data'];
					$next_step['identifier'] = $data['identifier'];
					do_action(MINT_PROCESS_AUTOMATION, $next_step);
				}
			}
		}
	}


	/**
	 * Calculate delay time and convert it into second
	 *
	 * @param array $data get Data .
	 *
	 * @return int
	 */
	private function calculate_seconds( $data ) {
		$type  = isset( $data['unit'] ) ? strtoupper( $data['unit'] ) : 'MINUTES';
		$delay = isset( $data['delay'] ) ? $data['delay'] : 0;
		switch ( $type ) {
			case 'MINUTES':
				return $delay * MINUTE_IN_SECONDS;
			case 'HOURS':
				return $delay * HOUR_IN_SECONDS;
			case 'DAYS':
				return $delay * DAY_IN_SECONDS;
			case 'WEEKS':
				return $delay * WEEK_IN_SECONDS;
			default:
				return 0;
		}
	}


	/**
	 * Send a message to contact
	 *
	 * @param mixed $data Single message object.
	 * @return bool
	 * @since 1.0.0
	 */
	public function send_message( $data ) {
		$to      = isset( $data['receiver_email'] ) ? $data['receiver_email'] : '';
		$subject = isset( $data['subject'] ) ? $data['subject'] : 'Welcome to WPF';
		$body    = isset( $data['body'] ) ? html_entity_decode( $data['body'] ) : '';
		$body    = str_replace( '</html>', CampaignEmailBuilderModel::get_email_footer_watermark() . '</html>', $body );
		$body    = Helper::modify_email_for_rtl( $body );
		$headers = isset( $data['header'] ) ? $data['header'] : '';
		if ( $to ) {
			return MM()->mailer->send( $to, $subject, $body, $headers );
		}
		return false;
	}

}
