<?php
/**
 * Pabbly Connect action class for Mail Mint Automation.
 *
 * Class PabblyConnectSendData
 *
 * @package MintMail\App\Internal\Automation\Action
 * @namespace MintMail\App\Internal\Automation\Action
 * @author [getwpfunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-05-20 11:03:17
 * @modify date 2024-05-20 11:03:17
 */

namespace MintMail\App\Internal\Automation\Action;

use Mint\MRM\DataBase\Models\ContactModel;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;

/**
 * PabblyConnectSendData action
 *
 * Class PabblyConnectSendData
 *
 * @package MintMail\App\Internal\Automation\Action
 * @since 1.12.0
 */
class PabblyConnectSendData extends AbstractAutomationAction {

	use Singleton;

	/**
	 * Holds an instance of the ActionScheduler class.
	 *
	 * @var ActionScheduler
	 *
	 * @access private
	 * @since 1.12.0
	 */
	private $action_scheduler;

	/**
	 * Class constructor.
	 *
	 * @access public
	 * @since 1.12.0
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}

	/**
	 * Get the name of the action.
	 *
	 * Description: Retrieves the localized name for the "Send Message" action.
	 *
	 * @access public
	 * @return string The localized name of the action.
	 * @since 1.12.0
	 */
	public function get_name() {
		return 'Pabbly Connect';
	}

	/**
	 * Process automation step for creating a coupon.
	 *
	 * Description: Processes the automation step for creating a coupon. Updates logs, creates the coupon,
	 *              and handles the next step in the automation.
	 *
	 * @access public
	 * @param array $data The data for the automation step.
	 * @since 1.8.0
	 */
	public function process( $data ) {
		// Check if essential data is provided.
		if ( ! isset( $data['automation_id'] ) || ! isset( $data['step_id'] ) ) {
			return;
		}

		// Extract necessary data from the input.
		$user_email    = isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$step_id       = isset( $data['step_id'] ) ? $data['step_id'] : '';
		$identifier    = isset( $data['identifier'] ) ? $data['identifier'] : '';

		// Log that the step is in a 'hold' status.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'hold',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// Retrieve settings data for the current step.
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );
		// Check if coupon settings are available.
		if ( ! isset( $step_data['settings']['pabbly_data'] ) || empty( $step_data['settings']['pabbly_data'] ) ) {
			return;
		}

		// Get the contact details.
		$contact    = ContactModel::get_contact_by_email( $user_email );
		$contact_id = isset( $contact['id'] ) ? $contact['id'] : '';
		$contact    = ContactModel::get( $contact_id );

		// Prepare data for creating the coupon.
		$pabbly_data   = $step_data['settings']['pabbly_data'];
		$param_fields  = isset( $pabbly_data['data'] ) ? $pabbly_data['data'] : array();
		$endpoint_url  = isset( $pabbly_data['link'] ) ? $pabbly_data['link'] : '';
		$pabbly_fields = array();

		foreach ( $param_fields as $field ) {
			$pabbly_fields[$field['key']] = preg_replace_callback( '/{{(.*?)}}/', function( $matches ) use ( $contact ) {
				$key = ltrim( strstr( $matches[1], '.' ), '.' );
				if ( isset( $contact[$key] ) ) {
					return $contact[$key];
				} elseif ( isset( $contact['meta_fields'][$key] ) ) {
					return $contact['meta_fields'][$key];
				}
				return '';
			}, $field['value'] );
		}

		$response = MM()->wp_remote_request_handler->make_wp_requests( $endpoint_url, $pabbly_fields, array(), 2 );

		if( isset( $response['response'] ) && 200 === $response['response'] ) {
			// Log that the step is completed.
			$log_payload = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'status'        => 'completed',
				'identifier'    => $identifier,
				'email'         => $user_email,
			);
			HelperFunctions::update_log( $log_payload );
		} else {
			// Log that the step is completed.
			$log_payload = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'status'        => 'fail',
				'identifier'    => $identifier,
				'email'         => $user_email,
			);
			HelperFunctions::update_log( $log_payload );
		}

		// If there's a next step, enqueue it for processing.
		$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
		HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
		if ( $next_step ) {
			$next_step['data']       = $data['data'];
			$next_step['identifier'] = $data['identifier'];
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
		}
	}
}
