<?php
/**
 * Automation action class for MRM Automation
 *
 * Class CreateWordPressUser
 *
 * @package MintMail\App\Internal\Automation\Action
 */

namespace MintMail\App\Internal\Automation\Action;

use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\Internal\Parser\Parser;


/**
 * Create WordPress User action
 *
 * Class CreateWordPressUser
 *
 * @package MintMail\App\Internal\Automation\Action
 */
class CreateWordPressUser extends AbstractAutomationAction {

    use Singleton;

    /**
     * Action scheduler.
     *
     * @var $action_scheduler
     */
    private $action_scheduler;

    /**
     * Initialization
     */
    public function __construct() {
        $this->action_scheduler = new ActionScheduler();
    }

    /**
     * Get action name
     *
     * @return string
     * @since 1.18.0
     */
    public function get_name() {
        return __( 'Create WordPress User', 'mailmint-pro' );
    }

    /**
     * Process automation action.
     *
     * @param array $data Input data for automation.
     * @since 1.18.0
     */
	public function process($data) {
		// Check if essential data is provided.
		if ( ! isset( $data['automation_id'] ) || ! isset( $data['step_id'] ) ) {
			return;
		}

		// Extract necessary data from the input.
		$user_email    = isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$step_id       = isset( $data['step_id'] ) ? $data['step_id'] : '';
		$identifier    = isset( $data['identifier'] ) ? $data['identifier'] : '';

		// Log that the step is in a 'hold' status.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'hold',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );
	
		if (get_user_by('email', $user_email)) {
			// If there's a next step, enqueue it for processing.
			$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
			HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
			if ( $next_step ) {
				$next_step['data']       = $data['data'];
				$next_step['identifier'] = $data['identifier'];
				do_action(MINT_PROCESS_AUTOMATION, $next_step);
			}
		}
	
		// Retrieve settings data for the current step.
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );

		// Check if coupon settings are available.
		if ( ! isset( $step_data['settings']['create_wordpress_user_settings'] ) || empty( $step_data['settings']['create_wordpress_user_settings'] ) ) {
			return;
		}
	
		$settings          = $step_data['settings']['create_wordpress_user_settings'];
		$role              = isset( $settings['role'] ) ? $settings['role'] : get_option('default_role');
		$auto_pass         = isset( $settings['auto_pass'] ) ? $settings['auto_pass'] : true;
		$custom_pass       = isset( $settings['custom_pass'] ) ? $settings['custom_pass'] : '';
		$custom_username   = isset( $settings['custom_username'] ) ? $settings['custom_username'] : '';
		$notification_mail = isset( $settings['notification_mail'] ) ? $settings['notification_mail'] : false;

		$contact    = ContactModel::get_contact_by_email( $user_email );
		$contact_id = isset( $contact['id'] ) ? $contact['id'] : '';
		$contact    = ContactModel::get( $contact_id );
		if (isset($contact['meta_fields']) && is_array($contact['meta_fields'])) {
			$contact = array_merge($contact, $contact['meta_fields']);
			unset($contact['meta_fields']);
		}

		if ( $auto_pass || empty($custom_pass)) {
            $password = wp_generate_password();
        } else {
            $password = Parser::parse( $custom_pass, $contact );
        }

		$username = $user_email;
        if ( !empty( $custom_username ) ) {
            $custom_username = Parser::parse( $custom_username, $contact );
            if( ! username_exists( $custom_username ) ) {
                $username = $custom_username;
            }
        }
	
		$user_id = wp_create_user( sanitize_user( $username ), $password, $user_email );
		if ( is_wp_error( $user_id ) ) {
			return false;
		}

		if ( $role ) {
            $user = new \WP_User($user_id);
            $user->set_role( $role );
        }

		if ( $notification_mail ) {
            wp_send_new_user_notifications($user_id, 'user');
        }
	
		// Log that the step is completed.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'completed',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// If there's a next step, enqueue it for processing.
		$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
		HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
		if ( $next_step ) {
			$next_step['data']       = $data['data'];
			$next_step['identifier'] = $data['identifier'];
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
		}
	}
}
