<?php
/**
 * Automation action class for MRM Autoamtion
 *
 * Class CreateUser
 *
 * @package MintMail\App\Internal\Automation\Action
 */

namespace MintMail\App\Internal\Automation\Action;

use Mint\MRM\Admin\API\Controllers\MessageController;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;
use Mint\MRM\DataStores\ContactData;
use Mint\MRM\DataBase\Models\ContactModel;

/**
 * CreateUser action
 *
 * Class CreateUser
 *
 * @package MintMail\App\Internal\Automation\Action
 */
class CreateUser extends AbstractAutomationAction {

	use Singleton;
	/**
	 * Action scheduler.
	 *
	 * @var $action_scheduler
	 */
	private $action_scheduler;
	/**
	 * Initialization
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}


	/**
	 * Get action name
	 *
	 * @return String
	 * @since  1.0.0
	 */
	public function get_name() {
		return __( 'Create User', 'mailmint-pro' );
	}


	/**
	 * Process the data.
	 *
	 * @param array $data The data to process.
	 * @since 1.4.1
	 */
	public function process( $data ) {
		if ( !$data ) {
			return;
		}

		// Update log with initial payload.
		$this->update_log( $data, 'hold' );

		$email = $this->get_email_from_data( $data );
		if ( $email ) {
			$this->process_contact( $data, $email );
		}

		// Update log with completed payload.
		$this->update_log( $data, 'completed' );

		$next_step = HelperFunctions::get_next_step( $data['automation_id'], $data['step_id'] );
		$this->update_job( $data['automation_id'], $next_step );
		$this->enqueue_next_step( $next_step, $data );
	}

	/**
	 * Update the log for the current step with the given status.
	 *
	 * @param array  $data The data related to the current step.
	 * @param string $status The status to set for the current step.
	 *
	 * @return void
	 * @since 1.2.7
	 */
	private function update_log( $data, $status ) {
		$payload = array(
			'automation_id' => !empty( $data['automation_id'] ) ? $data['automation_id'] : 0,
			'step_id'       => !empty( $data['step_id'] ) ? $data['step_id'] : '',
			'identifier'    => !empty( $data['identifier'] ) ? $data['identifier'] : '',
			'status'        => $status,
			'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
		);

		HelperFunctions::update_log( $payload );
	}

	/**
	 * Get the email from the data.
	 *
	 * @param array $data The input data.
	 * @return string The email address.
	 * @since 1.4.1
	 */
	public function get_email_from_data( $data ) {
		return isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
	}

	/**
	 * Process the contact data.
	 *
	 * @param array  $data The input data.
	 * @param string $email The email address.
	 * @since 1.4.1
	 */
	public function process_contact( $data, $email ) {
		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$step_id       = isset( $data['step_id'] ) ? $data['step_id'] : '';
		$step_data     = HelperFunctions::get_step_data( $automation_id, $step_id );
		$status        = isset( $step_data['settings']['contact_status_settings']['status'] ) ? $step_data['settings']['contact_status_settings']['status'] : 'pending';
		$user_data     = array(
			'first_name'  => isset( $data['data']['first_name'] ) ? $data['data']['first_name'] : '',
			'last_name'   => isset( $data['data']['last_name'] ) ? $data['data']['last_name'] : '',
			'meta_fields' => array(
				'phone_number' => isset( $data['data']['phone_number'] ) ? $data['data']['phone_number'] : '',
			),
			'status'      => $status,
			'source'      => 'Automation',
		);
		$contact       = new ContactData( $email, $user_data );
		$exist_email   = ContactModel::is_contact_exist( $email );
		if ( !$exist_email && $automation_id && $step_id ) {
			$contact_id = ContactModel::insert( $contact );
			if ( isset( $user_data['status'] ) && 'pending' === $user_data['status'] ) {
				MessageController::get_instance()->send_double_opt_in( $contact_id );
			}
		}
	}


	/**
	 * Update the job.
	 *
	 * @param string $automation_id The automation ID.
	 * @param array  $next_step The next step.
	 * @since 1.4.1
	 */
	public function update_job( $automation_id, $next_step ) {
		$status = !empty( $next_step['step_id'] ) ? 'processing' : 'completed';
		HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, $status );
	}

	/**
	 * Enqueue the next step.
	 *
	 * @param array|null $next_step The next step.
	 * @param array      $data The input data.
	 * @since 1.4.1
	 */
	public function enqueue_next_step( $next_step, $data ) {
		if ( $next_step ) {
			$next_step['data']       = !empty( $data['data'] ) ? $data['data'] : array();
			$next_step['identifier'] = !empty( $data['identifier'] ) ? $data['identifier'] : '';
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
		}
	}



}
