<?php
/**
 * Automatic Coupon Create Class
 *
 * Class CreateCoupon
 *
 * @package MintMail\App\Internal\Automation\Action
 * @namespace MintMail\App\Internal\Automation\Action
 * @author [getwpfunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-01-03 11:03:17
 * @modify date 2024-01-03 11:03:17
 */

namespace MintMail\App\Internal\Automation\Action;

use MailMintPro\App\Utilities\Helper\MintAutomaticCoupon;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;

/**
 * CreateCoupon action
 *
 * Class CreateCoupon
 *
 * @package MintMail\App\Internal\Automation\Action
 * @since 1.8.0
 */
class CreateCoupon extends AbstractAutomationAction {

	use Singleton;

	/**
	 * Holds an instance of the ActionScheduler class.
	 *
	 * @var ActionScheduler
	 *
	 * @access private
	 * @since 1.8.0
	 */
	private $action_scheduler;

	/**
	 * Class constructor.
	 *
	 * @access public
	 * @since 1.8.0
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}

	/**
	 * Get the name of the action.
	 *
	 * Description: Retrieves the localized name for the "Create Coupon" action.
	 *
	 * @access public
	 * @return string The localized name of the action.
	 * @since 1.8.0
	 */
	public function get_name() {
		return __( 'Create Coupon', 'mailmint-pro' );
	}

	/**
	 * Process automation step for creating a coupon.
	 *
	 * Description: Processes the automation step for creating a coupon. Updates logs, creates the coupon,
	 *              and handles the next step in the automation.
	 *
	 * @access public
	 * @param array $data The data for the automation step.
	 * @since 1.8.0
	 */
	public function process( $data ) {
		// Check if essential data is provided.
		if ( ! isset( $data['automation_id'] ) || ! isset( $data['step_id'] ) ) {
			return;
		}

		// Extract necessary data from the input.
		$user_email    = isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$step_id       = isset( $data['step_id'] ) ? $data['step_id'] : '';
		$identifier    = isset( $data['identifier'] ) ? $data['identifier'] : '';

		// Log that the step is in a 'hold' status.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'hold',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// Retrieve settings data for the current step.
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );

		// Check if coupon settings are available.
		if ( ! isset( $step_data['settings']['wc_create_coupon_settings'] ) || empty( $step_data['settings']['wc_create_coupon_settings'] ) ) {
			return;
		}

		// Prepare data for creating the coupon.
		$settings    = $step_data['settings']['wc_create_coupon_settings'];
		$data_to_set = MintAutomaticCoupon::prepare_coupon_data( $user_email, $settings );
		MintAutomaticCoupon::create_coupon( $data_to_set, $automation_id, $step_id );

		// Log that the step is completed.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'completed',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// If there's a next step, enqueue it for processing.
		$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
		HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
		if ( $next_step ) {
			$next_step['data']       = $data['data'];
			$next_step['identifier'] = $data['identifier'];
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
		}
	}

}
