<?php
/**
 * WishlistItems class for checking if a contact has purchased a specific product or not.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * WishlistItems class for checking 
 * if a contact has purchased a specific product or not.
 * 
 * @since 1.2.7
 */
class WishlistItems {

	/**
	 * Checks if any segment ID from the given segments array matches any group ID in the automation data.
	 *
	 * @param array $segments The array of segments to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a segment ID matches a group ID, false otherwise.
	 * @since 1.2.7
	 */
	public function included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		$wishlist_posts = get_posts( array(
			'post_type'   => 'wishlist', // Replace with the actual wishlist custom post type
			'post_status' => 'publish',
			'numberposts' => -1,
			'meta_query'  => array(
				array(
					'key'   => '_wishlist_email',
					'value' => $user_email,
					'compare' => '='
				)
			),
		) );
	
		if ( empty( $wishlist_posts ) ) {
			return false;
		}

        $wishlist_product_ids = array();

		// Loop through wishlists to extract product IDs
		foreach ( $wishlist_posts as $wishlist ) {
			$wishlist_items = get_post_meta( $wishlist->ID, '_wishlist_items', true );

			if ( ! empty( $wishlist_items ) ) {
				$wishlist_items = maybe_unserialize( $wishlist_items );

				if ( is_array( $wishlist_items ) ) {
					foreach ( $wishlist_items as $item ) {
						if ( isset( $item['product_id'] ) ) {
							$wishlist_product_ids[] = $item['product_id'];
						}
					}
				}
			}
		}
		$wishlist_product_ids = array_unique( $wishlist_product_ids );
		$product_ids          = array_values( array_column( $value, 'value' ) );
		// Find the intersection between group IDs and segment IDs.
		$intersect = array_intersect($wishlist_product_ids, $product_ids);

		// If there is at least one common ID, return true.
		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function not_included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		$wishlist_posts = get_posts( array(
			'post_type'   => 'wishlist', // Replace with the actual wishlist custom post type
			'post_status' => 'publish',
			'numberposts' => -1,
			'meta_query'  => array(
				array(
					'key'   => '_wishlist_email',
					'value' => $user_email,
					'compare' => '='
				)
			),
		) );
	
		if ( empty( $wishlist_posts ) ) {
			return false;
		}

        $wishlist_product_ids = array();

		// Loop through wishlists to extract product IDs
		foreach ( $wishlist_posts as $wishlist ) {
			$wishlist_items = get_post_meta( $wishlist->ID, '_wishlist_items', true );

			if ( ! empty( $wishlist_items ) ) {
				$wishlist_items = maybe_unserialize( $wishlist_items );

				if ( is_array( $wishlist_items ) ) {
					foreach ( $wishlist_items as $item ) {
						if ( isset( $item['product_id'] ) ) {
							$wishlist_product_ids[] = $item['product_id'];
						}
					}
				}
			}
		}
		$wishlist_product_ids = array_unique( $wishlist_product_ids );
		$product_ids          = array_values( array_column( $value, 'value' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $wishlist_product_ids, $product_ids );

		if ( !empty( $diff ) ) {
			return true;
		}

		return false;
	}
}
