<?php
/**
 * SubscriptionItems class for checking if a contact has purchased a specific product or not.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * SubscriptionItems class for checking 
 * if a contact has purchased a specific product or not.
 * 
 * @since 1.2.7
 */
class SubscriptionItems {

	/**
	 * Checks if any segment ID from the given segments array matches any group ID in the automation data.
	 *
	 * @param array $segments The array of segments to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a segment ID matches a group ID, false otherwise.
	 * @since 1.2.7
	 */
	public function included_in( array $value, array $automation_data ): bool {
		$subscription_id = isset( $automation_data['data']['subscription_id'] ) ? $automation_data['data']['subscription_id'] : '';

		if ( empty( $subscription_id ) ) {
			return false;
		}

		$subscription = wcs_get_subscription($subscription_id);
		
		// Initialize an array to store product IDs
		$subs_product_ids = [];

		// Loop through subscription items
		foreach ( $subscription->get_items() as $item ) {
			// Get the product ID
			$product_id = $item->get_product_id();
	
			// Add the product ID to the array
			$subs_product_ids[] = $product_id;
		}

		$product_ids = array_values( array_column( $value, 'value' ) );

		// Find the intersection between group IDs and segment IDs.
		$intersect = array_intersect($subs_product_ids, $product_ids);

		// If there is at least one common ID, return true.
		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function not_included_in( array $value, array $automation_data ): bool {
		$subscription_id = isset( $automation_data['data']['subscription_id'] ) ? $automation_data['data']['subscription_id'] : '';

		if ( empty( $subscription_id ) ) {
			return false;
		}

		$subscription = wcs_get_subscription($subscription_id);
		
		// Initialize an array to store product IDs
		$subs_product_ids = [];

		// Loop through subscription items
		foreach ( $subscription->get_items() as $item ) {
			// Get the product ID
			$product_id = $item->get_product_id();
	
			// Add the product ID to the array
			$subs_product_ids[] = $product_id;
		}

		$product_ids = array_values( array_column( $value, 'value' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $subs_product_ids, $product_ids );

		if ( !empty( $diff ) ) {
			return true;
		}

		return false;
	}
}
