<?php
/**
 * HasActivePlans class for checking if a contact has purchased a specific product or not.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use WC_Memberships_Membership_Plan;
use WC_Memberships_User_Membership;

/**
 * HasActivePlans class for checking 
 * if a contact has purchased a specific product or not.
 * 
 * @since 1.2.7
 */
class HasActivePlans {

	/**
	 * Checks if any segment ID from the given segments array matches any group ID in the automation data.
	 *
	 * @param array $segments The array of segments to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a segment ID matches a group ID, false otherwise.
	 * @since 1.2.7
	 */
	public function included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

        if ( empty( $user_email ) ) {
            return false;
        }

        // Check if a user exists with the given email
        $user = get_user_by('email', $user_email);

		if (!$user) {
			return false;
		}

		$user_id     = $user->ID;
		$memberships = wc_memberships_get_user_memberships($user_id);

		if (empty($memberships)) {
			return false;
		}

		$plans = array();
		foreach ( $memberships as $membership ) {
			/** @var WC_Memberships_User_Membership $membership */
			if ( ! $membership instanceof WC_Memberships_User_Membership ) {
				continue;
			}
			$plan = $membership->get_plan();
			if ( ! $plan instanceof WC_Memberships_Membership_Plan ) {
				continue;
			}
			$plans[] = $plan->get_id();
		}

		/** empty $plans then return false **/
		if ( empty( $plans ) ) {
			return false;
		}
    
		$product_ids = array_values( array_column( $value, 'value' ) );

		// Find the intersection between group IDs and segment IDs.
		$intersect = array_intersect($plans, $product_ids);

		// If there is at least one common ID, return true.
		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function not_included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

        if ( empty( $user_email ) ) {
            return false;
        }

        // Check if a user exists with the given email
        $user = get_user_by('email', $user_email);

		if (!$user) {
			return false;
		}

		$user_id     = $user->ID;
		$memberships = wc_memberships_get_user_memberships($user_id);

		if (empty($memberships)) {
			return false;
		}

		$plans = array();
		foreach ( $memberships as $membership ) {
			/** @var WC_Memberships_User_Membership $membership */
			if ( ! $membership instanceof WC_Memberships_User_Membership ) {
				continue;
			}
			$plan = $membership->get_plan();
			if ( ! $plan instanceof WC_Memberships_Membership_Plan ) {
				continue;
			}
			$plans[] = $plan->get_id();
		}

		/** empty $plans then return false **/
		if ( empty( $plans ) ) {
			return false;
		}
    
		$product_ids = array_values( array_column( $value, 'value' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $plans, $product_ids );

		if ( !empty( $diff ) ) {
			return true;
		}

		return false;
	}
}
