<?php
/**
 * UsedCoupons class for checking if a contact has purchased a specific product or not.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * UsedCoupons class for checking 
 * if a contact has purchased a specific product or not.
 * 
 * @since 1.2.7
 */
class UsedCoupons {

	/**
	 * Checks if any segment ID from the given segments array matches any group ID in the automation data.
	 *
	 * @param array $segments The array of segments to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a segment ID matches a group ID, false otherwise.
	 * @since 1.2.7
	 */
	public function included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		global $wpdb;
        $table_name = $wpdb->prefix . 'mint_wc_customers';

        $purchased_coupons = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT used_coupons FROM $table_name WHERE email_address = %s",
                $user_email
            )
        );

        if (empty($purchased_coupons)) {
            return false;
        }
    
        // Decode the JSON string to an array
        $purchased_coupons_array = json_decode($purchased_coupons, true);
		$product_ids             = array_values( array_column( $value, 'value' ) );
		// Find the intersection between group IDs and segment IDs.
		$intersect = array_intersect($purchased_coupons_array, $product_ids);

		// If there is at least one common ID, return true.
		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function not_included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		global $wpdb;
        $table_name = $wpdb->prefix . 'mint_wc_customers';

        $purchased_coupons = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT used_coupons FROM $table_name WHERE email_address = %s",
                $user_email
            )
        );

        if (empty($purchased_coupons)) {
            return false;
        }
    
        // Decode the JSON string to an array
        $purchased_coupons_array = json_decode($purchased_coupons, true);
		$product_ids              = array_values( array_column( $value, 'value' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $purchased_coupons_array, $product_ids );

		if ( !empty( $diff ) ) {
			return true;
		}

		return false;
	}
}
