<?php
/**
 * TotalOrderCount class for checking the total order count of a user.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * TotalOrderCount class for checking the total order count of a user.
 *
 * @since 1.17.2
 */
class TotalOrderCount {

    /**
     * Fetch the total order count for a given user email.
     *
     * @param string $user_email The email address of the user.
     * 
     * @return int|null The total order count or null if not found.
     * @since 1.17.2
     */
    private function get_total_order_count($user_email) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'mint_wc_customers';

        $total_order = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT total_order_count FROM $table_name WHERE email_address = %s",
                $user_email
            )
        );

        return $total_order !== null ? (int)$total_order : null;
    }

    /**
     * Check if the total order count is greater than the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return bool True if the total order count is greater than the value, false otherwise.
     * @since 1.17.2
     */
    public function greater_than($value, $automation_data): bool {
        $user_email = $automation_data['data']['user_email'] ?? '';

        if (empty($user_email)) {
            return false;
        }

        $total_order = $this->get_total_order_count($user_email);

        if ($total_order === null) {
            return false;
        }

        return $total_order > (int)$value;
    }

    /**
     * Check if the total order count is less than the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return bool True if the total order count is less than the value, false otherwise.
     * @since 1.17.2
     */
    public function less_than($value, $automation_data): bool {
        $user_email = $automation_data['data']['user_email'] ?? '';

        if (empty($user_email)) {
            return false;
        }

        $total_order = $this->get_total_order_count($user_email);

        if ($total_order === null) {
            return false;
        }

        return $total_order < (int)$value;
    }

    /**
     * Check if the total order count is equal to the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return bool True if the total order count is equal to the value, false otherwise.
     * @since 1.17.2
     */
    public function equal($value, $automation_data): bool {
        $user_email = $automation_data['data']['user_email'] ?? '';

        if (empty($user_email)) {
            return false;
        }

        $total_order = $this->get_total_order_count($user_email);

        if ($total_order === null) {
            return false;
        }

        return $total_order === (int)$value;
    }

    /**
     * Check if the total order count is not equal to the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return bool True if the total order count is not equal to the value, false otherwise.
     * @since 1.17.2
     */
    public function does_not_equal($value, $automation_data): bool {
        $user_email = $automation_data['data']['user_email'] ?? '';

        if (empty($user_email)) {
            return false;
        }

        $total_order = $this->get_total_order_count($user_email);

        if ($total_order === null) {
            return false;
        }

        return $total_order !== (int)$value;
    }
}