<?php

/**
 * Email Condition Class
 *
 * Provides various conditions to evaluate email strings for automation logic.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-12-04 09:03:17
 * @modify date 2024-12-10 09:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;

use DateTime;

/**
 * Class Email
 *
 * Provides various conditions to evaluate email strings for automation logic.
 * Includes methods to check if an email contains, equals, starts with, or ends with a given value.
 * 
 * @since 1.17.1
 */
class LastOrderDate {

	/**
	 * Checks if the contact's last order date is after a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's last order date is after the specified date, false otherwise.
	 * @since 1.17.0
	 */
	public function after( $value, $automation_data ) {
		$date = $this->get_contact_last_order_date( $automation_data );
		return $this->compare_date( $date, $value, '>' );
	}

	/**
	 * Checks if the contact's last order date is before a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's last order date is before the specified date, false otherwise.
	 * @since 1.17.0
	 */
	public function before( $value, $automation_data ) {
		$date = $this->get_contact_last_order_date( $automation_data );
		return $this->compare_date( $date, $value, '<' );
	}

	/**
	 * Checks if the contact's last order date matches a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's last order date matches the specified date, false otherwise.
	 * @since 1.17.0
	 */
	public function in_the_date( $value, $automation_data ) {
		$date = $this->get_contact_last_order_date( $automation_data );
		return $this->compare_date( $date, $value, '=' );
	}

	/**
	 * Gets the contact's last order date based on their email.
	 *
	 * @param array $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return string The contact's last order date in 'Y-m-d' format.
	 * @since 1.17.0
	 */
	public function get_contact_last_order_date( $automation_data ) {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

        global $wpdb;
        $table_name = $wpdb->prefix . 'mint_wc_customers';
		return $wpdb->get_var(
            $wpdb->prepare(
                "SELECT l_order_date FROM $table_name WHERE email_address = %s",
                $user_email
            )
        );
	}


	/**
	 * Compares a last order date with a specified date using a given comparison operator.
	 *
	 * @param string $date       The last order date in 'Y-m-d' format.
	 * @param string $value      The date to compare with (e.g., '2023-07-17').
	 * @param string $comparison_operator The comparison operator ('<', '>', '=').
	 *
	 * @access public
	 *
	 * @return bool True if the comparison is successful, false otherwise.
	 * @since 1.17.0
	 */
	public function compare_date( $date, $value, $comparison_operator ) {
		$formatted_date  = (new DateTime($date))->format('Y-m-d');
		$formatted_value = (new DateTime($value))->format('Y-m-d');
		return !empty($formatted_date) && version_compare($formatted_date, $formatted_value, $comparison_operator);
	}
}
