<?php
/**
 * ProductsInCurrentOrder class for checking if a contact has purchased a specific product or not.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * ProductsInCurrentOrder class for checking 
 * if a contact has purchased a specific product or not.
 * 
 * @since 1.2.7
 */
class ProductsInCurrentOrder {

	/**
	 * Checks if any segment ID from the given segments array matches any group ID in the automation data.
	 *
	 * @param array $segments The array of segments to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a segment ID matches a group ID, false otherwise.
	 * @since 1.2.7
	 */
	public function included_in( array $value, array $automation_data ): bool {
		$order_id = isset( $automation_data['data']['order_id'] ) ? $automation_data['data']['order_id'] : '';

		if ( empty( $order_id ) ) {
			return false;
		}

		$order = wc_get_order($order_id);
		$items = $order->get_items();
		
		if( empty( $items ) ){
			$order = isset( $automation_data['data']['order'] ) ? $automation_data['data']['order'] : '';
			$items = $order->get_items();
		}

		// Get the product IDs from the items.
		$current_product_ids = [];

		foreach ($items as $item) {
			$product_id            = $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
			$current_product_ids[] = $product_id;
		}

        if (empty($current_product_ids)) {
            return false;
        }
    
		$product_ids = array_values( array_column( $value, 'value' ) );
		// Find the intersection between group IDs and segment IDs.
		$intersect = array_intersect($current_product_ids, $product_ids);

		// If there is at least one common ID, return true.
		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function not_included_in( array $value, array $automation_data ): bool {
		$order_id = isset( $automation_data['data']['order_id'] ) ? $automation_data['data']['order_id'] : '';

		if ( empty( $order_id ) ) {
			return false;
		}

		$order = wc_get_order($order_id);
		$items = $order->get_items();

		// Get the product IDs from the items.
		$current_product_ids = [];

		foreach ($items as $item) {
			$current_product_ids[] = $item->get_product_id();
		}

        if (empty($current_product_ids)) {
            return false;
        }

		$product_ids = array_values( array_column( $value, 'value' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $current_product_ids, $product_ids );

		if ( !empty( $diff ) ) {
			return true;
		}

		return false;
	}
}
