<?php
/**
 * CurrentOrderTotalValue class for checking the current order total value of a user.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * CurrentOrderTotalValue class for checking the current order total value of a user.
 *
 * @since 1.17.2
 */
class CurrentOrderTotalValue {

    /**
     * Fetch the total order value for a given order ID.
     *
     * @param int $order_id The order ID of the user.
     * 
     * @return int|null The total order value or null if not found.
     * @since 1.17.2
     */
    private function get_total_order_value($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return false;
        }

        $total_order = $order->get_total();
        return $total_order !== null ? (float)$total_order : null;
    }

    /**
     * Check if the total order count is greater than the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the order ID.
     * 
     * @return bool True if the total order count is greater than the value, false otherwise.
     * @since 1.17.2
     */
    public function greater_than($value, $automation_data): bool {
        $order_id = $automation_data['data']['order_id'] ?? '';

        if (empty($order_id)) {
            return false;
        }

        $total_order = $this->get_total_order_value($order_id);

        if ($total_order === null) {
            return false;
        }

        return $total_order > (float)$value;
    }

    /**
     * Check if the total order count is less than the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the order ID.
     * 
     * @return bool True if the total order count is less than the value, false otherwise.
     * @since 1.17.2
     */
    public function less_than($value, $automation_data): bool {
        $order_id = $automation_data['data']['order_id'] ?? '';

        if (empty($order_id)) {
            return false;
        }

        $total_order = $this->get_total_order_value($order_id);

        if ($total_order === null) {
            return false;
        }

        return $total_order < (float)$value;
    }

    /**
     * Check if the total order count is equal to the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the order ID.
     * 
     * @return bool True if the total order count is equal to the value, false otherwise.
     * @since 1.17.2
     */
    public function equal($value, $automation_data): bool {
        $order_id = $automation_data['data']['order_id'] ?? '';

        if (empty($order_id)) {
            return false;
        }

        $total_order = $this->get_total_order_value($order_id);

        if ($total_order === null) {
            return false;
        }

        return $total_order === (float)$value;
    }

    /**
     * Check if the total order count is not equal to the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the order ID.
     * 
     * @return bool True if the total order count is not equal to the value, false otherwise.
     * @since 1.17.2
     */
    public function does_not_equal($value, $automation_data): bool {
        $order_id = $automation_data['data']['order_id'] ?? '';

        if (empty($order_id)) {
            return false;
        }

        $total_order = $this->get_total_order_value($order_id);

        if ($total_order === null) {
            return false;
        }

        return $total_order !== (float)$value;
    }
}