<?php
/**
 * ListSegment class for checking contact first name
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use MintMailPro\Mint_Pro_Helper;

/**
 * ListSegment class for checking contact first name
 */
class ListSegment {

	/**
	 * Checks if any segment ID from the given segments array matches any group ID in the automation data.
	 *
	 * @param array $segments The array of segments to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a segment ID matches a group ID, false otherwise.
	 * @since 1.2.7
	 */
	public function includes( array $segments, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		// Extract group IDs and segment IDs.
		$group_ids   = Mint_Pro_Helper::get_group_ids_by_email( $user_email, 'lists' );
		$segment_ids = array_values( array_column( $segments, 'id' ) );

		// Find the intersection between group IDs and segment IDs.
		$intersect = array_intersect( $segment_ids, $group_ids );

		// If there is at least one common ID, return true.
		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_includes( array $segments, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		// Extract group IDs and segment IDs.
		$group_ids   = Mint_Pro_Helper::get_group_ids_by_email( $user_email, 'lists' );
		$segment_ids = array_values( array_column( $segments, 'id' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $segment_ids, $group_ids );

		if ( !empty( $diff ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if all segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the first name matches the value, false otherwise.
	 * @since 1.2.7
	 */
	public function includes_all_of( array $segments, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		// Extract group IDs and segment IDs.
		$group_ids   = Mint_Pro_Helper::get_group_ids_by_email( $user_email, 'lists' );
		$segment_ids = array_values( array_column( $segments, 'id' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $segment_ids, $group_ids );

		if ( empty( $diff ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool              True if the first name is not equal to the value (case-insensitive), false otherwise.
	 * @since 1.2.7
	 */
	public function includes_none_of( array $segments, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		// Extract group IDs and segment IDs.
		$group_ids   = Mint_Pro_Helper::get_group_ids_by_email( $user_email, 'lists' );
		$segment_ids = array_values( array_column( $segments, 'id' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $segment_ids, $group_ids );

		if ( count( $diff ) === count( $segment_ids ) ) {
			return true;
		}
		return false;
	}
}
