<?php
/**
 * QuizScore class for checking the total order count of a user.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * QuizScore class for checking the total order count of a user.
 *
 * @since 1.17.2
 */
class QuizScore {

    /**
     * Fetch the quiz score of the user.
     *
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return int|null The quiz score of the user, or null if the score is not found.
     * @since 1.17.2
     */
    private function get_quiz_score($automation_data) {
        $score = $automation_data['data']['quiz_data']['score'] ?? '';
        return $score !== null ? (float)$score : null;
    }

    /**
     * Check it the total score is greater than the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return bool True if the total score is greater than the value, false otherwise.
     * @since 1.17.2
     */
    public function greater_than($value, $automation_data): bool {
        $user_email = $automation_data['data']['user_email'] ?? '';

        if (empty($user_email)) {
            return false;
        }

        $total_score = $this->get_quiz_score($automation_data);

        if ($total_score === null) {
            return false;
        }

        return $total_score > (float)$value;
    }

    /**
     * Check if the total score is less than the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return bool True if the total score is less than the value, false otherwise.
     * @since 1.17.2
     */
    public function less_than($value, $automation_data): bool {
        $user_email = $automation_data['data']['user_email'] ?? '';

        if (empty($user_email)) {
            return false;
        }

        $total_score = $this->get_quiz_score($automation_data);

        if ($total_score === null) {
            return false;
        }

        return $total_score < (float)$value;
    }

    /**
     * Check if the total score is greater than or equal to the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return bool True if the total score is greater than or equal to the value, false otherwise.
     * @since 1.17.2
     */
    public function equal($value, $automation_data): bool {
        $user_email = $automation_data['data']['user_email'] ?? '';

        if (empty($user_email)) {
            return false;
        }

        $total_score = $this->get_quiz_score($automation_data);

        if ($total_score === null) {
            return false;
        }

        return $total_score === (float)$value;
    }

    /**
     * Check if the total score is not equal to the given value.
     *
     * @param int $value The value to compare against.
     * @param array $automation_data The automation data containing the user's email.
     * 
     * @return bool True if the total score is not equal to the value, false otherwise.
     * @since 1.17.2
     */
    public function does_not_equal($value, $automation_data): bool {
        $user_email = $automation_data['data']['user_email'] ?? '';

        if (empty($user_email)) {
            return false;
        }

        $total_score = $this->get_quiz_score($automation_data);

        if ($total_score === null) {
            return false;
        }

        return $total_score !== (float)$value;
    }
}